//////////////////////////////////////////////////////////////////////////////
//
//  File:    soundl.cpp
//
//  Class:   SoundLinux
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The SoundLinux class is a class encapsulating sound on the Linux
//      platform.  This does not use any external libraries like SEAL,
//      it operates directly on the OSS.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>


//  Application Headers.
#include "soundl.h"
#include "config.h"
#include "sample.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Data Member Initialization.
///////////////////////////////////////////////////////////////////////////////

//  The number of channels supported.
const int32 SoundLinux::sm_nNumChannels = 16;

//  The bounds on the number of fragments.
const DWord SoundLinux::sm_dwNumFragsMin = 2;
const DWord SoundLinux::sm_dwNumFragsMax = 16;
const DWord SoundLinux::sm_dwNumFragsDef = 5;

//  The bounds on the size of a fragment.
const DWord SoundLinux::sm_dwFragSizeMin = 4;
const DWord SoundLinux::sm_dwFragSizeMax = 16;
const DWord SoundLinux::sm_dwFragSizeDef = 10;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: SoundLinux
//
//  Description:
//
//      This is the main constructor for a Linux sound object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SoundLinux::SoundLinux(
    const KString&  iName
)
:
    Sound            ( iName ),
    m_nDevice        ( 0 ),
    m_dwNumFrags     ( sm_dwNumFragsDef ),
    m_dwFragSize     ( sm_dwFragSizeDef ),
    m_bSoundPending  ( FALSE ),
    m_pbAudioBuffer  ( NULL ),
    m_pChannelList   ( NULL ),
    m_nWaitingIndex  ( -1 )
{
    //  The configuration object.
    Configuration& config = Configuration::s_instance( );

    //  An integer command line parameter.
    int32 nParam;


    //  If sound is disabled then don't bother initializing.
    if( !m_bEnabled )
    {
        return;
    }


    //  Retrieve the number of fragments from the command line.
    if( config.getParam( "-numfrags", &nParam ) )
    {
        CONFIRM( 
            ( ( DWord )nParam >= sm_dwNumFragsMin ) && 
            ( ( DWord )nParam <= sm_dwNumFragsMax ),
            "Number of fragments must be between %d and %d.",
            sm_dwNumFragsMin,
            sm_dwNumFragsMax
        );

        m_dwNumFrags = ( DWord )nParam;
    }

    //  Retrieve the size of the fragments from the command line.
    if( config.getParam( "-fragsize", &nParam ) )
    {
        CONFIRM( 
            ( ( DWord )nParam >= sm_dwFragSizeMin ) && 
            ( ( DWord )nParam <= sm_dwFragSizeMax ),
            "Fragments size must be between %d and %d.",
            sm_dwFragSizeMin,
            sm_dwFragSizeMax
        );

        m_dwFragSize = ( DWord )nParam;
    }


    //  Allocate space for the audio buffer.
    m_pbAudioBuffer = new Byte[ m_dwNumFrags * ( 1 << m_dwFragSize ) ];

    //  Allocate the list of channels.
    m_pChannelList = new Channel[ sm_nNumChannels ];
    for( int32 nI = 0 ; nI < sm_nNumChannels ; nI += 1 )
    {
        m_pChannelList[ nI ].m_dwNumWaiting = 0;
    }

    //  Initialize the pool of waiting samples.
    for( int32 nI = 0 ; nI < 64 ; nI += 1 )
    {
        m_aWaitingPool[ nI ].m_pData = NULL;
        addWaiting( &( m_aWaitingPool[ nI ] ) );
    }


    //  Open the audio device.
    m_nDevice = open( "/dev/dsp", O_WRONLY, 0 );
    if( m_nDevice < 0 )
    {
        CHECK0( FALSE, "Could not open /dev/dsp." );
        m_bEnabled = FALSE;
        return;
    }

    //  Set the fragment size.  The format is MMMMSSSS where MMMM is the number
    //  of fragments and 2^SSSS is the size of the fragment.
    int nFragment = ( m_dwNumFrags << 16 ) | m_dwFragSize;
    if( ioctl( m_nDevice, SNDCTL_DSP_SETFRAGMENT, &nFragment ) < 0 )
    {
        perror( "Setting Sound Fragment" );
        fatalError( "Could not set sound fragment to 0x%08x.", nFragment );
    }

    //  Use 8-bit format.
    int nFormat = AFMT_U8;
    if( ioctl( m_nDevice, SNDCTL_DSP_SETFMT, &nFormat ) < 0 )
    {
        perror( "Setting Sound Format" );
        fatalError( "Could not set sound format to 8-bit." );
    }

    //  Set to mono.
    int nStereo = 0;
    if( ioctl( m_nDevice, SNDCTL_DSP_STEREO, &nStereo ) < 0 )
    {
        perror( "Setting Mono Sound" );
        fatalError( "Could not set sound to mono" );
    }

    //  Set the sample rate.
    int nSampleRate = m_dwSampleRate;
    if( ioctl( m_nDevice, SNDCTL_DSP_SPEED, &nSampleRate ) < 0 )
    {
        perror( "Setting Sound Sample Rate" );
        fatalError( "Could not set sound to sample rate of: %d", nSampleRate );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SoundLinux
//
//  Description:
//
//      This is the destructor for a Linux sound object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SoundLinux::~SoundLinux(
)
{
    //  If sound is disabled then nothing to clean up.
    if( !m_bEnabled )
    {
        return;
    }

    //  Tear down the voices.
    for( int32 nI = 0 ; nI < sm_nNumChannels ; nI += 1 )
    {
        stop( nI );
    }

    //  Close the audio device.
    close( m_nDevice );

    //  Free the audio buffer.
    delete [] m_pbAudioBuffer;

    //  Free the list of channels.
    delete [] m_pChannelList;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the sound object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
SoundLinux::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "SoundLinux" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: getNumChannels
//
//  Description:
//
//      This member is called to return the number of voice channels.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The number of voices.
//
///////////////////////////////////////////////////////////////////////////////
DWord
SoundLinux::getNumChannels(
) const
{
    return( sm_nNumChannels );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member is called to allow the sound system to update itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      FALSE.
//
///////////////////////////////////////////////////////////////////////////////
Byte
SoundLinux::update(
)
{
    //  The information about the OSS audio buffer.
    audio_buf_info info;


    //  If sound is disabled or there is no pending sound then do nothing.
    if( !m_bEnabled || !m_bSoundPending )
    {
        return( FALSE );
    }
    
    //  How much room is left in the audio buffer?  If less than 256
    //  bytes then let the audio device catch up.
    ioctl( m_nDevice, SNDCTL_DSP_GETOSPACE, &info );
    if( info.bytes <= 256 )
    {
        return( FALSE );
    }

    //  Indicate that there is no pending sound until we find some.
    m_bSoundPending = FALSE;

    //  Initialize our audio buffer.
    memset( ( void* )m_pbAudioBuffer, 0x80, info.bytes );

    //  We have to mix each channel together into the audio buffer.
    for( int32 nI = 0 ; nI < sm_nNumChannels ; nI += 1 )
    {
        //  A convient pointer to the current channel.
        Channel* pChannel = &( m_pChannelList[ nI ] );

        //  If there are no samples waiting on this channel then 
        //  continue on.
        if( pChannel->m_dwNumWaiting == 0 )
        {
            continue;
        }
        else
        {
            m_bSoundPending = TRUE;
        }

        //  A convenience to the sample to play.
        Waiting* pWaiting = pChannel->m_apWaiting[ 0 ];

        //  Add the data for the waiting sample to the audio buffer.
        for( int32 nJ = 0 ; nJ < info.bytes ; nJ += 1 )
        {
            //  Update the current byte in the audio buffer.
            m_pbAudioBuffer[ nJ ] += ( Byte )(
                *( pWaiting->m_pCurrent ) * pWaiting->m_fRelVolume
            );

            //  Update our position within the sample.
            for( 
                pWaiting->m_fAcc += pWaiting->m_fFactor ;
                pWaiting->m_fAcc > 1 ;
                pWaiting->m_fAcc -= 1
            ) 
            {
                pWaiting->m_pCurrent += 1;
            }
    
            //  If we're finished with the sample, move on to the next one.
            if( pWaiting->m_pCurrent > pWaiting->m_pBufferEnd )
            {
                //  If we aren't looping then quit out now.
                if( !pWaiting->m_bLoop )
                { 
                    stop( nI );
                    break;
                }

                //  If there is a sample waiting then move on to it, otherwise
                //  we'll continue playing this one.
                if( pChannel->m_dwNumWaiting > 1 )
                {
                    delete [] pWaiting->m_pData;
                    pWaiting->m_pData = NULL;
                    addWaiting( pWaiting );
                    for( 
                        DWord dwK = 0 ; 
                        dwK < pChannel->m_dwNumWaiting - 1 ; 
                        dwK += 1 
                    )
                    {
                        pChannel->m_apWaiting[ dwK ] = 
                            pChannel->m_apWaiting[ dwK + 1 ];
                    }
                    pChannel->m_dwNumWaiting -= 1;
                }

                pWaiting = pChannel->m_apWaiting[ 0 ];
                pWaiting->m_pCurrent = ( int8* )pWaiting->m_pData;
            }
        }
    }
 
    //  Write the audio buffer out to the audio device.
    if( m_bSoundPending )
    {
        write( m_nDevice, m_pbAudioBuffer, info.bytes );
    }

    return( FALSE );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: play
//
//  Description:
//
//      This member is called to play the specified sample.  It simply adds
//      the sample to a list that will be played during update().
//
//  Parameters:
//
//      pSample (input)
//          The sample to play.
//
//      bLoop (input)
//          The sample is to loop.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
SoundLinux::play(
    Sample*    pSample,
    const Byte bLoop
)
{
    //  If sound is disabled then do nothing.
    if( !m_bEnabled )
    {
        return;
    }

    //  There is no sound to play.
    m_bSoundPending = TRUE;

    //  A convenience to the channel.
    ASSERT( pSample->getChannel( ) < sm_nNumChannels );
    Channel* pChannel = &( m_pChannelList[ pSample->getChannel( ) ] );


    //  Stop anything currently playing on the sample.
    stop( pSample->getChannel( ) );


    //  If we've run out of room then we ignore the request.
    Waiting* pWaiting = getWaiting( );
    if( ( pWaiting == NULL ) || ( pChannel->m_dwNumWaiting >= 64 ) )
    {
        CHECK0( pWaiting == NULL, "Waiting Sample list exceeded." );
        return;
    }

    //  Fill in the "Waiting Sample" object.
    pWaiting->m_pData      = new int8[ pSample->getSize( ) ];
    pWaiting->m_pCurrent   = pWaiting->m_pData;
    pWaiting->m_pBufferEnd = pWaiting->m_pData + pSample->getSize( ) - 1;
    pWaiting->m_bLoop      = bLoop;
    pWaiting->m_fFactor    = ( float )pSample->getFrequency( ) / m_dwSampleRate;
    pWaiting->m_fAcc       = 0.0;
    pWaiting->m_fRelVolume = ( float )pSample->getVolume( ) / 512;
    memcpy( 
        ( void* )( pWaiting->m_pData ), 
        ( void* )( pSample->getBuffer( ) ), 
        pSample->getSize( ) 
    );

    //  Assign the waiting sample to the channel.
    pChannel->m_apWaiting[ pChannel->m_dwNumWaiting ] = pWaiting;
    pChannel->m_dwNumWaiting += 1;
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: playStreamed
//
//  Description:
//
//      This member is called to stream the specified sample.
//
//  Parameters:
//
//      pSample (input)
//          The sample to play.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
SoundLinux::playStreamed(
    Sample* pSample
)
{
    //  If sound is disabled then do nothing.
    if( !m_bEnabled )
    {
        return;
    }

    //  There is no sound to play.
    m_bSoundPending = TRUE;

    //  A convenience to the channel.
    ASSERT( pSample->getChannel( ) < sm_nNumChannels );
    Channel* pChannel = &( m_pChannelList[ pSample->getChannel( ) ] );

    //  If we've run out of room then we ignore the request.
    Waiting* pWaiting = getWaiting( );
    if( ( pWaiting == NULL ) || ( pChannel->m_dwNumWaiting >= 64 ) )
    {
        CHECK0( pWaiting == NULL, "Waiting streamed sample list exceeded." );
        return;
    }

    //  Fill in the "Waiting Sample" object.
    pWaiting->m_pData      = new int8[ pSample->getSize( ) ];
    pWaiting->m_pCurrent   = pWaiting->m_pData;
    pWaiting->m_pBufferEnd = pWaiting->m_pData + pSample->getSize( ) - 1;
    pWaiting->m_bLoop      = TRUE;
    pWaiting->m_fFactor    = ( float )pSample->getFrequency( ) / m_dwSampleRate;
    pWaiting->m_fAcc       = 0.0;
    pWaiting->m_fRelVolume = ( float )pSample->getVolume( ) / 512;
    memcpy( 
        ( void* )( pWaiting->m_pData ), 
        ( void* )( pSample->getBuffer( ) ), 
        pSample->getSize( ) 
    );

    //  Assign the waiting sample to the channel.
    pChannel->m_apWaiting[ pChannel->m_dwNumWaiting ] = pWaiting;
    pChannel->m_dwNumWaiting += 1;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: stop
//
//  Description:
//
//      This member is called to stop sample playback on the specified
//      channel.
//
//  Parameters:
//
//      dwChannel (input)
//          The channel to stop playback on.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
SoundLinux::stop(
    const DWord dwChannel
)
{
    //  If sound is disabled then do nothing.
    if( !m_bEnabled )
    {
        return;
    }

    //  A convenience to the channel.
    ASSERT( dwChannel < ( DWord )sm_nNumChannels );
    Channel* pChannel = &( m_pChannelList[ dwChannel ] );

    //  Free up each "Waiting Sample".
    for( DWord dwI = 0 ; dwI < pChannel->m_dwNumWaiting ; dwI += 1 )
    {
        delete [] pChannel->m_apWaiting[ dwI ]->m_pData;
        pChannel->m_apWaiting[ dwI ]->m_pData = NULL;
        addWaiting( pChannel->m_apWaiting[ dwI ] );
    }
    pChannel->m_dwNumWaiting = 0;
} 
