///////////////////////////////////////////////////////////////////////////////
//
//  File:       soundl.h
//
//  Class:      SoundLinux - Concrete
//  
//  Hierarchy:  SoundLinux - Sound - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      The SoundLinux class is a class encapsulating sound on the Linux
//      platform.  This does not use any external libraries like SEAL,
//      it operates directly on the OSS.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _soundl_h_
#define _soundl_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>
#include <sys/soundcard.h>

//  Application Headers.
#include "sound.h"
#include "kstring.h"


///////////////////////////////////////////////////////////////////////////////
//  SoundLinux Class Definition
///////////////////////////////////////////////////////////////////////////////
class SoundLinux : public Sound
{
    public:

        //  Creation/Deletion.
                 SoundLinux ( const KString& iName );
        virtual ~SoundLinux ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  General Interface.
        virtual Byte update       ( );
        virtual void play         ( Sample* pSample, const Byte bLoop );
        virtual void playStreamed ( Sample* pSample );
        virtual void stop         ( const DWord dwChannel );

    protected:

        //  The following embedded class holds information about a sample
        //  that is waiting to be played.
        class Waiting
        {
            public:
                int8*   m_pData;
                int8*   m_pCurrent;
                int8*   m_pBufferEnd;
                Byte    m_bLoop;
                float   m_fFactor;
                float   m_fAcc;
                float   m_fRelVolume;
        };
 
        //  The following embedded class indicates the current status of
        //  a channel.
        class Channel
        {
            public:
                DWord    m_dwNumWaiting;
                Waiting* m_apWaiting[ 64 ];
        };

        //  Utility.
        virtual DWord getNumChannels ( ) const;
        void          addWaiting     ( Waiting* pWaiting );
        Waiting*      getWaiting     ( );

        //  Member Data.
        int      m_nDevice;
        DWord    m_dwNumFrags;
        DWord    m_dwFragSize;
        Byte     m_bSoundPending;
        Byte*    m_pbAudioBuffer;
        Channel* m_pChannelList;
        Waiting  m_aWaitingPool[ 64 ];
        Waiting* m_apWaitingStack[ 64 ];
        int32    m_nWaitingIndex;
        

    private:

        //  Static Member Data.
        static const int32 sm_nNumChannels;
        static const DWord sm_dwNumFragsMin;
        static const DWord sm_dwNumFragsMax;
        static const DWord sm_dwNumFragsDef;
        static const DWord sm_dwFragSizeMin;
        static const DWord sm_dwFragSizeMax;
        static const DWord sm_dwFragSizeDef;
};

///////////////////////////////////////////////////////////////////////////////
//  SoundLinux Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
void
SoundLinux::addWaiting(
    SoundLinux::Waiting* pWaiting
)
{
    ASSERT( m_nWaitingIndex < 63 );
    ASSERT( pWaiting != NULL );
    m_nWaitingIndex += 1;
    m_apWaitingStack[ m_nWaitingIndex ] = pWaiting;
}

inline
SoundLinux::Waiting*
SoundLinux::getWaiting(
)
{
    if( m_nWaitingIndex < 0 )
    {
        return( NULL );
    }
    m_nWaitingIndex -= 1;
    return( m_apWaitingStack[ m_nWaitingIndex + 1 ] );
}

#endif
