///////////////////////////////////////////////////////////////////////////////
//
//  File:    client.cpp
//
//  Class:   NetGameClientMeddler
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is used to handle the duties of a client during a net
//      game.  The client informs the server of its state and processes
//      coordination messages from the server.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "client.h"
#include "game.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a net game client meddler object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object.
//
//      pGame (input)
//          The game the meddler is associated with.
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
NetGameClientMeddler*
NetGameClientMeddler::s_build(
    const KString& iName,
    Game*          pGame
)
{
    //  Create the new object.
    NetGameClientMeddler* pThis = new NetGameClientMeddler( iName, pGame );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new object.
    return( pThis );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: NetGameClientMeddler
//
//  Description:
//
//      This is the main constructor for a net game client meddler object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          A pointer to the game that is currently being run by Replay.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
NetGameClientMeddler::NetGameClientMeddler(
    const KString&  iName,
    Game*           pGame
)
:
    NetGameMeddler ( iName, pGame )
{
    //  Initialization is done in init( ).
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is initialization function for a net game client meddler object.  
//      It is to be called from the build method and performs functions we 
//      would normally do in the constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetGameClientMeddler::init(
)
{
    //  First let the base class initialize.
    NetGameMeddler::init( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~NetGameClientMeddler
//
//  Description:
//
//      This is the destructor for a net game client meddler object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
NetGameClientMeddler::~NetGameClientMeddler(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the net game client meddler object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
NetGameClientMeddler::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "NetGameClientMeddler" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member is called periodically when the meddler is to update 
//      itself.  It sends the current state of it's controls to the server
//      and then waits for a response indicating the full state of the 
//      controls.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetGameClientMeddler::update(
)
{
    //  Update should not be called if the meddler has completed.
    ASSERT( !m_bComplete );

    //  A control message.
    Network::MsgCtrl* pMsgCtrl = ( Network::MsgCtrl* )m_pMsg;

    //  Send the current state of our controls to the server.
    pMsgCtrl->eType = Network::MSG_CTRL;
    getKeyState( &( pMsgCtrl->dwCtrlMask ) );
    m_pNetwork->writeMsg( 0, m_pMsg );

    //  Now keep looping until we receive the combined control message from
    //  the server.
    for( ; ; )
    {
        //  Read the next message from the server.
        m_pNetwork->readMsg( 0, m_pMsg );

        //  Does the message contain the combined control information?
        if( m_pMsg->eType == Network::MSG_CTRL )
        {
            //  Set our virtual control state to that passed from the server.
            setKeyState( &( pMsgCtrl->dwCtrlMask ) );

            //  We've received the control message so we may continue on
            //  with the emulation.
            break;
        }
        //  Is the message an indication of which player we are?
        else
        if( m_pMsg->eType == Network::MSG_PLAYER )
        {
            //  Cast to the appropriate message.
            Network::MsgPlayer* pMsgPlayer = ( Network::MsgPlayer* )m_pMsg;

            //  Set the player number.
            setPlayerNum( ( DWord )pMsgPlayer->bPlayerNum );
        }
        //  Has the server closed it's connection?
        else
        if( m_pMsg->eType == Network::MSG_CLOSE )
        {
            //  Terminate the meddler and proceed no further.
            terminate( );
            break; 
        }
    }
}
