///////////////////////////////////////////////////////////////////////////////
//
//  File:    netgame.cpp
//
//  Class:   NetGameMeddler
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a base class for Meddlers that are used in
//      the playing of networked Replay games.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "netgame.h"
#include "replay.h"
#include "game.h"
#include "network.h"
#include "ctrlmap.h"
#include "keyb.h"
#include "hiscore.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: NetGameMeddler
//
//  Description:
//
//      This is the main constructor for a net game meddler object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          A pointer to the game that is currently being run by Replay.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
NetGameMeddler::NetGameMeddler(
    const KString&  iName,
    Game*           pGame
)
:
    Meddler           ( iName, pGame ),
    m_pNetwork        ( Replay::s_instance( ).getNetwork( ) ),
    m_pKeyboard       ( Replay::s_instance( ).getKeyboard( ) ),
    m_pJoystick       ( Replay::s_instance( ).getJoystick( ) ),
    m_dwPlayerNum     ( PLAYER_UNASSIGNED ),
    m_pMsg            ( NULL ),
    m_rCtrlMapList    ( pGame->getCtrlMapList( ) ),
    m_decoration      ( "Net Game: Player ??" ),
    m_bShowDecoration ( FALSE )
{
    ASSERT( m_pGame     != NULL );
    ASSERT( m_pNetwork  != NULL );
    ASSERT( m_pKeyboard != NULL );
    ASSERT( m_pJoystick != NULL );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is initialization function for a Net Game Meddler object.  
//      It is to be called from the build method and performs functions we 
//      would normally do in the constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetGameMeddler::init(
)
{
    //  Call the base class first.
    Meddler::init( );

    ASSERT( m_pGame != NULL );

    //  Create the generic network message.
    m_pMsg = new Network::Msg;

    //  Disable the high scores since the scores will be different on
    //  each networked player and we want *all* inputs to be equivalent.
    if( m_pGame->getHiScore( ) )
    {
        m_pGame->getHiScore( )->disable( );
    }

    //  Reset the game.
    m_pGame->reset( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~NetGameMeddler
//
//  Description:
//
//      This is the destructor for a net game meddler object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
NetGameMeddler::~NetGameMeddler(
)
{
    //  Free the network message.
    delete m_pMsg;

    //  Reenable the high scores.
    if( m_pGame->getHiScore( ) )
    {
        m_pGame->getHiScore( )->enable( );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getDecoration
//
//  Description:
//
//      This member is to return the decoration that is to be used to indicate
//      to the user that the meddler is meddling. 
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The decoration.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
NetGameMeddler::getDecoration(
) const
{
    //  Return the decoration.
    if( m_bShowDecoration )
    {
        return( m_decoration );
    }
    else
    {
        return( KStringNULL );
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: getKeyState
//
//  Description:
//
//      This function is called to get the physical state of the keys used in 
//      the control map of the current game.  They are returned in bitmap form
//      using a 32 bit double word.  The first control will be the bit on
//      the far right, and will move towards the left.
//
//  Parameters:
//
//      pdwState (output)
//          A pointer to the bitmap that is to contain the state.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetGameMeddler::getKeyState(
    DWord *pdwState
)
{
    //  Clear the state.
    *pdwState = 0;

    //  Loop through each control map.
    for( DWord dwI = 0 ; dwI < m_rCtrlMapList.entries( ) ; dwI += 1 )
    {
        //  If the key is pressed then flick the bit.
        if( 
            m_pKeyboard->getSwitch(
                m_rCtrlMapList[ dwI ]->get( CtrlMap::KEYBOARD ) 
            )->onPhysically( ) ||
            m_pJoystick->getSwitch(
                m_rCtrlMapList[ dwI ]->get( CtrlMap::JOYSTICK ) 
            )->onPhysically( )
        )
        {
            *pdwState |= ( 1 << dwI );
        }
    }

    //  If the decoration is being shown and the state is non-zero (i.e. 
    //  a key is being pressed) then unshow the decoration.  This results
    //  in the decoration being shown when the player is assigned a number
    //  and the decoration being hidden when the player first makes use
    //  of a control after the assignment.  This prevents the decoration
    //  from cluttering the screen throughout the game.
    if( m_bShowDecoration && *pdwState )
    {
        m_bShowDecoration = FALSE;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setKeyState
//
//  Description:
//
//      This function is called to set the state of the keys used in the
//      control map of the current game.  They are specified in bitmap form
//      using a 32 bit double word.  The first control will be the bit on
//      the far right, and will move towards the left.
//
//  Parameters:
//
//      pdwState (input)
//          A pointer to the bitmap containing the state of the keys.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetGameMeddler::setKeyState(
    DWord *pdwState
)
{
    //  Loop through each control map.
    for( DWord dwI = 0 ; dwI < m_rCtrlMapList.entries( ) ; dwI += 1 )
    {
        //  Set the state appropriately.
        m_pKeyboard->getSwitch( 
            m_rCtrlMapList[ dwI ]->get( CtrlMap::KEYBOARD ) 
        )->virtualOn( 
            ( *pdwState & ( 1 << dwI ) ) ? TRUE : FALSE
        );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setPlayerNum
//
//  Description:
//
//      This function is called to set the player number that this
//      replay instance represents in the network game.  It assigns the
//      player number and sets the decoration text to match.
//
//  Parameters:
//
//      dwPlayerNum (input)
//          Indicates which player this instance represents.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetGameMeddler::setPlayerNum(
    DWord dwPlayerNum
)
{
    //  A temporary string to hold the text for the decoration.
    char strTemp[ 32 ];


    //  If the player number hasn't changed then just return.
    if( dwPlayerNum == m_dwPlayerNum )
    {
        return;
    }

    //  Assign the player number.
    m_dwPlayerNum = dwPlayerNum;

    //  If the player number is a valid player in the current game then
    //  indicate this in the decoration, otherwise this instance is just
    //  an observer.
    if( m_dwPlayerNum <= m_pGame->getMaxPlayers( ) )
    {
        sprintf( strTemp, "Net Game: Player %ld.", m_dwPlayerNum );
    }
    else
    {
        sprintf( strTemp, "Net Game: Observing %ld.", m_dwPlayerNum );
    }

    //  Assign the decoration and indicate that it should be shown.
    m_decoration      = strTemp;
    m_bShowDecoration = TRUE;
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: terminate
//
//  Description:
//
//      This member is called when the network game is to be stopped.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetGameMeddler::terminate(
)
{
    //  If we've already completed then just return.
    if( m_bComplete )
    {
        return;
    }

    //  Shutdown the network.
    ASSERT( m_pNetwork != NULL );
    m_pNetwork->closeNetwork( );

    //  Note that the connection has closed.
    m_decoration = "Net Connection Closed!";
    m_bShowDecoration = TRUE;

    //  Call the base class.
    Meddler::terminate( );
}
