///////////////////////////////////////////////////////////////////////////////
//
//  File:       bitmap.h
//
//  Class:      Bitmap - Concrete
//  
//  Hierarchy:  Bitmap - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class represents a bitmap which is of course just an array
//      of bits representing an image.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _bitmap_h_
#define _bitmap_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "kstring.h"
#include "matrix.h"
#include "canvas.h"

//  Forward class declarations.
class Canvas;


///////////////////////////////////////////////////////////////////////////////
//  Bitmap Class Definition
///////////////////////////////////////////////////////////////////////////////
class Bitmap : public RepBase
{
    public:

        //  The possible transparency values.
        typedef enum
        {
            TRANSPARENCY_NONE,
            TRANSPARENCY_RAW,
            TRANSPARENCY_MAP,
            TRANSPARENCY_REVERSE
        }
        Transparency;

        //  A clipped mapping.
        enum
        {
            MAP_CLIPPED = -1
        };


        //  Deletion.
        virtual ~Bitmap ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Access Interface.
        DWord     getWidth        ( ) const;
        DWord     getHeight       ( ) const;
        Byte*     getBuffer       ( ) const;
        Byte**    getRows         ( ) const;
        Byte*     getRow          ( const DWord dwRow ) const;
        Clipping* getFullClipping ( ) const;
        Byte      isScreen        ( ) const; 

        //  Utility Interface.
        void   loadPCX         ( const KString& pcxFileName );

        //  Notifications from the canvas.
        void changeOrientation(
            const Byte bTranspose, const Byte bFlipX, const Byte bFlipY
        );

        //  Mapping Interface.
        void setMapSize   ( const DWord dwMapSize );
        void setMapping   ( 
            const DWord dwItem, 
            const int32 nX,
            const int32 nY,
            Clipping*   pClipping = NULL
        );
        int32 getMappingX ( const DWord dwIndex ) const;
        int32 getMappingY ( const DWord dwIndex ) const;

        //  Blit Interface.
        void   clear             ( const DWord dwColour = 0x00 );
        void   clearArea         ( 
            Clipping* pClipping, const DWord dwColour = 0x00 
        );
        void   copy              ( Bitmap* pSrc );
        void   setPixel          ( 
            int32 nDestX, 
            int32 nDestY, 
            DWord dwColour, 
            Clipping* pClipping = NULL
        );
        DWord  getPixel          ( int32 nDestX, int32 nDestY );
        virtual void blit        (
            Bitmap*      pSrc,
            DWord        dwItem,
            DWord*       pdwColourTable
        );
        virtual void blit ( 
            Bitmap*      pSrc,
            int32        nOX,
            int32        nOY,
            DWord*       pdwColourTable,
            Byte         bFlipX,
            Byte         bFlipY,
            Clipping*    pClipping,
            Transparency eTransparency,
            Byte         bTransparentColour = 0x00
        );
        virtual void blitScrollFull    ( 
            Bitmap*      pSrc,
            int32        nScrollX,
            int32        nScrollY,
            Clipping*    pClipping,
            Transparency eTransparency,
            Byte         bTransparentColour = 0x00
        );
    

    protected:

        //  The following describes a PCX header.
        typedef struct
        {
            Byte bManufacturer;
            Byte bVersion;
            Byte bEncoding;
            Byte bBitsPerPixel;
            Word wMinX;
            Word wMinY;
            Word wMaxX;
            Word wMaxY;
            Word wHorizRes;
            Word wVertRes;
            Byte abPalette[48];
            Byte bReserved;
            Byte bNumPlanes;
            Word wBytesPerLine;
            Word wPaletteType;
            Byte abFiller[58];
        }
        PCXHeader;
 
        //  Creation is protected.  Clients should use a build method.
        Bitmap ( 
            const KString& iName, 
            const DWord    dwWidth, 
            const DWord    dwHeight,
            const Byte     bScreen = FALSE
        );

        //  This function should be called from the build method.
        virtual void init ( );

        //  Utility.
        virtual void swapXY                ( );

        //  Member data.
        Canvas*   m_pCanvas;
        DWord     m_dwWidth;
        DWord     m_dwHeight;
        Byte*     m_pbBuffer;
        Byte**    m_ppbRows;
        Byte**    m_ppbCols;
        Clipping* m_pFullClipping;
        DWord     m_dwMapSize;
        int32*    m_pnMapX;
        int32*    m_pnMapY;
        Byte      m_bScreen;

        //  The following static functions are used to obtain the current
        //  width and height of the bitmap (as opposed to the width and
        //  height adjusted for the current orientation).
        static DWord s_getWidth  ( Bitmap* pBitmap );
        static DWord s_getHeight ( Bitmap* pBitmap );

        //  Static Member Data.
        static Matrix<Byte> sm_matrix;

    private:

};


///////////////////////////////////////////////////////////////////////////////
//  Bitmap Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
DWord
Bitmap::getWidth(
) const
{
    return( m_pCanvas->getTranspose( ) ? m_dwHeight : m_dwWidth );
}

inline
DWord
Bitmap::s_getWidth(
    Bitmap* pBitmap
)
{
    return( pBitmap->m_dwWidth );
}

inline
DWord
Bitmap::getHeight(
) const
{
    return( m_pCanvas->getTranspose( ) ? m_dwWidth : m_dwHeight );
}

inline
DWord
Bitmap::s_getHeight(
    Bitmap* pBitmap
)
{
    return( pBitmap->m_dwHeight );
}

inline
Byte*
Bitmap::getBuffer(
) const
{
    return( m_pbBuffer );
}

inline
Byte**
Bitmap::getRows(
) const
{
    return( m_ppbRows );
}

inline
Byte*
Bitmap::getRow(
    const DWord dwRow
) const
{
    ASSERT( dwRow < m_dwHeight );
    return( m_ppbRows[ dwRow ] );
}

inline
Clipping*
Bitmap::getFullClipping(
) const
{
    return( m_pFullClipping );
}

inline
Byte
Bitmap::isScreen(
) const
{
    return( m_bScreen );
}

inline
int32
Bitmap::getMappingX(
    const DWord dwIndex
) const
{
    ASSERT( dwIndex < m_dwMapSize );
    
    return( m_pnMapX[ dwIndex ] );
}

inline
int32
Bitmap::getMappingY(
    const DWord dwIndex
) const
{
    ASSERT( dwIndex < m_dwMapSize );
    
    return( m_pnMapY[ dwIndex ] );
}

inline
void
Bitmap::copy(
    Bitmap* pSrc
)
{
    ASSERT( pSrc->m_dwWidth  == m_dwWidth );
    ASSERT( pSrc->m_dwHeight == m_dwHeight );
    memcpy( 
        ( void* )m_pbBuffer, ( void* )pSrc->m_pbBuffer, m_dwWidth * m_dwHeight
    );
}


#endif
