///////////////////////////////////////////////////////////////////////////////
//
//  File:    canvas.cpp
//
//  Class:   Canvas
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The Canvas class represents the screen for the Replay application.  
//      All screen access is done through the Canvas object.  This is an
//      abstract base class for platform specific derived classes that
//      implement the interface for the hardware of the platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>
#include <math.h>

//  Application Headers.
#include "canvas.h"
#include "page.h"
#include "config.h"
#include "gfxset.h"
#include "buffer.h"
#include "colour.h"
#include "ctable.h"
#include "bitmap.h"
#include "clip.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////

//  This table is used to determine what needs to be done to the bitmaps
//  to change from a certain orientation to another orientation.  This is 
//  used in the setOrientation( ) member function.
Byte Canvas::sm_abOrientTable[ 8 ][ 8 ] =
{
//  To:   txy    txY    tXy    tXY    Txy    TxY    TXy    TXY       From:
    {      0,     Y,     X,    X|Y,    T,    T|Y,   T|X,  T|X|Y    }, //  txy
    {      Y,     0,    X|Y,    X,    T|X,  T|X|Y,   T,    T|Y     }, //  txY
    {      X,    X|Y,    0,     Y,    T|Y,    T,   T|X|Y,  T|X     }, //  tXy
    {     X|Y,    X,     Y,     0,   T|X|Y,  T|X,   T|Y,    T      }, //  tXY
    {      T,    T|Y,   T|X,  T|X|Y,   0,     Y,     X,    X|Y     }, //  Txy
    {     T|X,  T|X|Y,   T,    T|Y,    Y,     0,    X|Y,    X      }, //  TxY
    {     T|Y,    T,   T|X|Y,  T|X,    X,    X|Y,    0,     Y      }, //  TXy
    {    T|X|Y,  T|X,   T|Y,    T,    X|Y,    X,     Y,     0      }  //  TXY
};

//  This table is used to determine the settings for the orientation rotated
//  90 degrees clockwise from the specified orientation.  This is used in the
//  rotate( ) member function.
Byte Canvas::sm_abRotateTable[ 8 ] =
{
//  From:  txy     txY    tXy    tXY    Txy    TxY    TXy    TXY
           T|X,     T,   T|X|Y,  T|Y,    X,     0,    X|Y,    Y
};

//  The lookup tables are used when calculating the closest colour.
Byte   Canvas::sm_bLUTInitialized = FALSE;
uint32 Canvas::sm_anRedLUT[ 256 ];
uint32 Canvas::sm_anGreenLUT[ 256 ];
uint32 Canvas::sm_anBlueLUT[ 256 ];



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Canvas
//
//  Description:
//
//      This is the main constructor for a canvas object.  It is protected
//      so that only derived classes have access to it.  Clients should
//      use a build method to create a canvas object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Canvas::Canvas(
    const KString&  iName
)
:
    RepBase            ( iName ),
    m_bOrientation     ( NOTRANSPOSE_NOFLIPX_NOFLIPY ),
    m_bUpdateNeeded    ( FALSE ),
    m_pFontSmall       ( NULL ),
    m_pFontLarge       ( NULL ),
    m_pPage            ( NULL ),
    m_bitmapList       ( 1024 ), 
    m_pPaletteColours  ( NULL ),
    m_pbCaptured       ( NULL )
{
    //  Initialization is to be done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called from the build method to initialize the object.
//      It allows virtual methods to be called during construction.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::init(
)
{
    //  The configuration object.
    Configuration& config = Configuration::s_instance( );

    //  Determine the orientation and set the flipping/rotating appropriately.
    if( config.getParam( "+90" ) )
    {
        m_bOrientation |= T;
        m_bOrientation |= X;
        m_bOrientation &= ~Y;
    }
    else
    if( config.getParam( "+180" ) || config.getParam( "-180" ) )
    {
        m_bOrientation &= ~T;
        m_bOrientation |= X;
        m_bOrientation |= Y;
    }
    else
    if( config.getParam( "-90" ) )
    {
        m_bOrientation |= T;
        m_bOrientation &= ~X;
        m_bOrientation |= Y;
    }
    if( config.getParam( "-flipx" ) )
    {
        if( m_bOrientation & X )
        {
            m_bOrientation &= ~X;
        }
        else
        {
            m_bOrientation |= X;
        }
    }
    if( config.getParam( "-flipy" ) )
    {
        if( m_bOrientation & Y )
        {
            m_bOrientation &= ~Y;
        }
        else
        {
            m_bOrientation |= Y;
        }
    }

    //  Assign the pages.
    for( DWord dwI = 0 ; dwI < PAGE_COUNT ; dwI += 1 )
    {
        m_apPageList[ dwI ] = NULL;
    }


    //  If the lookup tables haven't been initialized then do it here.
    //  The lookup formula is:
    //     dist = 
    //         0.299 * sqrt( abs( r1, r2 ) +
    //         0.587 * sqrt( abs( g1, g2 ) +
    //         0.114 * sqrt( abs( b1, b2 ) +
    //  Which uses the least squares method scaled for the eye's sensitivity
    //  to colour.
    if( !sm_bLUTInitialized )
    {
        sm_bLUTInitialized = TRUE;
        
        for( DWord dwI = 0 ; dwI < 256 ; dwI += 1 )
        {
            sm_anRedLUT[ dwI ]   = ( uint32 )( 299 * sqrt( dwI ) );
            sm_anGreenLUT[ dwI ] = ( uint32 )( 587 * sqrt( dwI ) );
            sm_anBlueLUT[ dwI ]  = ( uint32 )( 114 * sqrt( dwI ) );
        }
    }


    //  The number of colours.
    DWord dwNumColours = getNumColours( );

    //  Allocate space for the colours in the palette.
    m_pPaletteColours = new Colour[ dwNumColours ];

    //  Allocate space for the captured flags.
    m_pbCaptured = new Byte[ dwNumColours ];
    for( DWord dwI = 0 ; dwI < dwNumColours ; dwI += 1 )
    {
        m_pbCaptured[ dwI ] = FALSE;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Canvas
//
//  Description:
//
//      This is the destructor for a canvas object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Canvas::~Canvas(
)
{
    //  Delete the fonts.
    delete m_pFontSmall;
    delete m_pFontLarge;

    //  Assign the pages.
    for( DWord dwI = 0 ; dwI < PAGE_COUNT ; dwI += 1 )
    {
        delete m_apPageList[ dwI ];
        m_apPageList[ dwI ] = NULL;
    }

    //  Delete the colour list.
    delete [] m_pPaletteColours;

    //  Delete the captured flag list. 
    delete [] m_pbCaptured;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the canvas to a file for retrieval at a later
//      date.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the canvas to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Canvas::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  The number of errors encountered while saving.
    DWord dwNumErrors = 0;


    //  We always allow the base class to save itself first.
    RepBase::save( pSaveFile );

    //  Save the member data that keeps the state.
    for( DWord dwI = 0 ; dwI < getNumColours( ) ; dwI += 1 )
    {
        dwNumErrors += m_pPaletteColours[ dwI ].save( pSaveFile );
    }

    //  Was it saved successfully?
    return( dwNumErrors ? TRUE : FALSE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the canvas from a file that it was previously
//      saved to.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the canvas from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Canvas::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  The number of errors encountered while saving.
    DWord dwNumErrors = 0;


    //  We always allow the base class to load itself first.
    RepBase::load( pLoadFile );

    //  Load the member data that keeps the state.
    for( DWord dwI = 0 ; dwI < getNumColours( ) ; dwI += 1 )
    {
        dwNumErrors += m_pPaletteColours[ dwI ].load( pLoadFile );

        //  Set the colours in the current palette.
        DWord dwRed;
        DWord dwGreen;
        DWord dwBlue;
        m_pPaletteColours[ dwI ].get( &dwRed, &dwGreen, &dwBlue );
        setColour( dwI, dwRed, dwGreen, dwBlue );
    }

    //  Was it loaded successfully?
    return( dwNumErrors ? TRUE : FALSE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setOrientation
//
//  Description:
//
//      This member is called to change the orientation of the canvas to
//      an absolute setting..
//
//  Parameters:
//
//      bTranspose (input)
//          Should the canvas be transposed?
//
//      bFlipX (input)
//          Should the canvas be flipped in the X direction?
//
//      bFlipY (input)
//          Should the canvas be flipped in the Y direction?
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::setOrientation(
    const Byte bTranspose,
    const Byte bFlipX, 
    const Byte bFlipY
)
{
    //  Build the orientation from/to for use in the orientation table.
    Byte bFrom = m_bOrientation;
    Byte bTo = ( bTranspose ? T : 0 ) | ( bFlipX ? X : 0 ) | ( bFlipY ? Y : 0 );
  
    //  What's required to move from the current orientation to the desired
    //  orientation?
    Byte bChange = sm_abOrientTable[ bFrom ][ bTo ];


    //  Assign the new orientation.
    m_bOrientation = bTo;

    //  If anything has changed then we must notify the bitmaps.
    if( bChange )
    {
        //  An update could be needed so that the client will redraw the
        //  newly oriented bitmaps.
        m_bUpdateNeeded = TRUE;

        //  Inform each bitmap that the canvas orientation has changed.
        for( DWord dwI = 0 ; dwI < m_bitmapList.entries( ) ; dwI += 1 )
        {
            m_bitmapList[ dwI ]->changeOrientation( 
                ( bChange & T ) == T, ( bChange & X ) == X, ( bChange & Y ) == Y
            );
        }
    }

    //  If the transpose has changed then transpose the pages.
    if( bChange & T )
    {
        for( DWord dwI = 0 ; dwI < PAGE_COUNT ; dwI += 1 )
        {
            if( m_apPageList[ dwI ] != NULL )
            {
                m_apPageList[ dwI ]->transpose( );
            }
        }
    }
}

        
///////////////////////////////////////////////////////////////////////////////
//
//  Function: rotate
//
//  Description:
//
//      This member is called to change the orientation of the canvas in
//      a rotational manner relative to the current orientation.
//
//  Parameters:
//
//      dwCount (input)
//          The number of 90 degree rotations to make.  This defaults to 1.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::rotate(
    const DWord dwCount /* = 1 */
)
{
    //  Get the current orientation.
    Byte bOrientation = m_bOrientation;

    //  Rotate the number of times indicated.
    for( DWord dwI = 0 ; dwI < dwCount ; dwI += 1 )
    {
        bOrientation = sm_abRotateTable[ bOrientation ];
    }

    //  Set the new orientation.
    setOrientation( 
        ( bOrientation & T ) == T, 
        ( bOrientation & X ) == X, 
        ( bOrientation & Y ) == Y 
    );
}


        
///////////////////////////////////////////////////////////////////////////////
//
//  Function: flip
//
//  Description:
//
//      This member is called to change the orientation of the canvas in
//      a flip-flop manner relative to the current orientation.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::flip(
)
{
    //  If both axis are flipped then flip none.
    if( ( m_bOrientation & X ) && ( m_bOrientation & Y ) )
    {
        setOrientation( ( m_bOrientation & T ) == T, FALSE, FALSE );
    }
    else
    //  If only the X axis was flip the X back and flip the Y.
    if( m_bOrientation & X )
    {
        setOrientation( ( m_bOrientation & T ) == T, FALSE, TRUE );
    }
    else
    //  If only the Y axis was flipped then flip the X as well.
    if( m_bOrientation & Y )
    {
        setOrientation( ( m_bOrientation & T ) == T, TRUE, TRUE );
    }
    else
    //  Otherwise, flip just the X.
    {
        setOrientation( ( m_bOrientation & T ) == T, TRUE, FALSE );
    }
}

        
///////////////////////////////////////////////////////////////////////////////
//
//  Function: loadSmallFont
//
//  Description:
//
//      This member is called to load the small font.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::loadSmallFont(
)
{
    //  If the font hasn't already been loaded then load it now.
    if( m_pFontSmall == NULL )
    {
        //  Create the graphics set to hold the font.
        m_pFontSmall = new GfxSet( "Small Font" );

        //  We will use a buffer to load the small font ROM into.
        //  The small font is 8x8 pixels and there are 128 characters.  
        Buffer fontBuffer( "Temp", 8 * 128 );
        
        //  The small font is first in the font ROM.
        fontBuffer.loadFile( 0x00000, ".", "small.fnt", 0x00000, 8 * 128 );
    
        //  Set the attributes for the small font.
        m_pFontSmall->setNumber( 128 );
        m_pFontSmall->setDimensions( 8, 8 );
        m_pFontSmall->setBPP( 1 );
        m_pFontSmall->setBitPlanes( 0 );
        m_pFontSmall->setXBits(  0,  1,  2,  3,  4,  5,  6,  7 );
        m_pFontSmall->setYBits(  0,  8, 16, 24, 32, 40, 48, 56 );
        m_pFontSmall->setIncrement( 64 );
    
        //  Decode the font.
        m_pFontSmall->decode( fontBuffer.getBuffer( ) );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: loadLargeFont
//
//  Description:
//
//      This member is called to load the large font.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::loadLargeFont(
)
{
    //  If the font hasn't already been loaded then load it now.
    if( m_pFontLarge == NULL )
    {
        //  Create the graphics set to hold the font.
        m_pFontLarge = new GfxSet( "Large Font" );

        //  We will use a buffer to load the large font ROM into.
        //  The large font is 16x32 pixels and there are 128 characters.  
        Buffer fontBuffer( "Temp", 16 * 32 / 8 * 128 );
        
        //  The large font is first in the font ROM.
        fontBuffer.loadFile( 
            0x00000, ".", "large.fnt", 0x00000, 16 * 32 / 8 * 128 
        );
    
        //  Set the attributes for the large font.
        m_pFontLarge->setNumber( 128 );
        m_pFontLarge->setDimensions( 16, 32 );
        m_pFontLarge->setBPP( 1 );
        m_pFontLarge->setBitPlanes( 0 );
        m_pFontLarge->setXBits(  
              0,   1,   2,   3,   4,   5,   6,   7,
              8,   9,  10,  11,  12,  13,  14,  15
        );
        m_pFontLarge->setYBits(  
              0,  16,  32,  48,  64,  80,  96, 112,
            128, 144, 160, 176, 192, 208, 224, 240, 
            256, 272, 288, 304, 320, 336, 352, 368, 
            384, 400, 416, 432, 448, 464, 480, 496
        );
        m_pFontLarge->setIncrement( 512 );
    
        //  Decode the font.
        m_pFontLarge->decode( fontBuffer.getBuffer( ) );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: removePage
//
//  Description:
//
//      This member is used to remove a page.
//
//  Parameters:
//
//      ePageNumber (input)
//          The page to remove.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::removePage(
    PageNumber ePageNumber
)
{
    //  Check the parameters.
    ASSERT( ePageNumber < PAGE_COUNT );
    ASSERT( m_apPageList[ ePageNumber ] != NULL );

    //  Are we deleting the current page.
    if( m_pPage == m_apPageList[ ePageNumber ] )
    {
        m_pPage = NULL;
    }
    
    //  Delete the specified page.
    delete m_apPageList[ ePageNumber ];
    m_apPageList[ ePageNumber ] = NULL;

    //  If we no longer have a current page then try to find a new one.
    if( m_pPage == NULL )
    {
        //  Find a page to turn to.
        for( DWord dwI = 0 ; dwI < PAGE_COUNT ; dwI += 1 )
        {
            //  If the page exists then turn to it.
            if( m_apPageList[ dwI ] != NULL )
            {
                turnPage( ( PageNumber )dwI );
            }
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: turnPage
//
//  Description:
//
//      This member is called to set the current page on the canvas.
//
//  Parameters:
//
//      ePageNumber (input)
//          The page to turn to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::turnPage(
    PageNumber ePageNumber
)
{
    //  Check the parameters.
    ASSERT( ePageNumber < PAGE_COUNT );
    ASSERT( m_apPageList[ ePageNumber ] != NULL );

    //  Find the page.
    m_pPage = m_apPageList[ ePageNumber ];

    //  An update should be done on the new page.
    m_bUpdateNeeded = TRUE;
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: drawText
//
//  Description:
//
//      This member is called to draw the specified text in the given font.
//
//  Parameters:
//
//      drawText (input)
//          The text to draw.
//
//      pDestBitmap (input/output)
//          The bitmap to draw on to.
//
//      eFont (input)
//          The font to use to draw in.
//  
//      pClipping (input)
//          The clipping region.  This is used as a bounding box for the text.
//
//      pdwColourTable (input)
//          The colour table to use to draw the text.
//
//      bCenter (input)
//          Indicates whether to center the text horizontally within the
//          bounding box.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::drawText( 
    const KString& drawText,
    Bitmap*        pDestBitmap,
    const Font     eFont,
    Clipping*      pClipping,
    DWord*         pdwColourTable,
    const Byte     bCenter         /* = FALSE */
) 
{
    ASSERT( pDestBitmap != NULL );
    ASSERT( pClipping   != NULL );

    //  The width of the clipping box and of the text to blit.
    DWord dwClipWidth;
    DWord dwTextWidth;

    //  The X/Y Position of the current character.
    int32 nX;
    int32 nY;

    //  Get the font to draw in.
    GfxSet& rFont = getFont( eFont );


    //  Initialize.
    dwClipWidth  = pClipping->m_nMaxX - pClipping->m_nMinX + 1;
    nX           = pClipping->m_nMinX;
    nY           = pClipping->m_nMinY;

    //  If the text is to be centered then we need to calculate the starting
    //  position, otherwise, it starts at the left side of the clipping box.
    if( bCenter )
    {
        //  Add in the length of each character.
        dwTextWidth = 0;
        for( int32 nI = 0 ; nI < drawText.length( ) ; nI += 1 )
        {
            dwTextWidth += rFont[ drawText[ nI ] ]->getWidth( );
        }

        //  If the width is less than the clipping box then calculate the
        //  starting point.
        if( dwTextWidth < dwClipWidth )
        {
            nX += ( ( dwClipWidth - dwTextWidth ) / 2 );
        }
    }

    //  Time to draw the text.
    for( int32 nI = 0 ; nI < drawText.length( ) ; nI += 1 )
    {
        //  Blit the current character using the clipping box as a clipping
        //  region and no transparency.
        pDestBitmap->blit(
            rFont[ drawText[ nI ] ],
            nX,
            nY,
            pdwColourTable,
            0,
            0,
            pClipping,
            Bitmap::TRANSPARENCY_NONE,
            0
        );

        //  Move on to the next location.
        nX += rFont[ drawText[ nI ] ]->getWidth( );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getFont
//
//  Description:
//
//      This member is called to return the specified font.
//
//  Parameters:
//
//      eFont (input)
//          The font to return.
//
//  Returns:
//
//      A reference to the specified font.
//
///////////////////////////////////////////////////////////////////////////////
GfxSet& 
Canvas::getFont( 
    Font eFont 
) 
{
    //  The requested font.
    GfxSet* pFont( NULL );


    //  Return the specified font.
    switch( eFont )
    {
        case FONT_SMALL:  
        {
            loadSmallFont( );
            pFont = m_pFontSmall; 
            break;
        }
        case FONT_LARGE:  
        {
            loadLargeFont( );
            pFont = m_pFontLarge; 
            break;
        }
        default: 
        {
            fatalError( "Unsupported Font: %d", eFont ); 
            break;
        }
    }

    //  Return the font.
    return( *pFont );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setColour
//
//  Description:
//
//      This member can be used to set a colour in the palette.  This should
//      be overridden to actually set the colour.  This simply saves the
//      request so that a future unlock can revert back.
//
//  Parameters:
//
//      dwIndex (input)
//          The colour to change.
//
//      dwRed (input)
//          The red component to set.
//
//      dwGreen (input)
//          The green component to set.
//
//      dwBlue (input)
//          The blue component to set.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::setColour( 
    DWord dwIndex, 
    DWord dwRed, 
    DWord dwGreen, 
    DWord dwBlue
) 
{
    //  Check the parameters.
    ASSERT( dwIndex < getNumColours( ) );

    //  Save the colour.
    m_pPaletteColours[ dwIndex ].set( dwRed, dwGreen, dwBlue );

    //  If the colour is not captured then create the new colour.
    if( !m_pbCaptured[ dwIndex ] )
    {
        changeColour( dwIndex, dwRed, dwGreen, dwBlue );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getColour
//
//  Description:
//
//      This member is used to return the index of the colour with the
//      specified RGB values (if it exists).
//
//  Parameters:
//
//      pdwIndex (input)
//          A buffer to store the resulting colour index.
//
//      dwRed (input)
//          The red component to get.
//
//      dwGreen (input)
//          The green component to get.
//
//      dwBlue (input)
//          The blue component to get.
//
//      dwStart (input)
//          The colour to start at.
//
//  Returns:
//
//      TRUE  if the colour is found.
//      FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Canvas::getColour( 
    DWord* pdwIndex, 
    DWord  dwRed, 
    DWord  dwGreen, 
    DWord  dwBlue,
    DWord  dwStart /* = 0 */
)
{
    //  RGB values to compare with.
    DWord dwR;
    DWord dwG;
    DWord dwB;

    ASSERT( dwStart < getNumColours( ) );

    //  Search the list of colours.
    for( DWord dwI = dwStart ; dwI < getNumColours( ) ; dwI += 1 )
    {
        m_pPaletteColours[ dwI ].get( &dwR, &dwG, &dwB );
        if( ( dwR == dwRed ) && ( dwG == dwGreen ) && ( dwB == dwBlue ) )
        {
            ASSERT( pdwIndex != NULL );
            *pdwIndex = dwI;
            return( TRUE );
        }
    }
    return( FALSE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getFreeColour
//
//  Description:
//
//      This member is used to return the index of the first colour that
//      is unlocked.
//
//  Parameters:
//
//      pdwIndex (input)
//          A buffer to store the resulting colour index.
//
//  Returns:
//
//      TRUE  if the colour is found.
//      FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Canvas::getFreeColour( 
    DWord* pdwIndex
)
{
    //  Search the list of colours.
    for( DWord dwI = 0 ; dwI < getNumColours( ) ; dwI += 1 )
    {
        if( !m_pPaletteColours[ dwI ].isLocked( ) )
        {
            ASSERT( pdwIndex != NULL );
            *pdwIndex = dwI;
            return( TRUE );
        }
    }
    return( FALSE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClosestColour
//
//  Description:
//
//      This member is used to return the index of the colour in the
//      palette that is closest to the one specified.
//
//      Closeness is based on the weighted difference between the R/G/B values.
//
//  Parameters:
//
//      pdwIndex (input)
//          A buffer to store the resulting colour index.
//
//      dwRed (input)
//          The red component to get.
//
//      dwGreen (input)
//          The green component to get.
//
//      dwBlue (input)
//          The blue component to get.
//
//      dwStart (input)
//          The colour to start at.
//
//  Returns:
//
//      TRUE  if the colour is found.
//      FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Canvas::getClosestColour( 
    DWord* pdwIndex, 
    DWord  dwRed, 
    DWord  dwGreen, 
    DWord  dwBlue,
    DWord  dwStart /* = 0 */
)
{
    //  RGB values to compare with.
    DWord dwR;
    DWord dwG;
    DWord dwB;

    //  The difference between the colours and the best
    //  difference between colours that we've found so far.
    DWord dwDifference;
	DWord dwBestDifference = 0xffffff;

    ASSERT( pdwIndex != NULL );
    ASSERT( dwStart < getNumColours( ) );

    //  Start at the first colour.
    *pdwIndex = 0;

    //  Search the list of colours.
    for( DWord dwI = dwStart ; dwI < getNumColours( ) ; dwI += 1 )
    {
        m_pPaletteColours[ dwI ].get( &dwR, &dwG, &dwB );

        dwDifference = 
            sm_anRedLUT[ abs( dwR - dwRed ) ] +
            sm_anGreenLUT[ abs( dwG - dwGreen ) ] +
            sm_anBlueLUT[ abs( dwB - dwBlue ) ];

        if( dwDifference < dwBestDifference )
        {
            dwBestDifference = dwDifference;
            *pdwIndex = dwI;
        }
    }
    return( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: lockColour
//
//  Description:
//
//      This member locks the specified colour.
//
//  Parameters:
//
//      dwIndex (input)
//          The colour to change.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::lockColour( 
    DWord dwIndex
) 
{
    //  Check the parameters.
    ASSERT( dwIndex < getNumColours( ) );

    //  Lock the colour.
    m_pPaletteColours[ dwIndex ].lock( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: unlockColour
//
//  Description:
//
//      This member unlocks the specified colour.
//
//  Parameters:
//
//      dwIndex (input)
//          The colour to unlock.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::unlockColour( 
    DWord dwIndex
) 
{
    //  Check the parameters.
    ASSERT( dwIndex < getNumColours( ) );

    //  Unlock the colour.
    m_pPaletteColours[ dwIndex ].unlock( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: captureColour
//
//  Description:
//
//      This member overrides the colour specified with the new colour values.
//
//  Parameters:
//
//      dwIndex (input)
//          The colour to change.
//
//      dwRed (input)
//          The red component to set.
//
//      dwGreen (input)
//          The green component to set.
//
//      dwBlue (input)
//          The blue component to set.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::captureColour( 
    DWord dwIndex, 
    DWord dwRed, 
    DWord dwGreen, 
    DWord dwBlue
) 
{
    //  Check the parameters.
    ASSERT( dwIndex < getNumColours( ) );

    //  Capture the colour.
    m_pbCaptured[ dwIndex ] = TRUE;
    changeColour( dwIndex, dwRed, dwGreen, dwBlue );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: freeColours
//
//  Description:
//
//      This member frees all captured colours.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Canvas::freeColours( 
) 
{
    //  Loop through all the colours making sure they are no longer captured
    //  and reflect the real colours.
    for( DWord dwI = 0 ; dwI < getNumColours( ) ; dwI += 1 )
    {
        //  Create the real colour.
        DWord dwRed;
        DWord dwGreen;
        DWord dwBlue;
        m_pPaletteColours[ dwI ].get( &dwRed, &dwGreen, &dwBlue );
        changeColour( dwI, dwRed, dwGreen, dwBlue );

        //  Mark this colour as uncaptured.
        m_pbCaptured[ dwI ] = FALSE;
    }
}
