///////////////////////////////////////////////////////////////////////////////
//
//  File:    network.cpp
//
//  Class:   Network
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The Network class is an abstract base class encapsulating a network
//      connection between 2 or more Replay processes.  A derived class 
//      should be created for each network protocol.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "network.h"
#include "config.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////
DWord Network::sm_dwMaxConnections = 8;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: Network
//
//  Description:
//
//      This is the main constructor for a network object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Network::Network(
    const KString&  iName
)
:
    RepBase            ( iName ),
    m_bEnabled         ( TRUE ),
    m_eConnectType     ( UNKNOWN ),
    m_dwNumConnections ( 0 )
{
    //  Is the network disabled?
    if( Configuration::s_instance( ).getParam( "-nonet" ) )
    {
        m_bEnabled = FALSE;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Network
//
//  Description:
//
//      This is the destructor for a network object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Network::~Network(
)
{
    //  If we were connected then close the network.
    if( m_eConnectType != UNKNOWN )
    {
        closeNetwork( );
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: openNetwork
//
//  Description:
//
//      This opens the network.
//
//  Parameters:
//
//      eType (input)
//          Indicates whether or not the connection should be opened as
//          client or server.
//
//  Returns:
//
//      TRUE if the network was successfully opened, FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Network::openNetwork(
    ConnectType eType
)
{
    ASSERT( m_bEnabled );

    //  Make sure either client of server was specified.
    CONFIRM( eType != UNKNOWN, "Network must be opened as client or server" );

    //  Set the type.
    m_eConnectType = eType;

    return( TRUE );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: closeNetwork
//
//  Description:
//
//      This shuts down the network.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Network::closeNetwork(
)
{
    //  The network must be open to be able to be closed.
    CONFIRM( isOpen( ), "Cannot close a network that hasn't been opened." );

    m_eConnectType = UNKNOWN;
}
