///////////////////////////////////////////////////////////////////////////////
//
//  File:    page.cpp
//
//  Class:   Page
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The Page class represents a viewable screen for the Replay 
//      application.  Pages are contained on the Canvas of the application.
//      This is an abstract base class for platform specific derived classes 
//      that implement the interface for the hardware of the platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "page.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Page
//
//  Description:
//
//      This is the main constructor for a page object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwWidth (input)
//          The width of the page.
//
//      dwHeight (input)
//          The height of the page.
//
//      bTransposed (input)
//          Indicates that the page should be created with the
//          dimensions swapped.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Page::Page(
    const KString&  iName,
    const DWord     dwWidth,
    const DWord     dwHeight,
    const Byte      bTransposed  /* = FALSE */
)
:
    RepBase           ( iName ),
    m_dwWidth         ( bTransposed ? dwHeight : dwWidth ),
    m_dwHeight        ( bTransposed ? dwWidth : dwHeight ),
    m_bTransposed     ( bTransposed )
{
    //  Check the parameters.
    CONFIRM( 
        ( dwWidth > 0 ) && ( dwHeight > 0 ), 
        "Invalid dimensions (%ldx%ld) for page.",
        dwWidth,
        dwHeight
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Page
//
//  Description:
//
//      This is the destructor for a page object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Page::~Page(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: transpose
//
//  Description:
//
//      This is called to transpose (i.e. swap X/Y) the page.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Page::transpose(
)
{
    //  Swap the dimensions.
    DWord dwSwap = m_dwWidth;
    m_dwWidth    = m_dwHeight;
    m_dwHeight   = dwSwap;

    //  Toggle the transposed flag.
    m_bTransposed = m_bTransposed ? FALSE : TRUE;
}
