///////////////////////////////////////////////////////////////////////////////
//
//  File:       replay.h
//
//  Class:      Replay - Concrete
//  
//  Hierarchy:  Replay - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This object is the main Replay application object.  It controls the
//      running of the application.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _replay_h_
#define _replay_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "kplist.h"
#include "reptypes.h"
#include "repbase.h"
#include "meddler.h"


//  Forward class declarations.
class ReplayFactory;
class GameFactory;
class CPUFactory;
class Canvas;
class Keyboard;
class Joystick;
class Mouse;
class Sound;
class Disk;
class Network;
class Clock;
class Bitmap;
class ColourTable;
class GameInfo;
class Game;


///////////////////////////////////////////////////////////////////////////////
//  Replay Class Definition
///////////////////////////////////////////////////////////////////////////////
class Replay : public RepBase
{
    public:

        //  Deletion.
        virtual ~Replay ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Singleton Access.
        static Replay& s_instance ( );
   
        //  Interaction.
        int32 run ( );

        //  Modification.
        void newMeddler    ( Meddler* pMeddler );
        void deleteMeddler ( );

        //  Access.
        Canvas*      getCanvas      ( ) { return( m_pCanvas );     }
        Keyboard*    getKeyboard    ( ) { return( m_pKeyboard );   }
        Joystick*    getJoystick    ( ) { return( m_pJoystick );   }
        Mouse*       getMouse       ( ) { return( m_pMouse );      }
        Sound*       getSound       ( ) { return( m_pSound );      }
        Disk*        getDisk        ( ) { return( m_pDisk );       }
        Network*     getNetwork     ( ) { return( m_pNetwork );    }
        Clock*       getClock       ( ) { return( m_pClock );      }
        Meddler*     getMeddler     ( ) { return( m_pMeddler );    }
        CPUFactory*  getCPUFactory  ( ) { return( m_pCPUFactory ); }
        GameFactory* getGameFactory ( ) { return( m_pGameFactory); }
        Game*        getGame        ( ) { return( m_pGame );       }
        void         setGame        ( const DWord dwIdx );
        void         clearGame      ( );

        //  The following should be called when the status of the system
        //  can be updated.
        Byte update ( );

    protected:

        //  Creation is protected since this is a singleton.
        Replay ( const KString& iName );

        //  Startup.
        void initializeCanvas   ( ); 
        void buildGameList      ( );

        //  Utility.
        void drawDecorations    ( Bitmap* pDest );

        //  Member Data.
        ReplayFactory*         m_pReplayFactory;
        Canvas*                m_pCanvas;
        Keyboard*              m_pKeyboard;
        Joystick*              m_pJoystick;
        Mouse*                 m_pMouse;
        Sound*                 m_pSound;
        Disk*                  m_pDisk;
        Network*               m_pNetwork;
        Clock*                 m_pClock;
        Meddler*               m_pMeddler;
        KString                m_runGame;
        CPUFactory*            m_pCPUFactory;
        KPtrList<GameInfo>     m_gameInfoList;
        GameFactory*           m_pGameFactory;
        Game*                  m_pGame;
        ColourTable*           m_pDecColTable;
        Byte                   m_bDecColIndex;
        Byte                   m_bThrottle; 
        Byte                   m_bShowFPS;

    private:

        //  Initial Checks.
        void checkBitByteOrder ( ) const;
};

///////////////////////////////////////////////////////////////////////////////
//  Replay Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
void
Replay::newMeddler(
    Meddler* pMeddler
)
{
    ASSERT( m_pMeddler == NULL );
    m_pMeddler = pMeddler;
}

inline
void
Replay::deleteMeddler(
)
{
    ASSERT( m_pMeddler != NULL );
    m_pMeddler->terminate( );
    delete m_pMeddler;
    m_pMeddler = NULL;
}
    
#endif
