///////////////////////////////////////////////////////////////////////////////
//
//  File:       select.h
//
//  Class:      SelectScreen - Concrete
//  
//  Hierarchy:  SelectScreen - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class is used by the Replay application to select a game from
//      the list of games available.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _select_h_
#define _select_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "kplist.h"
#include "replay.h"

//  Forward class declarations.
class GameInfo;
class Game;
class Canvas;
class Bitmap;
class GfxSet;
class ColourTable;
class SSStateBase;
class SSStateMainMenu;
class SSStateInitialize;
class SSStateDemo;
class SSStatePause;
class SSStateDipSwitch;
class SSStateServer;
class SSStateClient;
class SSStateKeyConfig;
class SSStateJoyConfig;
class SSStateMissingFiles;
class SSStateMessage;


///////////////////////////////////////////////////////////////////////////////
//  SelectScreen Class Definition
///////////////////////////////////////////////////////////////////////////////
class SelectScreen : public RepBase
{
    public:

        //  The possible states that the selection screen can be in.
        typedef enum
        {
            STATE_PREVIOUS,

            STATE_MAIN_MENU,
            STATE_INITIALIZE,
            STATE_DEMO,
            STATE_PAUSE,
            STATE_KEY_CONFIG,
            STATE_JOY_CONFIG,
            STATE_DIP_SWITCH,
            STATE_SERVER,
            STATE_CLIENT,
            STATE_MISSING_FILES,
            STATE_MESSAGE
        }
        State;

        //  Creation/Deletion.
        SelectScreen ( 
            const KString& iName, const KPtrList<GameInfo>& gameInfoList 
        );
        virtual ~SelectScreen ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Selection Screen execution.
        void execute ( State eState );
        void update  ( );

        //  Set the state of the selection screen.
        void setState ( State eState );

        //  Get and set the current game.
        DWord getNumGames ( ) const;
        DWord getCurGame  ( ) const;
        void  setCurGame  ( const DWord    dwGame );
        void  setCurGame  ( const KString& gameId );
        Game* getGame     ( ) const;
        void  setGame     ( ) const;
        void  clearGame   ( ) const;

        //  Get the default state bitmap.
        Bitmap* getDefStateBitmap ( ) const;

    protected:

        //  Utility.
        void setColours   ( );
        void startUpPage  ( );
        void shutDownPage ( );

        //  Member Data.
        State                           m_eState;
        SSStateBase*                    m_pState;
        SSStateMainMenu*                m_pStateMainMenu;
        SSStateInitialize*              m_pStateInitialize;
        SSStateDemo*                    m_pStateDemo;
        SSStatePause*                   m_pStatePause;
        SSStateKeyConfig*               m_pStateKeyConfig;
        SSStateJoyConfig*               m_pStateJoyConfig;
        SSStateDipSwitch*               m_pStateDipSwitch;
        SSStateServer*                  m_pStateServer;
        SSStateClient*                  m_pStateClient;
        SSStateMissingFiles*            m_pStateMissingFiles;
        SSStateMessage*                 m_pStateMessage;
        const KPtrList<GameInfo>&       m_gameInfoList;
        DWord                           m_dwCurGame;
        Canvas*                         m_pCanvas; 
        Bitmap*                         m_pScreenBitmap;
        Bitmap*                         m_pHelpBitmap;
        Bitmap*                         m_pListBitmap;
        Bitmap*                         m_pCreditsBitmap;
        GfxSet*                         m_pFontSmall;
        GfxSet*                         m_pFontLarge;
        ColourTable*                    m_pHelpColTable;
        ColourTable*                    m_pListColTable;
        ColourTable*                    m_pListHiColTable;
        ColourTable*                    m_pCreditsColTable;
        ColourTable*                    m_pStateColTable;
        Bitmap*                         m_pDefStateBitmap;
        Bitmap*                         m_pStateBitmap;
        int32                           m_nStateX;
        int32                           m_nStateY;
        DWord                           m_dwStateWidth;
        DWord                           m_dwStateHeight;
        Byte                            m_bDrawBG;

    private:

};

///////////////////////////////////////////////////////////////////////////////
//  SelectScreen Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
DWord
SelectScreen::getNumGames(
) const
{
    return( m_gameInfoList.entries( ) );
}

inline
DWord
SelectScreen::getCurGame(
) const
{
    return( m_dwCurGame );
}

inline
Game*
SelectScreen::getGame(
) const
{
    return( Replay::s_instance( ).getGame( ) );
}

inline
void
SelectScreen::setGame(
) const
{
    Replay::s_instance( ).setGame( m_dwCurGame );
}

inline
void
SelectScreen::clearGame(
) const
{
    Replay::s_instance( ).clearGame( );
}

inline
Bitmap*
SelectScreen::getDefStateBitmap(
) const
{
    return( m_pDefStateBitmap );
}

#endif
