///////////////////////////////////////////////////////////////////////////////
//
//  File:    sssbase.cpp
//
//  Class:   SSStateBase
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      Replay's selection screen uses a State Pattern to allow for easy
//      flexibility and extensibility of game options and this class serves
//      as a base class for all of the states of the selection screen.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "sssbase.h"
#include "replay.h"
#include "keyb.h"
#include "select.h"
#include "game.h"
#include "gfxset.h"
#include "bitmap.h"
#include "ctable.h"
#include "clip.h"




///////////////////////////////////////////////////////////////////////////////
//
//  Function: SSStateBase
//
//  Description:
//
//      This is the main constructor for the selection screen base state
//      object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pSelectScreen (input)
//          The selection screen the state belongs to.
//
//      pCanvas (input)
//          The canvas used by the selection screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::SSStateBase(
    const KString& iName,
    SelectScreen*  pSelectScreen,
    Canvas*        pCanvas
)
:
    RepBase              ( iName ),
    m_pSelectScreen      ( pSelectScreen ),
    m_pCanvas            ( pCanvas ),
    m_pKeyboard          ( Replay::s_instance( ).getKeyboard( ) ),
    m_pGameInfo          ( NULL ),
    m_pPreviousState     ( NULL )
{
    //  Check the parameters.
    ASSERT( m_pSelectScreen != NULL );
    ASSERT( m_pCanvas       != NULL );
    ASSERT( m_pKeyboard     != NULL );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SSStateBase
//
//  Description:
//
//      This is the destructor for the selection screen base state object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::~SSStateBase(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function:  setGameInfo
//
//  Description:
//
//      This is called when a new game has been selected.
//
//  Parameters:
//
//      pGameInfo (input)
//          The information for the new game.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateBase::setGameInfo(
    GameInfo* pGameInfo
)
{
    ASSERT( pGameInfo != NULL );
    m_pGameInfo = pGameInfo;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function:  setState
//
//  Description:
//
//      This is called when the state has just been set as the current state
//      of the selection screen.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateBase::setState(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: drawHelp
//
//  Description:
//
//      This is used to draw the specified text in the help bitmap.
//
//  Parameters:
//
//      pBitmap (input)
//          The help bitmap.
//
//      pFont (input)
//          The font to draw with.
//
//      pColourTable (input)
//          The colour table to draw with.
//
//      ppstrHelpText (input)
//          The text to draw.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateBase::drawHelp(
    Bitmap*      pBitmap,
    Canvas::Font eFont,
    ColourTable* pColourTable,
    char**       ppstrHelpText 
)
{
    //  Check the arguments.
    ASSERT( pBitmap       != NULL );
    ASSERT( pColourTable  != NULL );

    //  The height of the font.
    int32 nFontHeight;

    //  The bounding box for the text being drawn.
    Clipping clipping( 
        0, pBitmap->getWidth( ) - 1, 0, pBitmap->getHeight( ) - 1 
    );

    //  Get the height of the font.  (The first character in the font
    //  will do fine).
    nFontHeight = m_pCanvas->getFont( eFont )[ 0 ]->getHeight( );

    //  Draw the help text.
    for( DWord dwI = 0 ; ppstrHelpText[ dwI ] != NULL ; dwI += 1 )
    {
        m_pCanvas->drawText(
            ppstrHelpText[ dwI ], pBitmap, eFont, &clipping, *pColourTable
        );
        clipping.m_nMinY += nFontHeight;
    }
}
        





///////////////////////////////////////////////////////////////////////////////
//
//  Function: checkKeys
//
//  Description:
//
//      This is used to check the default keys that are used in many states.
//
//  Parameters:
//
//      dwActions (input)
//          Indicates which actions are allowed.
//
//  Returns:
//
//      The action based on the key pressed.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStateBase::checkKeys( 
    DWord dwActions /* = 0xffff */
)
{
    //  Main Menu?
    if( 
        ( dwActions & DEF_MAIN_MENU ) && 
        ( m_pKeyboard->switchOn( Keyboard::KEY__ESC ) )
    )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ESC );
        m_pSelectScreen->setState( SelectScreen::STATE_MAIN_MENU );
    }
    else
    //  Run Game?
    if( 
        ( dwActions & DEF_GAME_RUN ) && 
        ( m_pKeyboard->switchOn( Keyboard::KEY__ENTER ) )
    )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ENTER );
        return( SSS_RUN );
    }
    else
    //  Move up?
    if(
        ( dwActions & DEF_GAME_UP ) &&
        ( m_pKeyboard->switchOnWait( Keyboard::KEY__UP ) ) &&
        ( m_pSelectScreen->getCurGame( ) > 0 )
    )
    {
        //  Move on to the previous game.
        m_pSelectScreen->setCurGame( m_pSelectScreen->getCurGame( ) - 1 );
        m_pSelectScreen->setState( SelectScreen::STATE_INITIALIZE );
    }
    else
    //  Move down?
    if(
        ( dwActions & DEF_GAME_DOWN ) &&
        ( m_pKeyboard->switchOnWait( Keyboard::KEY__DOWN ) ) &&
        ( m_pSelectScreen->getCurGame( ) < m_pSelectScreen->getNumGames( ) - 1 )
    )
    {
        //  Move on to the next game.
        m_pSelectScreen->setCurGame( m_pSelectScreen->getCurGame( ) + 1 );
        m_pSelectScreen->setState( SelectScreen::STATE_INITIALIZE );
    }
    else
    //  Flip Screen.
    if( m_pKeyboard->switchOn( Keyboard::KEY__F9 ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__F9 );
        m_pCanvas->flip( );
    }
    else
    //  Rotate Screen.
    if( m_pKeyboard->switchOn( Keyboard::KEY__F10 ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__F10 );
        m_pCanvas->rotate( );
    }
    

    //  If nothing exciting happening then just continue.
    return( SSS_CONTINUE );
}
