///////////////////////////////////////////////////////////////////////////////
//
//  File:    ssscli.cpp
//
//  Class:   SSStateClient
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the selection screen state responsible for synchronizing
//      with a server when Replay is acting as a client.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <stdlib.h>

//  Application Headers.
#include "ssscli.h"
#include "sssmsg.h"
#include "select.h"
#include "replay.h"
#include "canvas.h"
#include "bitmap.h"
#include "keyb.h"
#include "network.h"
#include "game.h"
#include "gameinfo.h"
#include "dip.h"
#include "client.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: SSStateClient
//
//  Description:
//
//      This is the main constructor for the selection screen client
//      state object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pSelectScreen (input)
//          The selection screen the state belongs to.
//
//      pCanvas (input)
//          The canvas used by the selection screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateClient::SSStateClient(
    const KString& iName,
    SelectScreen*  pSelectScreen,
    Canvas*        pCanvas
)
:
    SSStateList      ( iName, pSelectScreen, pCanvas, TRUE ),
    m_pNetwork       ( Replay::s_instance( ).getNetwork( ) ),
    m_eSynchro       ( SYNCHRO_NONE ),
    m_bAbort         ( FALSE ),
    m_pMsg           ( NULL )
{
    ASSERT( m_pNetwork != NULL );

    //  Create the network message.
    m_pMsg = new Network::Msg;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SSStateClient
//
//  Description:
//
//      This is the destructor for the selection screen client state
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateClient::~SSStateClient(
)
{
    //  Dispose of the network message.
    delete m_pMsg;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the select screen client state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
SSStateClient::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "SSStateClient" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setState
//
//  Description:
//
//      This is called when this state has just become the current state.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateClient::setState(
)
{
    //  Call the base class.
    SSStateBase::setState( );

    //  If the network game was aborted then go back to the main menu.
    if( m_bAbort )
    {
        m_bAbort = FALSE;
        m_pSelectScreen->setState( SelectScreen::STATE_MAIN_MENU );
        return;
    }

    //  Clear the list.
    clearHeader( );
    clearLines( );

    //  Begin synchronization.
    m_eSynchro = SYNCHRO_SEND_START;
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This is used to execute the client state.  The base method
//      does the display.  This member function will be responsible for
//      synchronizing with the server.
//
//  Parameters:
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      An action indicating what the selection screen should do.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStateClient::execute(
    ColourTable* pColourTable 
)
{
    //  Call the base class.
    SSStateList::execute( pColourTable );

    //  Allow the user to scroll the lines.
    SSStateList::checkForScroll( );

    //  Allow for the standard keys (like rotate, flip, etc.).
    SSStateBase::checkKeys( DEF_NONE );

    //  Exit?  Disconnect from the server and inform the user.
    if( m_pKeyboard->switchOn( Keyboard::KEY__ESC ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ESC );

        //  Kill the network.
        m_pNetwork->closeNetwork( );

        //  Note that we aborted so that we don't pass this way again.
        m_bAbort = TRUE;

        //  Inform the user of the fact.
        SSStateMessage::s_setMessage( SSStateMessage::MSG_NET_STOPPED );
        m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );
    }
    //  Is the game ready to go?
    else
    if( m_eSynchro == SYNCHRO_NONE )
    {
        //  Add a net game client meddler to Replay.
        ASSERT( m_pSelectScreen->getGame( ) != NULL );
        Meddler* pMeddler = NetGameClientMeddler::s_build( 
            "Client", m_pSelectScreen->getGame( )
        );
        Replay::s_instance( ).newMeddler( pMeddler );

        //  Run the game.
        return( SSS_RUN );
    } 
    //  Continue processing connections.
    else
    {
        //  Move the synchronization along.
        continueSynchro( );
    }

    return( SSS_CONTINUE );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: continueSynchro
//
//  Description:
//
//      This is called to continue the synchronization with the
//      server that we've just connected to.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateClient::continueSynchro(
)
{
    //  What are we currently doing?
    switch( m_eSynchro )
    {
        case SYNCHRO_SEND_START: synchroSendStart( ); break;
        case SYNCHRO_RECV_GAME:  synchroRecvGame( );  break;
        case SYNCHRO_RECV_DIPS:  synchroRecvDips( );  break;
        case SYNCHRO_SEND_READY: synchroSendReady( ); break;
        case SYNCHRO_RECV_GO:    synchroRecvGo( );    break;
        case SYNCHRO_NONE:                            break;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: synchroSendStart
//
//  Description:
//
//      This is called to send a start message to the server.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateClient::synchroSendStart(
)
{
    //  Create the message.
    m_pMsg->eType = Network::MSG_OK;

    //  Send the message.
    m_pNetwork->writeMsg( 0, m_pMsg );

    //  Move our state along.
    m_eSynchro = SYNCHRO_RECV_GAME;

    //  Indicate that we've sent the start to the server.
    m_lineList.add( new KString( "Initiating server contact." ) );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: synchroRecvGame
//
//  Description:
//
//      This is called to receive the id of the current game from the server.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateClient::synchroRecvGame(
)
{
    //  Read a message.
    m_pNetwork->readMsg( 0, m_pMsg );

    //  If there was no message then just continue on.
    if( m_pMsg->eType == Network::MSG_NONE )
    {
        return;
    }
    //  If the server closed the connection then return abruptly.
    else
    if( m_pMsg->eType == Network::MSG_CLOSE )
    {
        //  Pretend that we aborted so we don't come back this way.
        m_bAbort = TRUE;

        //  Inform the user of the fact.
        SSStateMessage::s_setMessage( SSStateMessage::MSG_NET_STOPPED );
        m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );
    }
    //  If we received the Game message then we should set the game.
    else
    if( m_pMsg->eType == Network::MSG_GAME )
    {
        //  Cast the message to the game type.
        Network::MsgGame* pMsgGame = ( Network::MsgGame* )m_pMsg;

        //  Indicate that we're starting the game.
        sprintf( m_strLine, "Game is %s.", pMsgGame->strGameName );
        m_lineList.add( new KString( m_strLine ) );

        //  Change to the new game.
        m_pSelectScreen->setCurGame( pMsgGame->strGameName );
        if( m_pSelectScreen->getGame( ) == NULL )
        {
            m_pSelectScreen->setGame( );
        }

        //  If the game isn't missing any files then start it up, otherwise
        //  we must abort the net game.
        if( m_pGameInfo->requiredFilesAvailable( ) )
        {
            while( m_pSelectScreen->getGame( )->startUp( ) )
            {
                //  All of the work is done in the loop.
            }
            
            //  Move on.
            m_eSynchro = SYNCHRO_RECV_DIPS;
        }
        else
        {
            //  Kill the network.
            m_pNetwork->closeNetwork( );

            //  Pretend that we aborted so we don't come back this way.
            m_bAbort = TRUE;

            //  Inform the user of the fact.
            SSStateMessage::s_setMessage( SSStateMessage::MSG_NET_MISSING );
            m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );
        }
    }
    //  Unexpected message.
    else
    {
        fatalError( "Unexpected message from server %d.", m_pMsg->eType );
    }
}
        


///////////////////////////////////////////////////////////////////////////////
//
//  Function: synchroRecvDips
//
//  Description:
//
//      This is called to receive the dip switch values of the current game 
//      from the server.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateClient::synchroRecvDips(
)
{
    //  The dip switch list.
    ASSERT( m_pSelectScreen->getGame( ) != NULL );
    const KPtrList<DipSwitch>& dipSwitchList = 
        m_pSelectScreen->getGame( )->getDipSwitchList( );

    //  Read a message.
    m_pNetwork->readMsg( 0, m_pMsg );

    //  If there was no message then just continue on.
    if( m_pMsg->eType == Network::MSG_NONE )
    {
        return;
    }
    //  If the server closed the connection then return abruptly.
    else
    if( m_pMsg->eType == Network::MSG_CLOSE )
    {
        //  Pretend that we aborted so we don't come back this way.
        m_bAbort = TRUE;

        //  Inform the user of the fact.
        SSStateMessage::s_setMessage( SSStateMessage::MSG_NET_STOPPED );
        m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );
    }
    //  If we received a dip switch message then we should set the value of
    //  the dip switch.
    else
    if( m_pMsg->eType == Network::MSG_DIP )
    {
        //  Cast the message to the dip switch message.
        Network::MsgDip* pMsgDip = ( Network::MsgDip* )m_pMsg;

        //  Set the value of the dip switch.
        dipSwitchList[ pMsgDip->bDipIndex ]->setValue( pMsgDip->bValue );
    }
    //  If we received an OK message then the dip switches have been sent
    //  and we're ready to go.
    else
    if( m_pMsg->eType == Network::MSG_OK )
    {
        //  Indicate that the server is ready to go.
        m_lineList.add( new KString( "Server OK with client." ) );

        m_eSynchro = SYNCHRO_SEND_READY;
    }
    //  Unexpected message.
    else
    {
        fatalError( "Unexpected message from server %d.", m_pMsg->eType );
    }
}
        


///////////////////////////////////////////////////////////////////////////////
//
//  Function: synchroSendReady
//
//  Description:
//
//      This is called to indicate to the server that we're ready to go.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateClient::synchroSendReady(
)
{
    //  Create the message.
    m_pMsg->eType = Network::MSG_OK;
 
    //  Send the message.
    m_pNetwork->writeMsg( 0, m_pMsg );

    //  Wait for the signal to start.
    m_eSynchro = SYNCHRO_RECV_GO;

    //  Indicate that we've sent the ready message to the server.
    m_lineList.add( new KString( "Client OK with server." ) );

    //  Indicate that we're waiting for the game start signal.
    m_lineList.add( new KString( "Waiting for start signal..." ) );
}
        


///////////////////////////////////////////////////////////////////////////////
//
//  Function: synchroRecvGo
//
//  Description:
//
//      This is called to receive the message from the server that we're
//      ready to start.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateClient::synchroRecvGo(
)
{
    //  Read a message.
    m_pNetwork->readMsg( 0, m_pMsg );

    //  If there was no message then just continue on.
    if( m_pMsg->eType == Network::MSG_NONE )
    {
        return;
    }
    //  If the server closed the connection then return abruptly.
    else
    if( m_pMsg->eType == Network::MSG_CLOSE )
    {
        //  Pretend that we aborted so we don't come back this way.
        m_bAbort = TRUE;

        //  Inform the user of the fact.
        SSStateMessage::s_setMessage( SSStateMessage::MSG_NET_STOPPED );
        m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );
    }
    //  If we received a dip switch message then we should set the value of
    //  the dip switch.
    else
    if( m_pMsg->eType == Network::MSG_OK )
    {
        //  Indicate that the server is ready to go.
        m_lineList.add( new KString( "Here we go!" ) );

        m_eSynchro = SYNCHRO_NONE;
    }
    //  Unexpected message.
    else
    {
        fatalError( "Unexpected message from server %d.", m_pMsg->eType );
    }
}
        





///////////////////////////////////////////////////////////////////////////////
//
//  Function: fillHelp
//
//  Description:
//
//      This is used to fill in the help area for this state.  The following
//      keys are allowed during this state:
//          ESC Abort
//
//  Parameters:
//
//      pBitmap (input)
//          The help bitmap.
//
//      eFont (input)
//          The font to draw with.
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateClient::fillHelp(
    Bitmap*      pBitmap,
    Canvas::Font eFont,
    ColourTable* pColourTable 
)
{
    //  Check the arguments.
    ASSERT( pBitmap       != NULL );
    ASSERT( pColourTable  != NULL );

    //  The help text.
    static char* ppstrHelpText[ ] = 
    {
        "ESC Abort",
        NULL
    };

    //  Draw the text.
    drawHelp( pBitmap, eFont, pColourTable, ppstrHelpText );
}
