///////////////////////////////////////////////////////////////////////////////
//
//  File:    sssdemo.cpp
//
//  Class:   SSStateDemo
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the selection screen state responsible for showing the
//      demo of the current game while the selection is being made.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "sssdemo.h"
#include "select.h"
#include "keyb.h"
#include "game.h"
#include "screen.h"


///////////////////////////////////////////////////////////////////////////////
//
//  Function: SSStateDemo
//
//  Description:
//
//      This is the main constructor for the selection screen demo 
//      state object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pSelectScreen (input)
//          The selection screen the state belongs to.
//
//      pCanvas (input)
//          The canvas used by the selection screen.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateDemo::SSStateDemo(
    const KString& iName,
    SelectScreen*  pSelectScreen,
    Canvas*        pCanvas
)
:
    SSStateBase     ( iName, pSelectScreen, pCanvas )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SSStateDemo
//
//  Description:
//
//      This is the destructor for the selection screen demo state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateDemo::~SSStateDemo(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the select screen demo state object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
SSStateDemo::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "SSStateDemo" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This is used to execute the demo state.  We execute a slice of
//      the current game and check for game-related keys being pressed
//      (e.g. F1 for Game Info, etc.).
//
//  Parameters:
//
//      pColourTable (input)
//          The colour table to draw with (ignored).
//
//  Returns:
//      An action indicating what the selection screen should do.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStateDemo::execute(
    ColourTable* /* pColourTable is unused */
)
{
    //  We should have a selection screen and a game by now.
    ASSERT( m_pSelectScreen             != NULL );
    ASSERT( m_pSelectScreen->getGame( ) != NULL );
 
        
    //  Continue running the game.
    m_pSelectScreen->getGame( )->run( );

    //  Return the result of checking the default keys.
    return( checkKeys( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: fillHelp
//
//  Description:
//
//      This is used to fill in the help area for this state.  The following
//      keys are allowed during this state:
//          ESC Main Menu
//          UP  Prev Game
//          DN  Next Game
//          RET Run Game
//
//  Parameters:
//
//      pBitmap (input)
//          The help bitmap.
//
//      eFont (input)
//          The font to draw with.
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateDemo::fillHelp(
    Bitmap*      pBitmap,
    Canvas::Font eFont,
    ColourTable* pColourTable 
)
{
    //  Check the arguments.
    ASSERT( pBitmap       != NULL );
    ASSERT( pColourTable  != NULL );

    //  The help text.
    static char* ppstrHelpText[ ] = 
    {
        "ESC Main Menu",  
        "UP  Prev Game",
        "DN  Next Game",
        "Ret Run  Game",
        NULL
    };

    //  Draw the text.
    drawHelp( pBitmap, eFont, pColourTable, ppstrHelpText );
}
        


///////////////////////////////////////////////////////////////////////////////
//
//  Function: getScreen
//
//  Description:
//
//      This returns the bitmap of the current game being demo'ed.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      A pointer to the screen bitmap of the state.
//
///////////////////////////////////////////////////////////////////////////////
Bitmap* 
SSStateDemo::getScreen ( 
)
{
    ASSERT( m_pSelectScreen->getGame( ) != NULL );
    return( m_pSelectScreen->getGame( )->getScreen( )->getBitmap( ) );
} 
