///////////////////////////////////////////////////////////////////////////////
//
//  File:    sssdip.cpp
//
//  Class:   SSStateDipSwitch
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the selection screen state responsible for setting the
//      dip switches of the current game.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "sssdip.h"
#include "select.h"
#include "canvas.h"
#include "registry.h"
#include "bitmap.h"
#include "keyb.h"
#include "gameinfo.h"
#include "game.h"
#include "dip.h"
#include "setting.h"




///////////////////////////////////////////////////////////////////////////////
//  File Constants.
///////////////////////////////////////////////////////////////////////////////
//  Some strings.
static const char* NO_DIPS  = "No Dip Switches Defined!";



///////////////////////////////////////////////////////////////////////////////
//
//  Function: SSStateDipSwitch
//
//  Description:
//
//      This is the main constructor for the selection screen dip switch 
//      state object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pSelectScreen (input)
//          The selection screen the state belongs to.
//
//      pCanvas (input)
//          The canvas used by the selection screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateDipSwitch::SSStateDipSwitch(
    const KString& iName,
    SelectScreen*  pSelectScreen,
    Canvas*        pCanvas
)
:
    SSStateList     ( iName, pSelectScreen, pCanvas, FALSE, TRUE )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SSStateDipSwitch
//
//  Description:
//
//      This is the destructor for the selection screen dip switch state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateDipSwitch::~SSStateDipSwitch(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the select screen dip switch state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
SSStateDipSwitch::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "SSStateDipSwitch" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setGameInfo
//
//  Description:
//
//      This is called when a new game has been selected.  It is overridden
//      so that we can clear out our list of lines from the previous game
//      and create the new ones.
//
//  Parameters:
//
//      pGameInfo (input)
//          The new game.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateDipSwitch::setGameInfo(
    GameInfo* pGameInfo
)
{
    //  Call the base class.
    SSStateBase::setGameInfo( pGameInfo );

    //  Add the dip switch settings.
    addItems( );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: addItems
//
//  Description:
//
//      This function is responsible for adding the current key mappings
//      to the list of lines to be displayed.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateDipSwitch::addItems(
)
{
    //  Get the list of dip switches.
    ASSERT( m_pSelectScreen->getGame( ) != NULL );
    const KPtrList<DipSwitch>& dipSwitchList = 
        m_pSelectScreen->getGame( )->getDipSwitchList( );

    //  Clear the header and lines.
    clearHeader( );
    clearLines( );

    //  If there are no dip switches then place a message in the header and
    //  return.
    if( dipSwitchList.entries( ) == 0 )
    {
        m_headerList.add( new KString( NO_DIPS ) );
    }
    else
    {
        //  Loop through each dip switch.
        for( DWord dwI = 0 ; dwI < dipSwitchList.entries( ) ; dwI += 1 )
        {
            //  Now through each setting within the dip switch.
            for( 
                DWord dwJ = 0 ; 
                dwJ < dipSwitchList[ dwI ]->getNumSettings( ) ; 
                dwJ += 1 
            )
            {
                //  The current setting.
                Setting* pCurSetting = dipSwitchList[ dwI ]->getSetting( dwJ );

                //  Add the current description of the setting.
                m_lineList.add( 
                    new KString( pCurSetting->getCurDescription( ) ) 
                );
            }
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateCurItem
//
//  Description:
//
//      This function is responsible for updating the currently selected
//      dip switch setting.  This involves updating the display and saving
//      the setting to the registry.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateDipSwitch::updateCurItem(
)
{
    //  Get the registry that contains the settings.
    ASSERT( m_pSelectScreen->getGame( ) != NULL );
    Registry* pSettings = m_pSelectScreen->getGame( )->getSettingsRegistry( );

    //  Get the current setting.
    Setting* pCurSetting = curSetting( );
    ASSERT( pCurSetting != NULL );

    //  The dip switch that the setting belongs to.
    DipSwitch* pDipSwitch = pCurSetting->getDipSwitch( );


    //  Save the local control to the settings registry.
    pSettings->setValue( 
        m_pGameInfo->getGameId( ),
        pDipSwitch->getInstanceName( ),
        ( Byte* )pDipSwitch->getLocation( ),
        sizeof( Byte )
    );

    //  Update the display with the new description.
    *( m_lineList[ m_dwCurLine ] ) = pCurSetting->getCurDescription( );

}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This is used to execute the dip switch state.  The base
//      method is used to display the dip switches available for setting
//      and supports scrolling.  This function allows the user to
//      modify the current dip switch setting.
//
//  Parameters:
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      An action indicating what the selection screen should do.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStateDipSwitch::execute(
    ColourTable* pColourTable 
)
{
    //  Get the currently selected setting.
    Setting* pSetting = curSetting( );

    //  Call the base class.
    SSStateList::execute( pColourTable );


    //  Allow for the standard keys (like rotate, flip, etc.).
    SSStateBase::checkKeys( DEF_NONE );

    //  Exit?  Go back to the previous state.
    if( m_pKeyboard->switchOn( Keyboard::KEY__ESC ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ESC );

        //  Move back to the previous state.
        m_pSelectScreen->setState( SelectScreen::STATE_PREVIOUS );
    }
    else
    if( m_pKeyboard->switchOn( Keyboard::KEY__ENTER ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ENTER );

        //  Move back to the previous state.
        m_pSelectScreen->setState( SelectScreen::STATE_PREVIOUS );
    }

    //  If there are no dip switches then just return.
    if( pSetting == NULL )
    {
        return( SSS_CONTINUE );
    }

    //  Next value?
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__RIGHT ) )
    {
        //  Move the setting on to the next value.
        pSetting->next( );

        //  Update the setting.
        updateCurItem( );
    }
    else
    //  Previous value?
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__LEFT ) )
    { 
        //  Move the setting to the previous value.
        pSetting->prev( );
   
        //  Update the setting.
        updateCurItem( );
    }
    else
    {
        //  Let the base class check for scroll.
        SSStateList::checkForScroll( );
    }

    //  The game should not be started yet.
    return( SSS_CONTINUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: fillHelp
//
//  Description:
//
//      This is used to fill in the help area for this state.  The following
//      keys are allowed during this state:
//          ESC Exit
//          UP  Prev Setting
//          DN  Next Setting
//          RT  Next Value
//          LT  Prev Value
//
//  Parameters:
//
//      pBitmap (input)
//          The help bitmap.
//
//      eFont (input)
//          The font to draw with.
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateDipSwitch::fillHelp(
    Bitmap*      pBitmap,
    Canvas::Font eFont,
    ColourTable* pColourTable 
)
{
    //  Check the arguments.
    ASSERT( pBitmap       != NULL );
    ASSERT( pColourTable  != NULL );

    //  The help text.
    static char* ppstrHelpText[ ] = 
    {
        "ESC Exit",
        "UP  Prev Set",
        "DN  Next Set",
        "RT  Next Value",
        "LT  Prev Value",
        NULL
    };

    //  Draw the text.
    drawHelp( pBitmap, eFont, pColourTable, ppstrHelpText );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: curDipSwitch
//
//  Description:
//
//      This is used to return the dip switch corresponding to the currently
//      selected setting.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      A pointer to the current dip switch.
//      NULL if no dip switch is current.
//
///////////////////////////////////////////////////////////////////////////////
DipSwitch*
SSStateDipSwitch::curDipSwitch(
) const
{
    //  Get the list of dip switches.
    ASSERT( m_pSelectScreen->getGame( ) != NULL );
    const KPtrList<DipSwitch>& dipSwitchList = 
        m_pSelectScreen->getGame( )->getDipSwitchList( );

    //  The total number of settings traversed.
    DWord dwSettingTally;
 
    //  The index of the current dip switch of interest.
    DWord dwDipIndex;


    //  If there are no dip switches then return NULL.
    if( dipSwitchList.entries( ) == 0 )
    {
        return( NULL );
    }

    //  Loop until we find the correct dip switch.
    for( 
        dwSettingTally = 0, dwDipIndex = 0 ; 
        dwDipIndex < dipSwitchList.entries( ) ; 
        dwDipIndex += 1
    )
    {
        //  If the current index falls within the number of settings for
        //  this dip then we've found a match.
        if( 
            m_dwCurLine < 
            dwSettingTally + dipSwitchList[ dwDipIndex ]->getNumSettings( )
        )
        {
            break;
        }

        //  Up the tally.
        dwSettingTally += dipSwitchList[ dwDipIndex ]->getNumSettings( );
    }
    
    //  We should always have a dip switch by now.
    ASSERT( dwDipIndex < dipSwitchList.entries( ) );

    //  Return the dip switch found.
    return( dipSwitchList[ dwDipIndex ] );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: curSetting
//
//  Description:
//
//      This is used to return the dip switch setting corresponding to the 
//      currently selected in the display.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      A pointer to the current dip switch setting.
//      NULL if no dip switch setting is current.
//
///////////////////////////////////////////////////////////////////////////////
Setting*
SSStateDipSwitch::curSetting(
) const
{
    //  Get the list of dip switches.
    ASSERT( m_pSelectScreen->getGame( ) != NULL );
    const KPtrList<DipSwitch>& dipSwitchList = 
        m_pSelectScreen->getGame( )->getDipSwitchList( );

    //  The total number of settings traversed.
    DWord dwSettingTally;
 
    //  The index of the current dip switch of interest.
    DWord dwDipIndex;


    //  If there are no dip switches then return NULL.
    if( dipSwitchList.entries( ) == 0 )
    {
        return( NULL );
    }

    //  Loop until we find the correct dip switch.
    for( 
        dwSettingTally = 0, dwDipIndex = 0 ; 
        dwDipIndex < dipSwitchList.entries( ) ; 
        dwDipIndex += 1
    )
    {
        //  If the current index falls within the number of settings for
        //  this dip then we've found a match.
        if( 
            m_dwCurLine < 
            dwSettingTally + dipSwitchList[ dwDipIndex ]->getNumSettings( )
        )
        {
            break;
        }

        //  Up the tally.
        dwSettingTally += dipSwitchList[ dwDipIndex ]->getNumSettings( );
    }
    
    //  We should always have a dip switch by now.
    ASSERT( dwDipIndex < dipSwitchList.entries( ) );

    //  Return the dip switch setting found.
    return( 
        dipSwitchList[ dwDipIndex ]->getSetting( m_dwCurLine - dwSettingTally ) 
    );
}
