///////////////////////////////////////////////////////////////////////////////
//
//  File:    sssmenu.cpp
//
//  Class:   SSStateMainMenu
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the selection screen state responsible for displaying
//      and processing the main menu of options.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "sssmenu.h"
#include "sssmsg.h"
#include "select.h"
#include "replay.h"
#include "canvas.h"
#include "gameinfo.h"
#include "game.h"
#include "bitmap.h"
#include "keyb.h"
#include "record.h"
#include "play.h"
#include "joy.h"
#include "network.h"




///////////////////////////////////////////////////////////////////////////////
//  File Constants.
///////////////////////////////////////////////////////////////////////////////



///////////////////////////////////////////////////////////////////////////////
//
//  Function: SSStateMainMenu
//
//  Description:
//
//      This is the main constructor for the selection screen main menu
//      state object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pSelectScreen (input)
//          The selection screen the state belongs to.
//
//      pCanvas (input)
//          The canvas used by the selection screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateMainMenu::SSStateMainMenu(
    const KString& iName,
    SelectScreen*  pSelectScreen,
    Canvas*        pCanvas
)
:
    SSStateList         ( iName, pSelectScreen, pCanvas, FALSE, TRUE ),
    m_bMoviePlaying     ( FALSE ),
    m_bMovieRecording   ( FALSE ),
    m_bRunningServer    ( FALSE ),
    m_bRunningClient    ( FALSE ),
    m_bStartUp          ( FALSE )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SSStateMainMenu
//
//  Description:
//
//      This is the destructor for the selection screen main menu state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateMainMenu::~SSStateMainMenu(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the select screen main menu state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
SSStateMainMenu::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "SSStateMainMenu" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setState
//
//  Description:
//
//      This is called when the state has just been selected as the current
//      selection screen state.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateMainMenu::setState(
)
{
    //  Call the base class.
    SSStateBase::setState( );

    //  What's the state of the current game?
    if( m_pSelectScreen->getGame( ) == NULL )
    {
        m_pSelectScreen->setGame( );
    }
    m_bStartUp = m_pSelectScreen->getGame( )->startUp( );

    //  Add the menu items to the list of items displayed.
    addItems( );
}



///////////////////////////////////////////////////////////////////////////////
//  The following macro is used to add an item to the list.
///////////////////////////////////////////////////////////////////////////////
#define ADD_LINE( STR, CHOICE )                                               \
{                                                                             \
    m_lineList.add( new KString( STR ) );                                     \
    m_aeChoices[ dwItem ] = CHOICE;                                           \
    dwItem += 1;                                                              \
}
///////////////////////////////////////////////////////////////////////////////
//
//  Function: addItems
//
//  Description:
//
//      This function is responsible for adding the menu items appropriate
//      to the state of the current game to the items displayed in the list.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateMainMenu::addItems(
)
{
    //  There should be a game by now.
    ASSERT( m_pSelectScreen->getGame( ) != NULL );

    //  The current item being added.
    DWord dwItem = 0;
 
    //  Clear the header and lines.
    clearHeader( );
    clearLines( );

    //  If there is no longer any meddler than the movie must have ended.
    if( Replay::s_instance( ).getMeddler( ) == NULL )
    {
        m_bMoviePlaying   = FALSE;
        m_bMovieRecording = FALSE;
    }

    //  If the network is not open then the connection must be broken.
    if( !( Replay::s_instance( ).getNetwork( )->isOpen( ) ) )
    {
        m_bRunningServer = FALSE;
        m_bRunningClient = FALSE;
    }

    //  Can always return to the game.
    ADD_LINE( "Return to Game", CHOICE_RETURN );
    ADD_LINE( "", CHOICE_NONE );

    //  If we are recording a movie then the only option is to stop.
    if( m_bMovieRecording )
    {
        ADD_LINE( "Stop Recording", CHOICE_STOP_MOVIE ); 
        ADD_LINE( "", CHOICE_NONE );
    }
    //  If we are playing a movie then the only option is to stop.
    else
    if( m_bMoviePlaying )
    {
        ADD_LINE( "Stop Playback", CHOICE_STOP_MOVIE ); 
        ADD_LINE( "", CHOICE_NONE );
    }
    //  If we are running a server the only option is to stop.
    else
    if( m_bRunningServer )
    {
        ADD_LINE( "Stop Network Game", CHOICE_STOP_NET ); 
        ADD_LINE( "", CHOICE_NONE );
    }
    //  If we are running a client then the only option is to stop.
    else
    if( m_bRunningClient )
    {
        ADD_LINE( "Leave Network Game", CHOICE_STOP_NET );
        ADD_LINE( "", CHOICE_NONE );
    }
    else
    //  We only add the majority of items if no files are missing and there 
    //  is no startup of the game remaining to be done.
    if( 
        m_pGameInfo->requiredFilesAvailable( ) &&
        !m_pSelectScreen->getGame( )->startUp( ) 
    )
    {
        ADD_LINE( "Key Configuration",        CHOICE_KEY__CONFIG );
        if( Replay::s_instance( ).getJoystick( )->isEnabled( ) )
        {
            ADD_LINE( "Joystick Configuration",   CHOICE_JOY_CONFIG );
        }
        ADD_LINE( "Dip Switch Settings",      CHOICE_DIP_SWITCH ); 
        ADD_LINE( "",                         CHOICE_NONE );
        ADD_LINE( "Save Game",                CHOICE_SAVE_GAME );
        ADD_LINE( "Load Game",                CHOICE_LOAD_GAME );
        ADD_LINE( "",                         CHOICE_NONE );
        ADD_LINE( "Record Movie",             CHOICE_RECORD_MOVIE );
        ADD_LINE( "Play Movie",               CHOICE_PLAY_MOVIE );
        ADD_LINE( "",                         CHOICE_NONE );
        if( Replay::s_instance( ).getNetwork( )->isEnabled( ) )
        {
            ADD_LINE( "Start Network Game",       CHOICE_START_SERVER );
            ADD_LINE( "Join Network Game",        CHOICE_START_CLIENT );
        }
        ADD_LINE( "",                         CHOICE_NONE );
    }
    else
    {
        //  We don't have to have a current game to attempt to join a network
        //  game.
        if( Replay::s_instance( ).getNetwork( )->isEnabled( ) )
        {
            ADD_LINE( "Join Network Game",   CHOICE_START_CLIENT );
        }
        ADD_LINE( "",                    CHOICE_NONE );
    }

    //  The following always appears.
    ADD_LINE( "Exit Replay+", CHOICE_EXIT );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This is used to execute the main menu state.  The base
//      method is used to display the main menu options available for 
//      selection, and supports scrolling.  This function allows the user to
//      select an option.
//
//  Parameters:
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      An action indicating what the selection screen should do.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStateMainMenu::execute(
    ColourTable* pColourTable 
)
{
    //  A game should be specified by now.
    ASSERT( m_pSelectScreen->getGame( ) != NULL );

    //  Call the base class.
    SSStateList::execute( pColourTable );


    //  Allow for the standard keys (like rotate, flip, etc.).
    SSStateBase::checkKeys( DEF_NONE );


    //  Exit?  
    if( m_pKeyboard->switchOn( Keyboard::KEY__ESC ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ESC );

        //  Move to an appropriate state.
        if( m_pGameInfo->requiredFilesAvailable( ) )
        {
            //  If there is no further startup to be performed then move to
            //  the paused state, otherwise return to the previous state.
            if( !m_pSelectScreen->getGame( )->startUp( ) )
            {
                m_pSelectScreen->setState( SelectScreen::STATE_PAUSE );
            }
            else
            {
                m_pSelectScreen->setState( SelectScreen::STATE_PREVIOUS );
            }
        }
        else
        {
            m_pSelectScreen->setState( SelectScreen::STATE_MISSING_FILES );
        }
    }
    else
    //  Made a selection?
    if( m_pKeyboard->switchOn( Keyboard::KEY__ENTER ) )
    { 
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ENTER );

        return( processSelection( ) );
    }
    else
    {
        //  Let the base class check for scroll.
        SSStateList::checkForScroll( );

        //  Check on the game state.  If it has changed then we need to 
        //  update the items in the list of choices.
        Byte bStartUp = m_pSelectScreen->getGame( )->startUp( );
        if( bStartUp != m_bStartUp )
        {
            m_bStartUp = bStartUp;
            addItems( );
        }
    }

    //  The game should not be started yet.
    return( SSS_CONTINUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: processSelection
//
//  Description:
//
//      This is used to process the selection the user has made from the
//      main menu.  It then returns back instructions on what action the
//      selection screen should take.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      An action indicating what the selection screen should do.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStateMainMenu::processSelection(
)
{
    //  What was the selection?
    switch( m_aeChoices[ m_dwCurLine ] )
    {
        //  A Separator should never be chosen.
        case CHOICE_NONE:
        {
            fatalError( "Separator chosen in selection screen." );
            break;
        }

        //  Return to the game.
        case CHOICE_RETURN:
        {
            //  Move to an appropriate state.
            if( m_pGameInfo->requiredFilesAvailable( ) )
            {
                //  If there is no further startup to be performed then move to
                //  the paused state, otherwise return to the previous state.
                if( !m_pSelectScreen->getGame( )->startUp( ) )
                {
                    m_pSelectScreen->setState( SelectScreen::STATE_PAUSE );
                }
                else
                {
                    m_pSelectScreen->setState( SelectScreen::STATE_PREVIOUS );
                }
            }
            else
            {
                m_pSelectScreen->setState( SelectScreen::STATE_MISSING_FILES );
            }
            break;
        }

        //  Key Configuration.
        case CHOICE_KEY__CONFIG:
        {
            m_pSelectScreen->setState( SelectScreen::STATE_KEY_CONFIG );
            break;
        }

        //  Joystick Configuration.
        case CHOICE_JOY_CONFIG:
        {
            m_pSelectScreen->setState( SelectScreen::STATE_JOY_CONFIG );
            break;
        }

        //  Dip Switch Settings.
        case CHOICE_DIP_SWITCH:
        {
            m_pSelectScreen->setState( SelectScreen::STATE_DIP_SWITCH );
            break;
        }

        //  Save the current Game state.
        case CHOICE_SAVE_GAME:
        {
            ASSERT( m_pSelectScreen->getGame( ) != NULL );
            SSStateMessage::s_setMessage(
                m_pSelectScreen->getGame( )->save( ) ? 
                    SSStateMessage::MSG_SAVE_BAD : 
                    SSStateMessage::MSG_SAVE_GOOD
            );
            m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );
            break;
        }

        //  Load a Game state.
        case CHOICE_LOAD_GAME:
        {
            ASSERT( m_pSelectScreen->getGame( ) != NULL );
            SSStateMessage::s_setMessage(
                m_pSelectScreen->getGame( )->load( ) ? 
                    SSStateMessage::MSG_LOAD_BAD : 
                    SSStateMessage::MSG_LOAD_GOOD
            );
            m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );

            //  Crank the game forward one frame so that the
            //  screen is updated with the saved state.
            m_pSelectScreen->getGame( )->run( );

            break;
        }

        //  Begin recording of a game movie.
        case CHOICE_RECORD_MOVIE:
        {
            //  Add a movie recording meddler to Replay.
            ASSERT( m_pSelectScreen->getGame( ) != NULL );
            Meddler* pMeddler = MovieRecordMeddler::s_build( 
                "Record", m_pSelectScreen->getGame( ) 
            );
            Replay::s_instance( ).newMeddler( pMeddler );

            //  Tell the user that the recording has started as long as
            //  there are no errors.
            if( pMeddler->complete( ) )
            {
                SSStateMessage::s_setMessage( SSStateMessage::MSG_RECORD_BAD );
            }
            else
            {
                SSStateMessage::s_setMessage( SSStateMessage::MSG_RECORD_GOOD );
                m_bMovieRecording = TRUE;
            }

            m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );
            break;
        }

        //  Begin playback of a game movie.
        case CHOICE_PLAY_MOVIE:
        {
            //  Add a movie playing meddler to Replay.
            ASSERT( m_pSelectScreen->getGame( ) != NULL );
            Meddler* pMeddler = MoviePlayMeddler::s_build( 
                "Play", m_pSelectScreen->getGame( )
            );
            Replay::s_instance( ).newMeddler( pMeddler );

            //  Tell the user that the playback has started as long as
            //  there are no errors.
            if( pMeddler->complete( ) )
            {
                SSStateMessage::s_setMessage( SSStateMessage::MSG_PLAY_BAD );
            }
            else
            {
                SSStateMessage::s_setMessage( SSStateMessage::MSG_PLAY_GOOD );
                m_bMoviePlaying = TRUE;
            }

            m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );
            break;
        }

        //  Stop playback/recording of a game movie.
        case CHOICE_STOP_MOVIE:
        {
            //  Can't stop if a movie wasn't going.
            ASSERT( m_bMoviePlaying || m_bMovieRecording );

            //  Stop the current meddler.
            Replay::s_instance( ).deleteMeddler( );

            //  Tell the user that the movie has stopped.
            SSStateMessage::s_setMessage(
                m_bMoviePlaying ?
                    SSStateMessage::MSG_PLAY_STOP : 
                    SSStateMessage::MSG_RECORD_STOP
            );
            m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );

            //  Note that we are no longer playing/recording a movie.
            m_bMoviePlaying   = FALSE;
            m_bMovieRecording = FALSE;

            break;
        }

        //  Start a server.
        case CHOICE_START_SERVER:
        {
            //  Open the network as a server.
            if( 
                Replay::s_instance( ).getNetwork( )->openNetwork( 
                    Network::SERVER 
                )
            )
            {
                //  Note that we're now running a server.
                m_bRunningServer = TRUE;

                //  Synchronize with clients.
                m_pSelectScreen->setState( SelectScreen::STATE_SERVER );
            }
            else
            {
                //  There must already be a server.
                SSStateMessage::s_setMessage( SSStateMessage::MSG_NET_INUSE );
                m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );
            }

            break;
        }

        //  Connect to the server as a client.
        case CHOICE_START_CLIENT:
        {
            //  Open the network as a client.
            if( 
                Replay::s_instance( ).getNetwork( )->openNetwork( 
                    Network::CLIENT 
                ) &&
                Replay::s_instance( ).getNetwork( )->connectToServer( )
            )
            {
                //  Note that we're now a client.
                m_bRunningClient = TRUE;
    
                //  Synchronize with the server.
                m_pSelectScreen->setState( SelectScreen::STATE_CLIENT );
            }
            else
            {
                //  Could not join network game.
                SSStateMessage::s_setMessage( SSStateMessage::MSG_NET_NOSERV );
                m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );
            }

            break;
        }

        //  Stop the network.
        case CHOICE_STOP_NET:
        {
            ASSERT( m_bRunningServer || m_bRunningClient );

            //  Terminate the meddler.  This will close down the network.
            Replay::s_instance( ).deleteMeddler( );

            //  Inform the user.
            if( m_bRunningServer )
            {
                //  Note that we are no longer running a server.
                m_bRunningServer = FALSE;

                //  Inform the user of the fact.
                SSStateMessage::s_setMessage( SSStateMessage::MSG_NET_STOPPED );
            }
            else
            if( m_bRunningClient )
            {
                //  Note that we are no longer running a server.
                m_bRunningClient = FALSE;

                //  Inform the user of the fact.
                SSStateMessage::s_setMessage( SSStateMessage::MSG_NET_LEFT );
            }
 
            m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );
     
            break;
        }

        //  Exit Replay.
        case CHOICE_EXIT:
        {
            //  If there is a current game then shut it down.
            if( m_pSelectScreen->getGame( ) != NULL )
            {
                m_pSelectScreen->getGame( )->shutDown( );
            }
            return( SSS_QUIT );
        }

        //  Unknown.
        default:
        {
            fatalError( "Main Menu out of whack." );
            break;
        }
    }

    //  Just continue on.
    return( SSS_CONTINUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: fillHelp
//
//  Description:
//
//      This is used to fill in the help area for this state.  The following
//      keys are allowed during this state:
//          ESC Exit Menu
//          UP  Prev Option
//          DN  Next Option
//          RET Select    
//
//  Parameters:
//
//      pBitmap (input)
//          The help bitmap.
//
//      eFont (input)
//          The font to draw with.
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateMainMenu::fillHelp(
    Bitmap*      pBitmap,
    Canvas::Font eFont,
    ColourTable* pColourTable 
)
{
    //  Check the arguments.
    ASSERT( pBitmap       != NULL );
    ASSERT( pColourTable  != NULL );

    //  The help text.
    static char* ppstrHelpText[ ] = 
    {
        "ESC Exit Menu",
        "UP  Prev Option",
        "DN  Next Option",
        "RET Select",
        NULL
    };

    //  Draw the text.
    drawHelp( pBitmap, eFont, pColourTable, ppstrHelpText );
}
