///////////////////////////////////////////////////////////////////////////////
//
//  File:    sssserv.cpp
//
//  Class:   SSStateServer
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the selection screen state responsible for waiting
//      for network clients to connect when Replay is acting as a server.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <stdlib.h>

//  Application Headers.
#include "sssserv.h"
#include "sssmsg.h"
#include "select.h"
#include "replay.h"
#include "canvas.h"
#include "bitmap.h"
#include "keyb.h"
#include "network.h"
#include "gameinfo.h"
#include "game.h"
#include "dip.h"
#include "server.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: SSStateServer
//
//  Description:
//
//      This is the main constructor for the selection screen server
//      state object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pSelectScreen (input)
//          The selection screen the state belongs to.
//
//      pCanvas (input)
//          The canvas used by the selection screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateServer::SSStateServer(
    const KString& iName,
    SelectScreen*  pSelectScreen,
    Canvas*        pCanvas
)
:
    SSStateList      ( iName, pSelectScreen, pCanvas, TRUE ),
    m_pNetwork       ( Replay::s_instance( ).getNetwork( ) ),
    m_eSynchro       ( SYNCHRO_NONE ),
    m_dwClient       ( CLIENT_NONE ),
    m_bAbort         ( FALSE ),
    m_pMsg           ( NULL )
{
    ASSERT( m_pNetwork != NULL );

    //  Create the network message.
    m_pMsg = new Network::Msg;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SSStateServer
//
//  Description:
//
//      This is the destructor for the selection screen server state
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateServer::~SSStateServer(
)
{
    //  Dispose of the network message.
    delete m_pMsg;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the select screen server state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
SSStateServer::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "SSStateServer" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setState
//
//  Description:
//
//      This is called when this state has just become the current state.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateServer::setState(
)
{
    //  Call the base class.
    SSStateBase::setState( );

    //  If the network game was aborted then go back to the main menu.
    if( m_bAbort )
    {
        m_bAbort = FALSE;
        m_pSelectScreen->setState( SelectScreen::STATE_MAIN_MENU );
        return;
    }

    //  Update the Header.
    updateHeader( );

    //  Clear the list.
    clearLines( );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateHeader
//
//  Description:
//
//      This is called to update the header lines in the list.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateServer::updateHeader(
)
{
    //  Clear the header.
    clearHeader( );


    //  Create the header lines. 
    m_headerList.add( new KString( "Current Players:" ) );
    m_headerList.add( new KString( "  Server    - Ready" ) );
    for( DWord dwI = 0 ; dwI < m_pNetwork->getNumConnections( ) ; dwI += 1 )
    {
        if( ( m_eSynchro != SYNCHRO_NONE ) && ( dwI == m_dwClient ) )
        {
            sprintf( m_strLine, "  Client %02ld - Synchronizing", dwI );
        }
        else
        {
            sprintf( m_strLine, "  Client %02ld - Ready ", dwI );
        }
        m_headerList.add( new KString( m_strLine ) );
    }
    m_headerList.add( new KString( "--------------------------------" ) );

    //  Ask the selection screen to update itself so that our help text
    //  is in sync.
    m_pSelectScreen->update( );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This is used to execute the server state.  The base method
//      does the display.  This member function will be responsible for
//      polling for clients and synchronizing with them when they connect.
//
//  Parameters:
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      An action indicating what the selection screen should do.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStateServer::execute(
    ColourTable* pColourTable 
)
{
    //  Call the base class.
    SSStateList::execute( pColourTable );

    //  Allow the user to scroll the lines.
    SSStateList::checkForScroll( );

    //  Allow for the standard keys (like rotate, flip, etc.).
    SSStateBase::checkKeys( DEF_NONE );

    //  Exit?  Shut down the network and inform the user.
    if( m_pKeyboard->switchOn( Keyboard::KEY__ESC ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ESC );

        //  Kill the network.
        m_pNetwork->closeNetwork( );

        //  Note that we aborted so that we don't pass this way again.
        m_bAbort = TRUE;

        //  Inform the user of the fact.
        SSStateMessage::s_setMessage( SSStateMessage::MSG_NET_STOPPED );
        m_pSelectScreen->setState( SelectScreen::STATE_MESSAGE );
    }
    else
    //  Start?
    if( m_pKeyboard->switchOn( Keyboard::KEY__ENTER ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ENTER );

        //  To start the game, we must have connections and we can't
        //  be in the middle of synchronization.
        if( 
            ( m_pNetwork->getNumConnections( ) > 0 ) &&
            ( m_eSynchro == SYNCHRO_NONE )
        )
        {
            //  Send the start message to each client.
            sendGo( );

            //  Add a net game server meddler to Replay.
            ASSERT( m_pSelectScreen->getGame( ) != NULL );
            Replay::s_instance( ).newMeddler( 
                NetGameServerMeddler::s_build( 
                    "Server", m_pSelectScreen->getGame( ) 
                )
            );
    
            //  Run the net game.
            return( SSS_RUN );
        }
    }
    else
    //  Continue processing connections.
    if( m_eSynchro == SYNCHRO_NONE )
    {
        //  If a client was accepted then start the synchronization.
        if( m_pNetwork->acceptClient( ) )
        {
            m_eSynchro = SYNCHRO_RECV_START;
            m_dwClient = m_pNetwork->getNumConnections( ) - 1;
            updateHeader( );
        }
        else
        {
            //  Ping the clients.
            pingConnections( );
        }
    }
    else
    {
        //  Move the synchronization along.
        continueSynchro( );
    }

    return( SSS_CONTINUE );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: continueSynchro
//
//  Description:
//
//      This is called to continue the synchronization with a particular
//      client that has just connected to us.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateServer::continueSynchro(
)
{
    //  What are we currently doing?
    switch( m_eSynchro )
    {
        case SYNCHRO_NONE:                            break;
        case SYNCHRO_RECV_START: synchroRecvStart( ); break;
        case SYNCHRO_SEND_GAME:  synchroSendGame( );  break;
        case SYNCHRO_SEND_DIPS:  synchroSendDips( );  break;
        case SYNCHRO_SEND_READY: synchroSendReady( ); break;
        case SYNCHRO_RECV_READY: synchroRecvReady( ); break;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: synchroRecvStart
//
//  Description:
//
//      This is called to check for the start message from a client.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateServer::synchroRecvStart(
)
{
    //  Read a message.
    m_pNetwork->readMsg( m_dwClient, m_pMsg );

    //  If there was no message then just continue on.
    if( m_pMsg->eType == Network::MSG_NONE )
    {
        return;
    }
    //  If the client closed the connection then note this and update the 
    //  header.
    else
    if( m_pMsg->eType == Network::MSG_CLOSE )
    {
        //  Indicate that the client has left and readjust our state.
        sprintf( m_strLine, "Client %02ld left the game.", m_dwClient );
        m_lineList.add( new KString( m_strLine ) );
        m_dwClient = CLIENT_NONE;
        m_eSynchro = SYNCHRO_NONE;
        updateHeader( );
    }
    //  If we received the OK message then we are ready to start.
    else
    if( m_pMsg->eType == Network::MSG_OK )
    {
        //  Indicate that the client is ready.
        sprintf( m_strLine, "Client %02ld synchronizing.", m_dwClient );
        m_lineList.add( new KString( m_strLine ) );
        m_eSynchro = SYNCHRO_SEND_GAME;
    }
    //  Unexpected message.
    else
    {
        fatalError( "Unexpected message from client %d.", m_pMsg->eType );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: synchroSendGame
//
//  Description:
//
//      This is called to send the id of the current game to the client.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateServer::synchroSendGame(
)
{
    //  Cast the message to the network game message structure.
    Network::MsgGame* pMsgGame = ( Network::MsgGame* )m_pMsg;


    ASSERT( m_pSelectScreen->getGame( ) != NULL );

    //  Create the message.
    pMsgGame->eType = Network::MSG_GAME;
    strcpy( pMsgGame->strGameName, m_pGameInfo->getGameId( ).data( ) );

    //  Send the message.
    m_pNetwork->writeMsg( m_dwClient, m_pMsg );

    //  Move our state along.
    m_eSynchro = SYNCHRO_SEND_DIPS;

    //  Indicate that we've sent the game to the client.
    sprintf( m_strLine, "Sent game id to client %02ld", m_dwClient );
    m_lineList.add( new KString( m_strLine ) );
}
        


///////////////////////////////////////////////////////////////////////////////
//
//  Function: synchroSendDips
//
//  Description:
//
//      This is called to send the dip switch values of the current game 
//      to the client.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateServer::synchroSendDips(
)
{
    //  Cast the message to the network dip switch message.
    Network::MsgDip* pMsgDip = ( Network::MsgDip* )m_pMsg;

    //  The dip switch list.
    ASSERT( m_pSelectScreen->getGame( ) != NULL );
    const KPtrList<DipSwitch>& dipSwitchList = 
        m_pSelectScreen->getGame( )->getDipSwitchList( );


    //  Loop through each dip switch in the game.
    for( DWord dwI = 0 ; dwI < dipSwitchList.entries( ) ; dwI += 1 )
    {
        //  Create the message.
        pMsgDip->eType     = Network::MSG_DIP;
        pMsgDip->bDipIndex = ( Byte )dwI;
        pMsgDip->bValue    = dipSwitchList[ dwI ]->getValue( );
    
        //  Send the message.
        m_pNetwork->writeMsg( m_dwClient, m_pMsg );
    }

    //  Move our state along.
    m_eSynchro = SYNCHRO_SEND_READY;

    //  Indicate that we've sent the dip switches to the client.
    sprintf( m_strLine, "Sent dips to client %02ld", m_dwClient );
    m_lineList.add( new KString( m_strLine ) );
}
        


///////////////////////////////////////////////////////////////////////////////
//
//  Function: synchroSendReady
//
//  Description:
//
//      This is called to indicate to the client that we're ready to go.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateServer::synchroSendReady(
)
{
    //  Create the message.
    m_pMsg->eType = Network::MSG_OK;
 
    //  Send the message.
    m_pNetwork->writeMsg( m_dwClient, m_pMsg );

    //  Move our state along.
    m_eSynchro = SYNCHRO_RECV_READY;

    //  Indicate that we've sent the ready message to the client.
    sprintf( m_strLine, "Server OK with client %02ld", m_dwClient );
    m_lineList.add( new KString( m_strLine ) );
}
        


///////////////////////////////////////////////////////////////////////////////
//
//  Function: synchroRecvReady
//
//  Description:
//
//      This is called to check for the message from a client that indicates
//      it's ready to go.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateServer::synchroRecvReady(
)
{
    //  Read a message.
    m_pNetwork->readMsg( m_dwClient, m_pMsg );

    //  If there was no message then just continue on.
    if( m_pMsg->eType == Network::MSG_NONE )
    {
        return;
    }
    //  If the client closed the connection then note this and update the 
    //  header.
    else
    if( m_pMsg->eType == Network::MSG_CLOSE )
    {
        //  Indicate that the client has left and readjust our state.
        sprintf( m_strLine, "Client %02ld left the game.", m_dwClient );
        m_lineList.add( new KString( m_strLine ) );
        m_dwClient = CLIENT_NONE;
        m_eSynchro = SYNCHRO_NONE;
        updateHeader( );
    }
    //  If we received the OK message then we're all ready.
    else
    if( m_pMsg->eType == Network::MSG_OK )
    {
        //  Indicate that the client is ready.
        sprintf( m_strLine, "Client %02ld OK with Server.", m_dwClient );
        m_lineList.add( new KString( m_strLine ) );
        m_dwClient = CLIENT_NONE;
        m_eSynchro = SYNCHRO_NONE;
        updateHeader( );
    }
    //  Unexpected message.
    else
    {
        fatalError( "Unexpected message from client %d.", m_pMsg->eType );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: pingConnections
//
//  Description:
//
//      This is called to check that all of the existing connections are
//      still valid.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateServer::pingConnections(
)
{
    //  Run through each connection.
    for( DWord dwI = 0 ; dwI < m_pNetwork->getNumConnections( ) ; dwI += 1 )
    {
        //  Read a message.
        m_pNetwork->readMsg( dwI, m_pMsg );

        //  If the connection has gone away then update ourselves.
        if( m_pMsg->eType == Network::MSG_CLOSE )
        {
            //  Indicate that the client has left and readjust our state.
            sprintf( m_strLine, "Client %02ld left the game.", dwI );
            m_lineList.add( new KString( m_strLine ) );
            updateHeader( );
        }
        else
        {
            //  There should be no extraneous messages.
            ASSERT( m_pMsg->eType == Network::MSG_NONE );
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: sendGo
//
//  Description:
//
//      This is called to tell all clients to begin running the networked game.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateServer::sendGo(
)
{
    //  Create the message.
    m_pMsg->eType = Network::MSG_OK;
 
    //  Indicate that we're starting.
    m_lineList.add( new KString( "Here we go!" ) );

    //  Send the message to all clients.
    m_pNetwork->writeMsg( m_pMsg );
}
        


///////////////////////////////////////////////////////////////////////////////
//
//  Function: fillHelp
//
//  Description:
//
//      This is used to fill in the help area for this state.  The following
//      keys are allowed during this state:
//          ESC Abort
//          RET Start Game  (Only if there are connections and we're not
//                           currently synchronizing).
//
//  Parameters:
//
//      pBitmap (input)
//          The help bitmap.
//
//      eFont (input)
//          The font to draw with.
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateServer::fillHelp(
    Bitmap*      pBitmap,
    Canvas::Font eFont,
    ColourTable* pColourTable 
)
{
    //  Check the arguments.
    ASSERT( pBitmap       != NULL );
    ASSERT( pColourTable  != NULL );

    //  The help text.
    static char* ppstrHelpReadyText[ ] = 
    {
        "ESC Abort",
        "RET Start Game",
        NULL
    };
    static char* ppstrHelpNotReadyText[ ] = 
    {
        "ESC Abort",
        NULL
    };

    //  Draw the text.
    if( 
        ( m_pNetwork->getNumConnections( ) > 0 ) &&
        ( m_eSynchro == SYNCHRO_NONE )
    )
    {
        drawHelp( pBitmap, eFont, pColourTable, ppstrHelpReadyText );
    }
    else
    {
        drawHelp( pBitmap, eFont, pColourTable, ppstrHelpNotReadyText );
    }
}
