///////////////////////////////////////////////////////////////////////////////
//
//  File:    waveform.cpp
//
//  Class:   WaveForm
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is responsible for emulating waveform based sound found
//      in some of the Bally/Midway games such as the pacman series and
//      the dual 6809 games.
//
//      Thanks go to the Mame team for code on which this object is based.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "waveform.h"
#include "replay.h"
#include "clock.h"
#include "sound.h"
#include "sample.h"
#include "appfile.h"
#include "schedule.h"
#include "game.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: WaveForm
//
//  Description:
//
//      This is the main constructor for a WaveForm object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwSampleRate (input)
//          The sample playback rate.
//
//      dwNumVoices (input)
//          The number of voices used.
//
//      dwGain (input)
//          16 * the gain adjustment.
//
//      dwVolume (input)
//          The playback volume. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
WaveForm::WaveForm(
    const KString& iName,
    const DWord    dwSampleRate,
    const DWord    dwNumVoices,
    const DWord    dwGain,
    const DWord    dwVolume /* = 0xff */
)
:
    SoundDevice       ( iName ),
    m_pSample         ( NULL ),
    m_dwNumVoices     ( dwNumVoices ),
    m_dwGain          ( dwGain ),
    m_pnMixerBuffer   ( NULL ),
    m_pnMixerLookup   ( NULL ),
    m_pnMixerTable    ( NULL ),
    m_nUpdatePos      ( 0 ),
    m_bEnabled        ( TRUE )
{
    ASSERT( m_dwNumVoices <= 8 );

    //  Create the sample.
    m_pSample = new Sample( 
        "Wave", dwSampleRate / Replay::s_instance( ).getClock( )->getFPS( )
    );
    m_pSample->setVolume( dwVolume & 0xff ); 
    m_pSample->setFrequency( 
        m_pSample->getSize( ) * Replay::s_instance( ).getClock( )->getFPS( ) 
    );
    m_pSample->setResolution( 8 );
    m_pSample->setChannel( m_pSound->getChannel( ) );


    //  Initialize the mixer.
    initializeMixer( );


    //  Initialize the voice table.
    for( DWord dwVoice = 0 ; dwVoice < m_dwNumVoices ; dwVoice += 1 )
    {
        m_aVoiceList[ dwVoice ].m_dwFrequency = 0;
        m_aVoiceList[ dwVoice ].m_dwVolume    = 0;
        m_aVoiceList[ dwVoice ].m_pbWave      = NULL;
        m_aVoiceList[ dwVoice ].m_dwCounter   = 0;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~WaveForm
//
//  Description:
//
//      This is the destructor for a WaveForm object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
WaveForm::~WaveForm(
)
{
    //  Free up the resources.
    ASSERT( m_pSample != NULL );
    delete m_pSample;
    delete [] m_pnMixerBuffer;
    delete [] m_pnMixerTable;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the WaveForm class.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
WaveForm::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "WaveForm" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the WaveForm to a file for retrieval at a later
//      date.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the input to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
WaveForm::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  We always allow the base class to save itself first.
    SoundDevice::save( pSaveFile );

    //  Allow the sample to save itself.
    m_pSample->save( pSaveFile );

    //  Save the member data that keeps the state.
    pSaveFile->clearTotal( );
    pSaveFile->write( ( Byte* )&m_pnMixerBuffer,  m_pSample->getSize( ) * 2 );
    pSaveFile->write( ( Byte* )&m_nUpdatePos,     sizeof( m_nUpdatePos ) );
    pSaveFile->write( &m_bEnabled,                sizeof( m_bEnabled ) );

    //  Was it saved successfully?
    return( 
        pSaveFile->total( ) != 
        (
            m_pSample->getSize( ) * 2 +
            sizeof( m_nUpdatePos ) +
            sizeof( m_bEnabled )
        )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the WaveForm from a file that it was previously
//      saved to.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the input from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
WaveForm::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  We always allow the base class to load itself first.
    SoundDevice::load( pLoadFile );

    //  Allow the sample to load itself.
    m_pSample->load( pLoadFile );

    //  Load the member data that keeps the state.
    pLoadFile->clearTotal( );
    pLoadFile->read( ( Byte* )&m_pnMixerBuffer,  m_pSample->getSize( ) * 2 );
    pLoadFile->read( ( Byte* )&m_nUpdatePos,     sizeof( m_nUpdatePos ) );
    pLoadFile->read( &m_bEnabled,                sizeof( m_bEnabled ) );

    //  Was it loaded successfully?
    return( 
        pLoadFile->total( ) !=
        (
            m_pSample->getSize( ) * 2 +
            sizeof( m_nUpdatePos ) +
            sizeof( m_bEnabled )
        )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: initializeMixer
//
//  Description:
//
//      This member is called to initialize the mixer information.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
WaveForm::initializeMixer(
)
{
    //  Create the mixer buffer.
    m_pnMixerBuffer = new int16[ m_pSample->getSize( ) ];

    //  Allocate the mixer table.
    m_pnMixerTable = new int8[ 256 * m_dwNumVoices ];

    //  The mixer values are looked up from the middle of the table.
    m_pnMixerLookup = m_pnMixerTable + ( m_dwNumVoices * 128 );

    //  Fill in the table.
    DWord dwCount = m_dwNumVoices * 128;
    for( DWord dwI = 0 ; dwI < dwCount ; dwI += 1 )
    {
        DWord dwValue = dwI * m_dwGain / ( m_dwNumVoices * 16 );
        if( dwValue > 127 )
        {
            dwValue = 127;
        }
        m_pnMixerLookup[  dwI ] =  dwValue;
        m_pnMixerLookup[ -dwI ] = -dwValue;
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member is called to allow the WaveForm to update itself after
//      a complete frame.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
WaveForm::update(
)
{
    //  If no sample rate then nothing to do.
    if( Replay::s_instance( ).getSound( )->getSampleRate( ) == 0 )
    {
        return;
    }

    //  Since this is the end of the frame, update to the end of the buffer.
    updateBuffer( TRUE );

    //  Reset the sample position.
    m_nUpdatePos = 0;

    //  Play the sample.
    m_pSound->playStreamed( m_pSample );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateBuffer
//
//  Description:
//
//      This member is called to allow the WaveForm to update a portion of
//      its audio stream.
//
//  Parameters:
//
//      bToEnd (input)
//          If TRUE the complete buffer will be updated, otherwise, the buffer
//          will be updated to the point that represents the same percentage
//          complete as the current frame progress.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
WaveForm::updateBuffer(
    const Byte bToEnd
)
{
    //  A pointer to position within the buffer where we want to start
    //  to update.
    Byte* pbBuffer = m_pSample->getBuffer( ) + m_nUpdatePos;

    //  The length to update.  This is the number of bytes since the end of
    //  the last update.
    int32 nLength;
    

    //  Calculate the length to update.
    if( bToEnd )
    {
        nLength = m_pSample->getSize( ) - m_nUpdatePos;
    }
    else
    {
        //  This position is the position we want to fill the sample buffer to.
        nLength = ( int32 )
            Game::sm_pGame->getScheduler( )->getFramePercent( ) *
            m_pSample->getSize( ) - 
            m_nUpdatePos;
        if( ( DWord )( m_nUpdatePos + nLength ) > m_pSample->getSize( ) )
        {
            nLength = m_pSample->getSize( ) - m_nUpdatePos;
        }
    }
    m_nUpdatePos += nLength;


    //  If the waveform is disabled we fill the sample with silence.
    if( !m_bEnabled )
    {
        for( int32 nI = 0 ; nI < nLength ; nI += 1 )
        {
            *pbBuffer = 0x00;
            pbBuffer++;
        }
        return;
    }
            
  
    //  Clear out the mixing table.
    int16* pnMixer = m_pnMixerBuffer;
    for( int32 nI = 0 ; nI < nLength ; nI += 1 )
    {
        *pnMixer = 0;
        pnMixer++;
    }

    //  Now process each voice.
    for( DWord dwI = 0 ; dwI < m_dwNumVoices ; dwI += 1 )
    {
        //  Some convenient variables.
        DWord dwFrequency = m_aVoiceList[ dwI ].m_dwFrequency;
        DWord dwCounter   = m_aVoiceList[ dwI ].m_dwCounter;
        DWord dwVolume    = m_aVoiceList[ dwI ].m_dwVolume;
        Byte* pbWave      = m_aVoiceList[ dwI ].m_pbWave;

        if( dwFrequency && dwVolume )
        {
            pnMixer = m_pnMixerBuffer;
            for( int32 nI = 0 ; nI < nLength ; nI += 1 )
            {
                dwCounter += dwFrequency;
                *pnMixer += ( 
                    ( pbWave[ ( dwCounter >> 15 ) & 0x1f ] & 0x0f ) - 8
                ) * dwVolume;
                pnMixer += 1;
            }

            m_aVoiceList[ dwI ].m_dwCounter = dwCounter;
        }
    }

    //  Now update the sample.
    pnMixer = m_pnMixerBuffer;
    for( int32 nI = 0 ; nI < nLength ; nI += 1 )
    {
        *pbBuffer = ( Byte )m_pnMixerLookup[ *pnMixer ];
        pbBuffer += 1;
        pnMixer  += 1;
    }
}
