///////////////////////////////////////////////////////////////////////////////
//
//  File:    kplist.cpp
//
//  Class:   KPtrList
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is a lightweight templated pointer list.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>
#include <assert.h>

//  Application Headers.
#include "kplist.h"


///////////////////////////////////////////////////////////////////////////////
//
//  Function: KPtrList
//
//  Description:
//
//      This constructor creates an empty KPtrList.
//
//  Parameters:
//
//      nIncrement (input)
//          The number of elements to grow the list by when more space is
//          needed.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
template <class TC>  
KPtrList<TC>::KPtrList(
    int nIncrement /*=20*/
)
:
    m_nIncrement  ( nIncrement ),
    m_nSize       ( 0 ),
    m_nNumItems   ( 0 ),
    m_ppList      ( NULL )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~KPtrList
//
//  Description:
//
//      This is the destructor for the KPtrList object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
template <class TC> 
KPtrList<TC>::~KPtrList(
)
{
    //  Free the list.
    if( m_nSize > 0 )
    {
        delete [] m_ppList;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: add
//
//  Description:
//
//      This function adds an item to the list.
//
//  Parameters:
//
//      pItem (input)
//          The item to add to the list.
//
//  Returns:
//
//      A pointer to the item added.
//
///////////////////////////////////////////////////////////////////////////////
template <class TC>
TC*
KPtrList<TC>::add(
    TC* pItem
)
{
    //  If there is no room then grow the list.
    if( m_nNumItems >= m_nSize )
    {
        growList( );
    }

    //  Add the item to the next open slot.
    m_ppList[ m_nNumItems ] = pItem;
    m_nNumItems += 1;

    //  Return the item just added.
    return( pItem );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: remove
//
//  Description:
//
//      This function removes an item from the list.
//
//  Parameters:
//
//      pItem (input)
//          The item to remove from the list.
//
//  Returns:
//
//      A pointer to the item removed.
//
///////////////////////////////////////////////////////////////////////////////
template <class TC>
TC*
KPtrList<TC>::remove(
    TC* pItem
)
{
    assert( pItem != NULL );

    //  Find the item to remove.
    for( unsigned int nI = 0 ; nI < m_nNumItems ; nI += 1 )
    {
        //  Is this the item to delete?
        if( m_ppList[ nI ] == pItem )
        {
            //  Move all following items up one spot.
            for( nI += 1 ; nI < m_nNumItems ; nI += 1 )
            {
                m_ppList[ nI - 1 ] = m_ppList[ nI ];
            }

            //  There is now one less item in the list.
            m_nNumItems -= 1;

            //  Return the item back, indicating that it was removed.
            return( pItem );
        } 
    }
 
    //  The item was not found.
    return( NULL );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: clear
//
//  Description:
//
//      This function clears out the list.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
template <class TC>
void
KPtrList<TC>::clear(
)
{
    //  Free the list if there were any items in it.
    if( m_nSize > 0 )
    {
        m_nNumItems = 0;
        m_nSize     = 0;
        delete []   m_ppList;
        m_ppList    = NULL;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: shrink
//
//  Description:
//
//      This function shrinks the list to the specified size.
//
//  Parameters:
//
//      nEntries (input)
//          The number of entries to leave in the list.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
template <class TC>
void
KPtrList<TC>::shrink(
    unsigned int nEntries
)
{
    //  If the list is to be shrunk to 0 then use clear() otherwise
    //  just reduce the number of entries.
    if( nEntries == 0 )
    {
        clear( );
    }
    else
    {
        if( nEntries <= m_nNumItems )
        {
            m_nNumItems = nEntries;
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: growList
//
//  Description:
//
//      The function is used to increase the space for the list.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
template <class TC>
void
KPtrList<TC>::growList(
)
{
    //  A pointer to the old list.
    TC** ppOldList;

    //  If there is an old list then we need to realloc.
    if( m_ppList != NULL )
    {
        ppOldList = m_ppList;
        m_ppList = new TC*[ m_nSize + m_nIncrement ];
        memcpy( 
            ( void* )m_ppList, ( void* )ppOldList, m_nSize * sizeof( TC* )
        );

        //  Delete the old list.
        delete [] ppOldList;
    }
    else
    {
        m_ppList = new TC*[ m_nSize + m_nIncrement ];
    }
       
    //  Increment the needed size.
    m_nSize += m_nIncrement;
}



//////////////////////
//  Templated classes.
//////////////////////
class KString;
template class KPtrList<KString>;
class Input;
template class KPtrList<Input>;
class CtrlMap;
template class KPtrList<CtrlMap>;
class DipSwitch;
template class KPtrList<DipSwitch>;
class Setting;
template class KPtrList<Setting>;
class AddrSpace;
template class KPtrList<AddrSpace>;
class CPU;
template class KPtrList<CPU>;
class ReadHandler;
template class KPtrList<ReadHandler>;
class WriteHandler;
template class KPtrList<WriteHandler>;
class Bitmap;
template class KPtrList<Bitmap>;
class GfxSet;
template class KPtrList<GfxSet>;
class Palette;
template class KPtrList<Palette>;
class Buffer;
template class KPtrList<Buffer>;
class ByteModifier;
template class KPtrList<ByteModifier>;
#include "hiscore.h"
template class KPtrList<HiScore::ScoreRange>;
class ColourTable;
template class KPtrList<ColourTable>;
class Clipping;
template class KPtrList<Clipping>;
class Sample;
template class KPtrList<Sample>;
class SoundDevice;
template class KPtrList<SoundDevice>;
#include "movie.h"
template class KPtrList<MovieMeddler::Frame>;
class Switch;
template class KPtrList<Switch>;
class BreakPoint;
template class KPtrList<BreakPoint>;
class Parameter;
template class KPtrList<Parameter>;
class Slice;
template class KPtrList<Slice>;
class GameInfo;
template class KPtrList<GameInfo>;
class GameFinder;
template class KPtrList<GameFinder>;
#ifdef DOS
class GraphicsMode;
template class KPtrList<GraphicsMode>;
#endif
