///////////////////////////////////////////////////////////////////////////////
//
//  File:    matrix.cpp
//
//  Class:   Matrix
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The Matrix class is a simple, lightweight class 
//      which is used for manipulating an mxn matrix represented 
//      by a one dimensional array in row form.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>
#include <assert.h>

//  Application Headers.
#include "matrix.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Matrix
//
//  Description:
//
//      This is the main constructor for a matrix object.
//
//  Parameters:
//
//      pBuffer (input)
//          A pointer to the array containing the entries that make up the
//          matrix.
//
//      nWidth (input)
//          The width of the matrix.
//
//      nHeight (input)
//          The height of the matrix.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
template <class TC>  
Matrix<TC>::Matrix(
    TC*       pBuffer  /* = NULL */,
    const int nWidth   /* = 0    */,
    const int nHeight  /* = 0    */
)
:
    m_pBuffer    ( pBuffer ),
    m_nWidth     ( nWidth ),
    m_nHeight    ( nHeight ),
    m_nSize      ( nWidth * nHeight )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Matrix
//
//  Description:
//
//      This is the destructor for a matrix object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
template <class TC> 
Matrix<TC>::~Matrix(
)
{
    //  Nothing to do.
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: buffer
//
//  Description:
//
//      This function sets up the matrix to prepare for operations.
//
//  Parameters:
//
//      pBuffer (input/output)
//          The buffer containing the elements of the matrix in one-dimensional
//          row-order fashion.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
template <class TC>
void
Matrix<TC>::setup(
    TC*       pBuffer,
    const int nWidth,
    const int nHeight
)
{
    //  Check the parameters.
    assert( pBuffer  != NULL );
    assert( nWidth  >  0 );
    assert( nHeight >  0 );

    //  Hold on to the info.
    m_pBuffer  = pBuffer;
    m_nWidth   = nWidth;
    m_nHeight  = nHeight;
    m_nSize    = nWidth * nHeight;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: transpose
//
//  Description:
//
//      This function transposes the matrix almost in-place.  It uses a 1 byte
//      buffer to perform the transformation.  The original algorithm takes
//      a variable sized work buffer but 1 byte works just swell (if a little
//      slower).
//
//      The original algorithm is from algorithm 380 of the "Collected
//      Algorithms from ACM".  It appeared in Comm. ACM, Vol. 13, No. 05,
//      Page 324.
//
//  DISCLAIMER:
//
//      This function was adapted from a fortran program found on the net.
//      It has been run through f2c to convert to 'C' code with many a goto
//      and then cleaned up somewhat to get rid of the gotos.  It is very
//      non-intuitive due to the process taken to get the algorithm into
//      'C' and I do *not* claim to understand how it works.  However
//      it seems to work and that's all that matters at the moment.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
template <class TC>
void
Matrix<TC>::transpose(
)
{
    //  Variables used for the transformation.  Your guess is as good as mine
    //  for what they all do.
    TC  item;
    int nI;
    int nI1;
    int nI2;
    int nIA;
    int nIB;
    int nJ;
    int nK;
    int nKMI;
    int nCount;
    int nWidth2;
    int nMax;
    int nMove;

    
    //  If the matrix has a width or height of 1 then it's already transposed.
    if( ( m_nWidth == 1 ) || ( m_nHeight == 1 ) )  
    {
        return;
    }

    //  If the matrix is square then we use a simplified algorithm that 
    //  simply swaps diagonals, otherwise we use the complex algorithm.
    if( m_nWidth == m_nHeight ) 
    {
        //  We do a nice easy swap based on the lower triangle.
        for( nI = 0 ; nI < m_nHeight ; nI += 1 )  
        {
            //  The start of the current row of interest.
            int nRowStart = nI * m_nWidth;

            //  Go halfway across the row, swapping as we go.
            for( nJ = 0 ; nJ < nI ; nJ += 1 )
            {
                //  Swapping locations.
                int nSwap1 = nRowStart + nJ;
                int nSwap2 = nJ * m_nWidth + nI;

                //  Swap the items.
                item                = m_pBuffer[ nSwap1 ];
                m_pBuffer[ nSwap1 ] = m_pBuffer[ nSwap2 ];
                m_pBuffer[ nSwap2 ] = item;
            }
        }

        //  All done.
        return;
    }

    //  OK, no simple square matrix to transpose, therefore we have to use
    //  the complex method.
    nCount  = 2;
    nWidth2 = m_nWidth - 2;
    nMove   = 0;

    if( nWidth2 > 0 )
    {
        for( nIA = 1 ; nIA <= nWidth2 ; nIA += 1 ) 
        {
            nIB = nIA * ( m_nHeight - 1 ) / ( m_nWidth - 1 );
            if( nIA * ( m_nHeight - 1 ) == nIB * ( m_nWidth - 1 ) ) 
            {
                nCount += 1;
                nI = nIA * m_nHeight + nIB;
                if( nI == 1 )
                {
                    nMove = 1;
                }
            }
        }
    }

    nK   = m_nSize - 1;
    nKMI = nK - 1;
    nMax = m_nSize;
    nI   = 1;

    for( ; ; )
    {
        nI1 = nI;
        for( ; ; )
        {
            item = m_pBuffer[ nI1 ];
            for( ; ; )
            {
                nI2 = m_nWidth * nI1 - nK * ( nI1 / m_nHeight );
                if( nI1 <= 1 ) 
                {
                    nMove = 2;
                }
                nCount += 1;
                if( ( nI2 == nI ) || ( nI2 >= nKMI ) ) 
                {
                    if( ( nMax == nKMI ) || ( nI2 == nI ) )
                    {
                        break;
                    }
                    nMax = nKMI;
                }
                m_pBuffer[ nI1 ] = m_pBuffer[ nI2 ];
                nI1 = nI2;
            }
            m_pBuffer[ nI1 ] = item;
            if( nCount >= m_nSize ) 
            {
                return;
            }
            if( ( nI2 == nMax ) || ( nMax == nKMI ) ) 
            {
                break;
            }
            nMax = nKMI;
            nI1  = nMax;
        }
        for( ; ; )
        {
            nMax = nK - nI;
            nI  += 1;
            nKMI = nK - nI;
            if( nI > nMax ) 
            {
                assert( 0 );
            }
            if( nI > 1 ) 
            {
                if( nI == m_nWidth * nI - nK * ( nI / m_nHeight ) ) 
                {
                    continue;
                }
                nI1 = nI;
                for( ; ; )
                {
                    nI2 = m_nWidth * nI1 - nK * ( nI1 / m_nHeight );
                    if( ( nI2 <= nI ) || ( nI2 >= nMax ) ) 
                    {
                        break;
                    }
                    nI1 = nI2;
                }
                if( nI2 != nI ) 
                {
                    continue;
                }
                break;
            }
            if( nMove < 1 ) 
            {
                break;
            }
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: flip
//
//  Description:
//
//      This function flips the matrix in-place.
//
//  Parameters:
//
//      bFlipX (input)
//          Flip in the X direction?
//
//      bFlipY (input)
//          Flip in the Y direction?
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
template <class TC>
void
Matrix<TC>::flip(
    unsigned char bFlipX,
    unsigned char bFlipY
)
{
    assert( bFlipX || bFlipY );

    //  The current item of interest in the matrix.
    TC item;

    //  Indices used for iterations.
    int nI;
    int nJ;

    //  The locations to swap.
    int nSwap1;
    int nSwap2;


    //  Make sure we have something to flip.
    assert( m_pBuffer != NULL );
    if( m_pBuffer == NULL )
    {
        return;
    }


    //  Deterimine how to flip.
    if( bFlipX )
    {
        if( bFlipY )
        {
            //  Flipping in both directions.
            for( nI = 0 ; nI < ( m_nHeight / 2 ) ; nI += 1 )
            {
                for( nJ = 0 ; nJ < m_nWidth ; nJ += 1 )
                {
                    //  Swapping locations.
                    nSwap1 = nI * m_nWidth + nJ;
                    nSwap2 = ( m_nHeight - nI ) * m_nWidth - nJ - 1;

                    //  Do the swap.
                    item                = m_pBuffer[ nSwap1 ];
                    m_pBuffer[ nSwap1 ] = m_pBuffer[ nSwap2 ];
                    m_pBuffer[ nSwap2 ] = item;
                }
            }
        }
        else
        {
            //  Flipping in the X direction.
            for( nI = 0 ; nI < m_nHeight ; nI += 1 )
            {
                for( nJ = 0 ; nJ < ( m_nWidth / 2 ) ; nJ += 1 )
                {
                    //  Swapping locations.
                    nSwap1 = nI * m_nWidth + nJ;
                    nSwap2 = ( nI + 1 ) * m_nWidth - nJ - 1;

                    //  Do the swap.
                    item                = m_pBuffer[ nSwap1 ];
                    m_pBuffer[ nSwap1 ] = m_pBuffer[ nSwap2 ];
                    m_pBuffer[ nSwap2 ] = item;
                }
            }
        }
    }
    else
    {
        //  Flipping in the Y direction.
        for( nI = 0 ; nI < ( m_nHeight / 2 ) ; nI += 1 )
        {
            for( nJ = 0 ; nJ < m_nWidth ; nJ += 1 )
            {
                //  Swapping locations.
                nSwap1 = nI * m_nWidth + nJ;
                nSwap2 = ( m_nHeight - nI - 1 ) * m_nWidth + nJ;

                //  Do the swap.
                item                = m_pBuffer[ nSwap1 ];
                m_pBuffer[ nSwap1 ] = m_pBuffer[ nSwap2 ];
                m_pBuffer[ nSwap2 ] = item;
            }
        }
    }
}



//////////////////////
//  Templated classes.
//////////////////////
template class Matrix<unsigned char>;
