///////////////////////////////////////////////////////////////////////////////
//
//  File:    bitmapx.cpp
//
//  Class:   BitmapUnixX
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a bitmap that can be displayed on the screen
//      for the Unix/X platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>
#include <string.h>
#ifdef XSHM
#include <sys/ipc.h>
#include <sys/shm.h>
#endif

//  Application Headers.
#include "reptypes.h"
#include "replay.h"
#include "bitmapx.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a Unix/X bitmap object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwWidth (input)
//          The width of the bitmap.
//
//      dwHeight (input)
//          The height of the bitmap.
//
//      bScreen (input)
//          Indicates whether or not this bitmap is a screen bitmap.
//
//      bScale (input)
//          Indicates the scale of the image buffer in relation to the
//          bitmap buffer.  A scale of 0 indicates that the same buffer
//          should be used for both.
//
//      pDisplay (input)
//          The X-Display of the Replay application.
//
//      pScreen (input)
//          The X-Screen of the Replay application.
//
//      pVisual (input)
//          The visual of the screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
BitmapUnixX*
BitmapUnixX::s_build(
    const KString&  iName,
    const DWord     dwWidth,
    const DWord     dwHeight,
    const Byte      bScreen    /* = FALSE */,
    const Byte      bScale     /* = 0     */,
    Display*        pDisplay   /* = NULL  */,
    Screen*         pScreen    /* = NULL  */,
    XVisualInfo*    pVisual    /* = NULL  */
)
{
    //  Create the new object.
    BitmapUnixX* pThis = new BitmapUnixX( 
        iName, dwWidth, dwHeight, bScreen, bScale, pDisplay, pScreen, pVisual
    );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new bitmap.
    return( pThis );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: BitmapUnixX
//
//  Description:
//
//      This constructor creates a blank bitmap of the specified dimensions.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwWidth (input)
//          The width of the bitmap.
//
//      dwHeight (input)
//          The height of the bitmap.
//
//      bScreen (input)
//          Indicates whether or not this bitmap is a screen bitmap.
//
//      pDisplay (input)
//          The X-Display of the Replay application.
//
//      pScreen (input)
//          The X-Screen of the Replay application.
//
//      pVisual (input)
//          The visual of the screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
BitmapUnixX::BitmapUnixX(
    const KString&  iName,
    const DWord     dwWidth,
    const DWord     dwHeight,
    const Byte      bScreen    /* = FALSE */,
    const Byte      bScale     /* = 0     */,
    Display*        pDisplay   /* = NULL  */,
    Screen*         pScreen    /* = NULL  */,
    XVisualInfo*    pVisual    /* = NULL  */
)
:
    Bitmap             ( iName, dwWidth, dwHeight, bScreen ),
    m_pImage           ( NULL ),
    m_pDisplay         ( pDisplay ),
    m_pScreen          ( pScreen ),
    m_pVisual          ( pVisual ),
    m_bScale           ( bScale ),
    m_pbImageBuffer    ( NULL )
{
    //  Initialization should be done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This function is used to initialize the bitmap.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
BitmapUnixX::init(
)
{
    //  If this is not a screen buffer then just use the default initialization
    //  and return.
    if( !m_bScreen )
    {
        Bitmap::init( );
        return;
    }

    //  Make sure we have a display and screen.
    CONFIRM( 
        m_pDisplay != NULL, "Cannot create screen bitmap without an X Display" 
    );
    CONFIRM( 
        m_pScreen != NULL, "Cannot create screen bitmap without an X Screen" 
    );
    CONFIRM(
        m_pVisual != NULL, "Cannot create screen bitmap without a Visual"
    );

#ifdef XSHM

    //  The bitmap is to be used for screen display so we need to create
    //  a real X bitmap using the shared memory extensions.
    //  Normally, init() would adjust the dimensions properly in case of
    //  rotation, however, since we can't call init() 'til the image has
    //  been created we have to explicitly check them here.
    m_pImage = XShmCreateImage(
        m_pDisplay,
        m_pVisual->visual,
        m_pVisual->depth,
        ZPixmap,
        0,
        &m_shmInfo,
        m_pCanvas->getTranspose( ) ? m_dwHeight : m_dwWidth,
        m_pCanvas->getTranspose( ) ? m_dwWidth : m_dwHeight
    );
    CONFIRM( m_pImage != NULL, "Could not create X screen bitmap." );

    //  Allocate a shared memory segment.
    m_shmInfo.shmid = shmget(
        IPC_PRIVATE,
        m_pImage->bytes_per_line * m_pImage->height,
        IPC_CREAT | 0777
    );
    CONFIRM( m_shmInfo.shmid >= 0, "Could not allocate shared memory." );

    //  Attach to the shared memory.
    m_shmInfo.shmaddr = ( char* )shmat( m_shmInfo.shmid, 0, 0 );
    CONFIRM( m_shmInfo.shmaddr != NULL, "Could not attach to shared memory." );

    //  Point the image to the shared memory.
    m_pImage->data = m_shmInfo.shmaddr;

    //  Point the buffer to the shared memory.
    m_pbImageBuffer = ( Byte* )m_shmInfo.shmaddr; 

    //  If the scale is 0 then we can use the image buffer for the bitmap
    //  buffer, otherwise we set the bitmap buffer to 
    //  NULL so that it will be created normally in 'init()'.
    if( m_bScale == 0 )
    {
        m_pbBuffer = m_pbImageBuffer;
    }
    else
    {
        m_pbBuffer = NULL;
    }

    //  Make sure we can write to the shared memory.
    m_shmInfo.readOnly = False;

    //  Attach the shared memory to the display.
    if( XShmAttach( m_pDisplay, &( m_shmInfo ) ) == 0 )
    {
        fatalError( "Could not attach shared memory to display." );
    }

    //  Now perform the regular initialization.  This must be done after
    //  creating the image, so that the shared memory buffer will be used.
    Bitmap::init( );

#else

    //  Perform the regular initialization.  This must be done before
    //  creating the image so that the regular bitmap buffer is available
    //  to pass to X.
    Bitmap::init( );

    //  If the image scale is 0 then the bitmap buffer and the 
    //  image buffer can be shared, otherwise, we need to allocate an image
    //  buffer of the correct size.
    if( m_bScale == 0 )
    {
        m_pbImageBuffer = m_pbBuffer;
    }
    else
    {
        m_pbImageBuffer = new Byte[ m_bScale * m_dwWidth * m_dwHeight ];
    }
     

    //  Create an image and pass it the image buffer for use.
    m_pImage = XCreateImage(
        m_pDisplay,
        m_pVisual->visual,
        m_pVisual->depth,
        ZPixmap,
        0,
        ( char* )m_pbImageBuffer,
        m_dwWidth,
        m_dwHeight,
        m_bScale,
        0
    );
    CONFIRM( m_pImage != NULL, "Could not create X image." );

#endif
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~BitmapUnixX
//
//  Description:
//
//      This is the destructor for a bitmap.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
BitmapUnixX::~BitmapUnixX(
)
{
    //  If this was a screen bitmap then destroy the associated image.
    if( m_bScreen )
    {
        //  If the image & bitmap buffer is the same (i.e. scale of 0) then
        //  assign the bitmap buffer to be NULL so that the memory is not
        //  freed twice.
        if( m_bScale == 0 )
        {
            m_pbBuffer = NULL;
        }

#ifdef XSHM

        //  Detach the shared memory segment from the display.
        XShmDetach( m_pDisplay, &m_shmInfo );

        //  Detach the shared memory.
        if( m_shmInfo.shmaddr )
        {
            shmdt( m_shmInfo.shmaddr );
        }

        //  Delete the shared memory.
        if( m_shmInfo.shmid >= 0 )
        {
            shmctl( m_shmInfo.shmid, IPC_RMID, 0 );
        }

        //  Destroy the image.
        m_pbImageBuffer = NULL;
        m_pImage->data = ( char* )NULL;
        XDestroyImage( m_pImage );

#else

        //  The buffer will be destroyed with the image so set it to NULL
        //  so memory isn't freed twice.
        m_pbImageBuffer = NULL;
        XDestroyImage( m_pImage );
#endif
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the Unix/X screen bitmap object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
BitmapUnixX::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "BitmapUnixX" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: transferImage
//
//  Description:
//
//      This function is used to transfer the bitmap buffer to the X-image
//      buffer before drawing.
//
//  Parameters:
//
//      pdwPixelList (input)
//          The list of pixel values that are equivalent to the values 
//          contained in the bitmap buffer.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
BitmapUnixX::transferBuffer(
    const DWord* pdwPixelList
)
{
    //  Transfer should only be called when we have separate image and bitmap
    //  buffers.
    ASSERT( m_pbBuffer != m_pbImageBuffer );
    ASSERT( m_bScale != 0 );

    //  The size of the buffer.
    DWord dwSize = m_dwWidth * m_dwHeight;

    //  Transfer based on scale.
    if( m_bScale == 1 )
    {
        //  Assign pointers that will be used for the traversal.
        Byte* pbSrc  = m_pbBuffer;
        Byte* pbDest = m_pbImageBuffer;

        //  Do the transfer.
        for( DWord dwI = 0 ; dwI < dwSize ; dwI += 1 )
        {
            *pbDest = pdwPixelList[ *pbSrc ];
            pbDest += 1;
            pbSrc  += 1;
        }
    }
    else
    if( m_bScale == 2 )
    {
        //  Assign pointers that will be used for the traversal.
        Byte* pbSrc  = m_pbBuffer;
        Word* pwDest = ( Word* )m_pbImageBuffer;

        //  Do the transfer.
        for( DWord dwI = 0 ; dwI < dwSize ; dwI += 1 )
        {
            *pwDest = pdwPixelList[ *pbSrc ];
            pwDest += 1;
            pbSrc  += 1;
        }
    }
    else
    {
        //  Assign pointers that will be used for the traversal.
        Byte*  pbSrc   = m_pbBuffer;
        DWord* pdwDest = ( DWord* )m_pbImageBuffer;

        //  Do the movement.
        for( DWord dwI = 0 ; dwI < dwSize ; dwI += 1 )
        {
            *pdwDest = pdwPixelList[ *pbSrc ];
            pdwDest += 1;
            pbSrc   += 1;
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: swapXY
//
//  Description:
//
//      This function is used to swap the X/Y axis in the bitmap.  This is
//      needed for rotated screen orientations.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
BitmapUnixX::swapXY(
)
{
    //  If this is a screen bitmap then we have to adjust the XImage structure
    //  to correspond to the swapped axis.
    if( m_bScreen )
    {
        //  Swap the width and height.
        m_pImage->width  = m_dwHeight; 
        m_pImage->height = m_dwWidth; 

        //  Adjust the bytes per line.
        m_pImage->bytes_per_line = 
            m_pImage->bytes_per_line * m_dwHeight / m_dwWidth;
    }

    //  Now allow the base class to perform the rest of the swapping.
    Bitmap::swapXY( );
}
