///////////////////////////////////////////////////////////////////////////////
//
//  File:    clockx.cpp
//
//  Class:   ClockUnixX
//
//  Author:  Kevin Brisley
//
//
//      The ClockUnixX class is a class encapsulating timing for the Unix/X
//      platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <sys/time.h>
#include <unistd.h>

//  Application Headers.
#include "clockx.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ClockUnixX
//
//  Description:
//
//      This is the main constructor for a Unix/X clock object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
ClockUnixX::ClockUnixX(
    const KString&  iName
)
:
    Clock    ( iName )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~ClockUnixX
//
//  Description:
//
//      This is the destructor for a Unix/X clock object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
ClockUnixX::~ClockUnixX(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the clock object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
ClockUnixX::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "ClockUnixX" );

    return( className );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: getTime
//
//  Description:
//
//      This member returns the current time in microseconds.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The current time.
//
///////////////////////////////////////////////////////////////////////////////
DWord
ClockUnixX::getTime(
) const
{
    //  The time value.
    struct timeval tv;

    //  Get the time of day.
    gettimeofday( &tv, NULL );

    //  return the time of day in microseconds.
    return( tv.tv_sec * 1000000 + tv.tv_usec );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: frameWait
//
//  Description:
//
//      This member is used to wait until the next frame is ready to be
//      drawn (as specified by specified values).
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
ClockUnixX::frameWait(
)
{
    //  The time value.
    struct timeval tv;
    
    //  The difference between the current time and the last time.
    DWord dwDiff;


    //  Get the current time and calculate the difference between this and
    //  the last time.
    gettimeofday( &tv, NULL );
    dwDiff = ( DWord )( tv.tv_sec * 1000000 + tv.tv_usec ) - m_dwLastTime;

    //  If throttling is on then wait for the next frame, otherwise just note
    //  the current clock.
    if( m_bThrottle )
    {
        //  Loop until the interval has expired.
        while( dwDiff < m_dwInterval )
        {
            gettimeofday( &tv, NULL );
            dwDiff = 
                ( DWord )( tv.tv_sec * 1000000 + tv.tv_usec ) - m_dwLastTime;
        }
    }

    //  Assign the measured FPS.  We only calculate based on a single 
    //  frame (the one just completed).  The .5 is used to ensure proper 
    //  rounding.
    m_dwMeasuredFPS = ( DWord )( ( double )1000000 / ( double )dwDiff + .5 );

    //  Record this time for next time.
    m_dwLastTime = tv.tv_sec * 1000000 + tv.tv_usec;
}
