///////////////////////////////////////////////////////////////////////////////
//
//  File:    networkx.cpp
//
//  Class:   NetworkUnixX
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The NetworkUnixX class implements a tcp/ip connection between 2
//      or more Replay processes.  
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <string.h>
#include <netdb.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>

//  Application Headers.
#include "networkx.h"
#include "config.h"
#include "replay.h"
#include "canvasx.h"



///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////
int NetworkUnixX::sm_nDefaultPort = 6444;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: NetworkUnixX
//
//  Description:
//
//      This is the main constructor for a network tcp/ip object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
NetworkUnixX::NetworkUnixX(
    const KString&  iName
)
:
    Network            ( iName ),
    m_nSockFD          ( -1 ),
    m_pConnectFDList   ( NULL ),
    m_pDisplay         ( NULL )
{
    //  The configuration object.
    Configuration& config = Configuration::s_instance( );

    //  A host and port number.
    KString host;
    int32   nPort;

    

    //  If the network is disabled then there's no point continuing.
    if( !m_bEnabled )
    {
        return;
    }

    //  Allocate space for the list of connected socket file descriptors.
    m_pConnectFDList = new int[ getMaxConnections( ) ];

    //  Fill in the server address structure.
    memset( ( void* )&m_serverAddress, 0x00, sizeof( m_serverAddress ) );

    //  First the protocol family.
    m_serverAddress.sin_family = AF_INET;

    //  Now if there was a port number specified then use that, otherwise
    //  we try and obtain the port number from the list of services.  If
    //  all else fails, we use a default port number.
    if( config.getParam( "-port", &nPort ) )
    {
        m_serverAddress.sin_port = htons( nPort );
    }
    else
    {
        //  The service entry.
        struct servent* pServiceEntry;

        //  Get the service information for Replay.
        pServiceEntry = getservbyname( "replay", "tcp" );
        if( pServiceEntry != NULL )
        {
            m_serverAddress.sin_port = pServiceEntry->s_port;
        }
        else
        {
            m_serverAddress.sin_port = htons( sm_nDefaultPort );
        }
    }

    //  If a host name was specified then use it.
    if( config.getParam( "-host", &host ) )
    {
        //  The internet address of the host.
        unsigned long nInAddr;

        //  Is the host in dotted-decimal notation?
        nInAddr = inet_addr( host.data( ) );
        if( nInAddr != ( unsigned long )-1 )
        {
            memcpy( 
                ( void* )&( m_serverAddress.sin_addr ),
                ( void* )&nInAddr,
                sizeof( nInAddr )
            );
        }
        else
        {
            //  A pointer to the host entry.
            struct hostent* pHostEntry;

            //  Get the host entry by name.
            pHostEntry = gethostbyname( host.data( ) );
            CONFIRM( 
                pHostEntry != NULL, 
                "Could not retrieve entry for %s",
                host.data( )
            );

            //  Record the address.
            memcpy(
                ( void* )&( m_serverAddress.sin_addr ),
                ( void* )( pHostEntry->h_addr ),
                pHostEntry->h_length
            );
        }
    }
    else
    {
        //  No host specified.
        m_serverAddress.sin_addr.s_addr = htonl( INADDR_ANY );
    }

    //  Obtain the display from the Unix/X canvas.  We use the display 
    //  to sync X and therefore flush out the socket events providing
    //  better network performance.
    CanvasUnixX* pCanvas = ( CanvasUnixX* )Replay::s_instance( ).getCanvas( ); 
    ASSERT( pCanvas != NULL );
    m_pDisplay = pCanvas->getDisplay( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~NetworkUnixX
//
//  Description:
//
//      This is the destructor for a network tcp/ip object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
NetworkUnixX::~NetworkUnixX(
)
{
    //  Free up the list of connected sockets.
    delete [] m_pConnectFDList;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the network tcp/ip object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
NetworkUnixX::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "NetworkUnixX" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: openNetwork
//
//  Description:
//
//      This member is called to open the tcp/ip network connection.
//
//  Parameters:
//
//      eType (input)
//          Indicates whether or not the connection should be opened as
//          client or server.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Byte
NetworkUnixX::openNetwork(
    ConnectType eType
)
{
    ASSERT( m_bEnabled );

    //  Make sure either client or server was specified.
    CONFIRM( eType != UNKNOWN, "Network must be opened as client or server" );

    //  Call the base class.
    Network::openNetwork( eType );

    //  If we're a server then we must create and bind a socket to the 
    //  server address and listen for connections.
    //  If we're a client then we don't do anything until the user attempts
    //  to connect to the server.
    if( eType == SERVER )
    {
        //  Create the socket.
        m_nSockFD = socket( AF_INET, SOCK_STREAM, 0 );
        CONFIRM( m_nSockFD >= 0, "Could not initialize socket" );

        //  Set the SO_REUSEADDR option on the socket so that the bind will
        //  succeed even if an old socket is still shutting down.
        int nFlag = 1;
        int nResult;
        nResult = setsockopt( 
            m_nSockFD, 
            SOL_SOCKET, 
            SO_REUSEADDR, 
            ( char* )&nFlag, 
            sizeof( nFlag ) 
        );
        CONFIRM( nResult >= 0, "Could not set reuse flag on socket." );

        //  Bind to the server address.
        int nBind = bind( 
            m_nSockFD, 
            ( struct sockaddr* )&m_serverAddress, 
            sizeof( m_serverAddress )
        );

        //  If the socket couldn't be bound to the address then there must
        //  be another server on the port.  Close the socket and inform the
        //  caller.
        if( nBind < 0 )
        {
            close( m_nSockFD );
            m_nSockFD = -1;
            return( FALSE );
        }

        //  Listen on the socket.
        listen( m_nSockFD, 5 );
    }

    //  The network was successfully opened.
    return( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: acceptClient
//
//  Description:
//
//      This member is called when the server is to accept client connections.
//      It performs a non-blocking accept.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE if a client connected, FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
NetworkUnixX::acceptClient(
)
{
    //  The address of the client and the file descriptor of the client 
    //  socket.
    struct sockaddr_in clientAddress;
    int                nClientSockFD;


    //  Client connections can only be accepted if we are serving as the 
    //  server.
    CONFIRM( m_eConnectType == SERVER, "Must be server to accept clients" );

    //  We can't accept more connections than allowed.
    CONFIRM( 
        getNumConnections( ) < getMaxConnections( ), 
        "Cannot accept more than %d connections.",   
        getMaxConnections( )
    );

    //  If there is no read pending on the socket then there are no clients
    //  attempting to connect.
    if( !isPending( READ, m_nSockFD ) )
    {
        return( FALSE );
    }

    //  We've made it past the select which means that a client is attempting
    //  to connect.  Accept the connection.
    int nClientLength = sizeof( clientAddress );
    nClientSockFD = accept(
        m_nSockFD, ( struct sockaddr* )&clientAddress, &nClientLength
    );
    CONFIRM( nClientSockFD >= 0, "Could not accept client connection" );
        
    //  Add the new client to the list of clients.
    m_pConnectFDList[ m_dwNumConnections ] = nClientSockFD;
    m_dwNumConnections += 1; 

    //  A client has connected.
    return( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: connectToServer
//
//  Description:
//
//      This member is called when the client is to connect to the server.
//      It performs a non-blocking connect.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE if the client connected, FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
NetworkUnixX::connectToServer(
)
{
    //  We can only connect to a server if we're a client.
    CONFIRM( m_eConnectType == CLIENT, "Must be client to connect to server." );

    //  We can only connect to a server if we haven't already connected.
    CONFIRM( m_dwNumConnections == 0, "Can't connect - already connected." );

    //  Create a socket used to communicate with the server.
    m_nSockFD = socket( AF_INET, SOCK_STREAM, 0 );
    CONFIRM( m_nSockFD >= 0, "Could not initialize socket" );

    //  Now perform the connect.  If this fails, the server mustn't be
    //  there yet.
    int nConnect = connect( 
        m_nSockFD, 
        ( struct sockaddr* )&m_serverAddress, 
        sizeof( m_serverAddress )
    );
    if( nConnect < 0 )
    {
        //  Close the socket.
        close( m_nSockFD );
        m_nSockFD = -1;

        //  Indicate that the connection has not yet been made.
        return( FALSE );
    }

    //  OK, we've connected so place the socket at the head of the connection
    //  list.
    m_pConnectFDList[ 0 ] = m_nSockFD;
    m_dwNumConnections = 1;

    //  A connection has been made.
    return( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: closeNetwork
//
//  Description:
//
//      This member is called to close the network down completely.  This
//      involves closing all client sockets (if we're a server) as well
//      as the original socket.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkUnixX::closeNetwork(
)
{
    //  If the network isn't opened then just return.
    if( !isOpen( ) )
    {
        return;
    }

    //  Close each connection. 
    while( m_dwNumConnections > 0 )
    {
        closeConnection( 0 );
    }

    //  If we are a server then we have to close our original socket.
    if( m_eConnectType == SERVER )
    {
        //  Close the original socket.
        close( m_nSockFD );
    }

    //  Call the base class.
    Network::closeNetwork( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: closeConnection
//
//  Description:
//
//      This member is called to close a specific connection within the
//      network.  It will shutdown the socket and remove the connection
//      from the list.
//
//  Parameters:
//
//      dwConnection (input)
//          Indicates which connection to close down.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkUnixX::closeConnection(
    DWord dwConnection
)
{
    //  Make sure that the specified connection exists.
    CONFIRM( 
        dwConnection < m_dwNumConnections,
        "Connection %d cannot be closed as it does not exist.",
        dwConnection
    );

    //  OK, close the connection.
    close( m_pConnectFDList[ dwConnection ] );
    
    //  We now have one less entry in the list.
    m_dwNumConnections -= 1;

    //  If this wasn't the last connection then move the connection at the
    //  end of the list into this position.
    if( dwConnection < m_dwNumConnections )
    {
        m_pConnectFDList[ dwConnection ] = 
            m_pConnectFDList[ m_dwNumConnections ];
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: readMsg
//
//  Description:
//
//      This member is called to read a message from the specified connection.
//
//  Parameters:
//
//      dwConnection (input)
//          Indicates which connection to read from.
//
//      pMsg (output)
//          The buffer to hold the message.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkUnixX::readMsg(
    const DWord dwConnection,
    Msg*        pMsg
)
{
    ASSERT( dwConnection < m_dwNumConnections );

    //  Flush out X events to ensure all socket communication is up to date
    //  and therefore performance is optimal.
    XSync( m_pDisplay, False );
 
    //  If there is a pending message then read it.  Otherwise
    //  indicate that there is no message to read.
    if( isPending( READ, m_pConnectFDList[ dwConnection ] ) )
    {
        //  Read data from the socket.
        int nRead = read( 
            m_pConnectFDList[ dwConnection ], ( Byte* )pMsg, sizeof( Msg )
        );

        //  If there was no data to read then the socket has closed.
        if( nRead <= 0 )
        {
            pMsg->eType = MSG_CLOSE;
            if( m_eConnectType == SERVER )
            {
                closeConnection( dwConnection );
            }
            else
            {
                closeNetwork( );
            }
        }
    }
    else
    {
        //  Indicate that there is no message.
        pMsg->eType = MSG_NONE;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: writeMsg
//
//  Description:
//
//      This member is called to write a message to the specified connection.
//
//  Parameters:
//
//      dwConnection (input)
//          Indicates which connection to write to.
//
//      pMsg (output)
//          The buffer that contains the message.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkUnixX::writeMsg(
    const DWord dwConnection,
    Msg*        pMsg
)
{
    ASSERT( dwConnection < m_dwNumConnections );

    //  Write data to the socket.
    int nCount = write( 
        m_pConnectFDList[ dwConnection ], ( Byte* )pMsg, sizeof( Msg )
    );
    CONFIRM( nCount >= 0, "Error writing to socket." );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: isPending
//
//  Description:
//
//      This member is called to check whether there is a pending event of
//      a particular type on the specified socket.
//
//  Parameters:
//
//      eEvent (input)
//          The type of event to check for.
//
//      nSockFD (input)
//          The file descriptor of the socket to check.
//
//  Returns:
//
//      TRUE if there is pending events, FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
NetworkUnixX::isPending(
    PendingEvent eEvent,
    int          nSockFD
)
{
    //  A zero timeout value.  This is used when we want a call to return
    //  immediately without any blocking.
    static struct timeval zeroTimeVal;

    //  A file descriptor set and pointers to file descriptor sets that
    //  will be used for the select call.
    fd_set  fdSet;
    fd_set* pReadSet = NULL;
    fd_set* pWriteSet = NULL;
    fd_set* pExceptSet = NULL;

    //  Clear the file descriptor set before use.
    FD_ZERO( &fdSet );

    //  Set the bit for our socket descriptor.
    FD_SET( nSockFD, &fdSet );

    //  Assign the pointers.
    switch( eEvent )
    {
        case READ:      pReadSet   = &fdSet; break;
        case WRITE:     pWriteSet  = &fdSet; break;
        case EXCEPTION: pExceptSet = &fdSet; break;
    }

    //  Now, we use a select call with an immediate timeout to poll for
    //  the specified data.
    zeroTimeVal.tv_sec  = 0;
    zeroTimeVal.tv_usec = 0;
    return(
        select( 
            nSockFD + 1, pReadSet, pWriteSet, pExceptSet, &zeroTimeVal 
        )
    );
}
