///////////////////////////////////////////////////////////////////////////////
//
//  File:    pagex.cpp
//
//  Class:   PageUnixX
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a window of the Replay application on
//      the Unix/X platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  X Headers.
#include <X11/Xlib.h>
#include <X11/Xutil.h>

//  Application Headers.
#include "pagex.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: PageUnixX
//
//  Description:
//
//      This is the main constructor for the Unix/X Page object.  
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwWidth (input)
//          The width of the page.
//
//      dwHeight (input)
//          The height of the page.
//
//      bTransposed (input)
//          Indicates that the page should be created  with 
//          the dimensions swapped.
//
//      pDisplay (input)
//          The X-display.
//
//      pScreen (input)
//          The X-screen.
//
//      colourMap (input)
//          The colour map to use in the window.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
PageUnixX::PageUnixX(
    const KString&  iName,
    const DWord     dwWidth,
    const DWord     dwHeight,
    const Byte      bTransposed,
    Display*        pDisplay,
    Screen*         pScreen,
    Colormap        colourMap
)
:
    Page         ( iName, dwWidth, dwHeight, bTransposed ),
    m_window     ( 0 ),
    m_pDisplay   ( pDisplay ),
    m_pScreen    ( pScreen )
{
    //  Opposite pixels.
    uint32 nWhite;
    uint32 nBlack;

    //  Used to set the characteristics of the page.
    XSizeHints sizeHints;
    XWMHints   wmHints;
    

    //  Get the opposite pixels.
    nWhite = WhitePixelOfScreen( m_pScreen );
    nBlack = BlackPixelOfScreen( m_pScreen );

    //  Create the window that the page represents.
    m_window = XCreateSimpleWindow(
        m_pDisplay,
        RootWindowOfScreen( m_pScreen ),
        0,
        0,
        m_dwWidth,
        m_dwHeight,
        0,
        nWhite,
        nBlack
    );
    CONFIRM( m_window != 0, "Could not create window for page." );

    //  Set the starting size and the minimum and maximum size of the window.
    //  These are all set to the same value so that the window can't be
    //  resized.
    sizeHints.flags        = PSize | PMinSize | PMaxSize;
    sizeHints.base_width   = m_dwWidth;
    sizeHints.max_width    = m_dwWidth;
    sizeHints.min_width    = m_dwWidth;
    sizeHints.base_height  = m_dwHeight;
    sizeHints.max_height   = m_dwHeight;
    sizeHints.min_height   = m_dwHeight;
    XSetWMNormalHints( m_pDisplay, m_window, &sizeHints );

    //  Allow input to the window.
    wmHints.flags = InputHint;
    wmHints.input = True;
    XSetWMHints ( m_pDisplay, m_window, &wmHints );
     
    //  Set the name of the window.
    XStoreName( m_pDisplay, m_window, iName.data() ); 
    
    //  Accept some input.
    XSelectInput(
        m_pDisplay,
        m_window,
        ExposureMask |
        KeyPressMask |
        KeyReleaseMask
    );

    //  Set the colour map for the window.
    XSetWindowColormap( m_pDisplay, m_window, colourMap );

    //  Map the window and place it on top.
    XMapRaised( m_pDisplay, m_window );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~PageUnixX
//
//  Description:
//
//      This is the main destructor for the Unix/X Page object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
PageUnixX::~PageUnixX(
)
{
    //  Destroy the window.
    XDestroyWindow( m_pDisplay, m_window );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the Unix/X Page object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
PageUnixX::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "PageUnixX" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: transpose
//
//  Description:
//
//      This is called to transpose the page.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
PageUnixX::transpose(
)
{
    //  Call the base class.
    Page::transpose( );

    //  Now resize the window to the new dimensions.
    XResizeWindow( m_pDisplay, m_window, m_dwWidth, m_dwHeight );
}
