;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  File:     bmp.asm
;;
;;  Author:   Kevin Brisley
;;
;;  Description:
;;
;;      This file contains some routines that are used to get a speed boost
;;      for bitmap related functions on an x86 platform.
;;
;;      Please note that this is my first stab at assembly language so 
;;      you'll excuse me if I'm completely out to lunch here.  I'm
;;      a long time C/C++ programmer so this is a bit of a mind set shift.
;;      I've tried to optimize the code as much as possible but I'm
;;      sure there is much more that can be done.  
;;
;;      I've read "Inner Loops" and most of the techniques used here come 
;;      from the Pentium optimization suggestions of that book.
;;
;;      If you have any improvements or suggestions, I'd be happy to hear
;;      from you.
;;
;;  Copyright (c) 1997,1998 Kevin Brisley
;;  All rights reserved.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Macros.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;  A macro for aligning to a double-word boundary.
%define align times ($$-$) & 3 nop

;  A macro for entering a function.
%macro EnterFunction 0
    push ebx
    push ecx
    push edx
    push esi
    push edi
    push ebp
%endmacro
;  For timing information, move the following lines into the above macro.
;    rdtsc
;    mov  [ StartTime ], eax

;  A macro for leaving a function
%macro ExitFunction 0
    pop  ebp
    pop  edi
    pop  esi
    pop  edx
    pop  ecx
    pop  ebx
    ret
%endmacro
;  For timing information, move the following lines into the above macro.
;    rdtsc
;    mov  [ EndTime ], eax

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Setup.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;  32 bit Protected mode.
bits 32


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Exports.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;  Our blit information structure.
global _blitInfo


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Data Segment.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

section	.data

align

;  Our global exported blit information structure.
_blitInfo:
DestStart            dd 00000000
DestWidth            dd 00000000
DestHeight           dd 00000000
SrcStart             dd 00000000
SrcWidth             dd 00000000
SrcHeight            dd 00000000
DestX                dd 00000000
DestY                dd 00000000
ClipMinX             dd 00000000
ClipMaxX             dd 00000000
ClipMinY             dd 00000000
ClipMaxY             dd 00000000
FlipX                dd 00000000
FlipY                dd 00000000
ColourTable          dd 00000000
Transparency         dd 00000000
TransparentColour    dd 00000000
;  Only used for timing information.
;StartTime            dd 00000000
;EndTime              dd 00000000

align

;  Various local variables.
numPillars     dd    0
numColumns     dd    0
destIncrement  dd    0
srcIncrement   dd    0
srcX           dd    0
srcY           dd    0

align

;  Jump table for sprite blits (CXYTT).
spriteBlitTable:
    dd  blitSprite                      ; _____
    dd  blitSpriteRaw                   ; ____T
    dd  blitSpriteMap                   ; ___T_
    dd  blitSpriteReverse               ; ___TT
    dd  blitSpriteY                     ; __Y__
    dd  blitSpriteYRaw                  ; __Y_T
    dd  blitSpriteYMap                  ; __YT_
    dd  blitSpriteYReverse              ; __YTT
    dd  blitSpriteX                     ; _X___
    dd  blitSpriteXRaw                  ; _X__T
    dd  blitSpriteXMap                  ; _X_T_
    dd  blitSpriteXReverse              ; _X_TT
    dd  blitSpriteXY                    ; _XY__
    dd  blitSpriteXYRaw                 ; _XY_T
    dd  blitSpriteXYMap                 ; _XYT_
    dd  blitSpriteXYReverse             ; _XYTT
    dd  blitSpriteColour                ; C____
    dd  blitSpriteColourRaw             ; C___T
    dd  blitSpriteColourMap             ; C__T_
    dd  blitSpriteColourReverse         ; C__TT
    dd  blitSpriteColourY               ; C_Y__
    dd  blitSpriteColourYRaw            ; C_Y_T
    dd  blitSpriteColourYMap            ; C_YT_
    dd  blitSpriteColourYReverse        ; C_YTT
    dd  blitSpriteColourX               ; CX___
    dd  blitSpriteColourXRaw            ; CX__T
    dd  blitSpriteColourXMap            ; CX_T_
    dd  blitSpriteColourXReverse        ; CX_TT
    dd  blitSpriteColourXY              ; CXY__
    dd  blitSpriteColourXYRaw           ; CXY_T
    dd  blitSpriteColourXYMap           ; CXYT_
    dd  blitSpriteColourXYReverse       ; CXYTT




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Code Segment.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

section	.text

align

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  Function: bitmapBlitChar
;;
;;  Description:
;;
;;      This is an external function which is used to blit a character onto
;;      a bitmap.  The features of blitting a character is that it blits
;;      without support for transparency, clipping or flipping.
;;
;;  Parameters:
;;
;;      None (the parameters are passed via our global blit info structure).
;;
;;  Returns:
;;
;;      Nothing.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
global	_bitmapBlitChar
global	 bitmapBlitChar
global	 bitmapBlitChar_

align

_bitmapBlitChar:
 bitmapBlitChar:
 bitmapBlitChar_:

    EnterFunction

    ;  We'll use esi for the source, edi for the destination, and edx for
    ;  the row increment.
    mov  eax, [ DestStart ]
    mov  edx, [ DestWidth ]
    mov  ebx, [ DestY ]
    mov  ecx, [ SrcStart ]
    mov  edi, [ eax + ebx * 4 ]
    mov  esi, [ ecx ]
    sub  edx, [ SrcWidth ]
    add  edi, [ DestX ]

    ;  Is there a colour table?
    cmp  dword [ ColourTable ], 0h
    je   near .woColourTable

.wColourTable:

    ;  Clear eax since we only use the low byte when transferring around the
    ;  pixel value and we don't want the upper bits mucking us up.
    xor  eax, eax

    ;  To help with Pentium dual-pipelining we start the destination buffer
    ;  back so we can preincrement instead of postincrement.
    dec  edi;

    ;  Move the colour table into a register.
    mov  ebp, [ ColourTable ]

    ;  Loop for each row of the source and each column within each row.
    mov  ebx, [ SrcHeight ]
.wRowLoop:
    mov  ecx, [ SrcWidth ]

.wColumnLoop:
    ;  Get the source byte and then translate it according to the colour table.
    ;  The loop has been unrolled 8 times to improve performance.
    mov  al, [ esi ]
    inc  edi
    inc  esi
    mov  al, [ ebp + eax * 4 ]
    mov  [ edi ], al
    mov  al, [ esi ]
    inc  edi
    inc  esi
    mov  al, [ ebp + eax * 4 ]
    mov  [ edi ], al
    mov  al, [ esi ]
    inc  edi
    inc  esi
    mov  al, [ ebp + eax * 4 ]
    mov  [ edi ], al
    mov  al, [ esi ]
    inc  edi
    inc  esi
    mov  al, [ ebp + eax * 4 ]
    mov  [ edi ], al
    mov  al, [ esi ]
    inc  edi
    inc  esi
    mov  al, [ ebp + eax * 4 ]
    mov  [ edi ], al
    mov  al, [ esi ]
    inc  edi
    inc  esi
    mov  al, [ ebp + eax * 4 ]
    mov  [ edi ], al
    mov  al, [ esi ]
    inc  edi
    inc  esi
    mov  al, [ ebp + eax * 4 ]
    mov  [ edi ], al
    mov  al, [ esi ]
    inc  edi
    inc  esi
    mov  al, [ ebp + eax * 4 ]
    mov  [ edi ], al

    ;  Decrement our column count.
    sub  ecx, 08h
    jnz  .wColumnLoop

    ;  Move to the next row.
    add  edi, edx

    ;  Decrement our row count.
    dec  ebx
    jnz  .wRowLoop 
    
    ExitFunction

.woColourTable:

    ;  Loop for each row of the source and each column within each row.
    mov  ebx, [ SrcHeight ]
.woRowLoop:
    mov  ecx, [ SrcWidth ]

.woColumnLoop:
    ;  For a speed improvement, we move double words instead of bytes.
    ;  The loop has been unrolled for 8 bytes worth to improve performance.
    mov  eax, dword [ esi ]
    add  esi, 4
    mov  dword [ edi ], eax
    add  edi, 4
    mov  eax, dword [ esi ]
    add  esi, 4
    mov  dword [ edi ], eax
    add  edi, 4

    ;  Decrement our column count.
    sub  ecx, 08h
    jnz  .woColumnLoop

    ;  Move to the next row.
    add  edi, edx

    ;  Decrement our row count.
    dec  ebx
    jnz  .woRowLoop 
    
    ExitFunction


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  Function: bitmapBlitSprite
;;
;;  Description:
;;
;;      This is an external function which is used to blit a sprite onto
;;      a bitmap.  The features of blitting a sprite is that it blits
;;      with support for transparency, clipping and flipping.
;;
;;  Parameters:
;;
;;      None (the parameters are passed via our global blit info structure).
;;
;;  Returns:
;;
;;      Nothing.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
global	_bitmapBlitSprite
global	 bitmapBlitSprite
global	 bitmapBlitSprite_

align

_bitmapBlitSprite:
 bitmapBlitSprite:
 bitmapBlitSprite_:

    EnterFunction

    ;  Here we attack the X, Y coordinates of the source and destination 
    ;  and keep a running total of the amount clipped in the X and Y
    ;  directions.
    ;  The following registers will be used:
    ;      esi - X clipped
    ;      edi - Y clipped
    xor  esi, esi
    xor  edi, edi
    mov  [ srcX ], edi
    mov  [ srcY ], edi

    ;  Clip all four points.
    mov  eax, [ DestX ]
    mov  ecx, [ SrcWidth ]
    mov  ebx, [ ClipMaxX ]
    add  ecx, eax
    inc  ebx
    mov  edx, [ ClipMinX ]
    sub  ecx, ebx
    jle  .noClipMaxXAdjust
    add  esi, ecx
.noClipMaxXAdjust
    sub  edx, eax
    jle  .noClipMinXAdjust
    add  esi, edx
    add  [ DestX ], edx
    add  [ srcX ], edx
.noClipMinXAdjust
    mov  eax, [ DestY ]
    mov  ecx, [ SrcHeight ]
    mov  ebx, [ ClipMaxY ]
    add  ecx, eax
    inc  ebx
    mov  edx, [ ClipMinY ]
    sub  ecx, ebx
    jle  .noClipMaxYAdjust
    add  edi, ecx
.noClipMaxYAdjust
    sub  edx, eax
    jle  .noClipMinYAdjust
    add  edi, edx
    add  [ DestY ], edx
    add  [ srcY ], edx
.noClipMinYAdjust
     
    ;  Calculate the number of rows (ebx) and columns (ecx) that are going 
    ;  to be blitted.
    mov  ebx, [ SrcHeight ]
    mov  ecx, [ SrcWidth ]
    sub  ebx, edi
    jle  .nothingToBlit
    sub  ecx, esi
    jle  .nothingToBlit
    mov  [ numColumns ], ecx

    ;  Calculate the destination buffer pointer (edi) and the row 
    ;  increment (edx).
    mov  eax, [ DestStart ] 
    mov  esi, [ DestY ]
    mov  edx, [ DestWidth ]
    mov  edi, [ eax + esi * 4 ]
    sub  edx, ecx
    add  edi, [ DestX ]

    ;  We use a jump table to go to the routine that will handle the
    ;  sprite blitting based on the transparency, the flipping status and
    ;  the presence/absence of a colour table.
    ;  The value used in the jump table is built as:
    ;           ___CXYTT
    ;  Where C is 1 if a colour table is used.  X indicates whether we are 
    ;  flipping horizontally, Y indicates whether we are flipping vertically
    ;  and TT is the transparency effect.
    mov  eax, [ Transparency ]
    cmp  dword [ ColourTable ], 0h
    je   .woColourTable
    or   eax, 10h
    ;  We make more use of the colour table then the row increment so 
    ;  for this case, move the row increment into memory and put the
    ;  colour table into the row increment register (edx).
    mov  [ destIncrement ], edx
    mov  edx, [ ColourTable ]
.woColourTable
    cmp  dword [ FlipX ], 0h
    je   .woFlipX
    or   eax, 08h
.woFlipX
    cmp  dword [ FlipY ], 0h
    je   .woFlipY
    or   eax, 04h
.woFlipY
    jmp dword [ spriteBlitTable + eax * 4 ]
    
.nothingToBlit
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSprite (No Colour, No Flip, No Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSprite:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  eax, [ SrcStart ]
    mov  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ srcX ]
    xor  eax, eax
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    inc  esi
    mov  [ edi ], al
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, edx
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteRaw (No Colour, No Flip, Raw Trans)
;;  JumpPoint: blitSpriteMap (No Colour, No Flip, Map Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteRaw:
blitSpriteMap:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  eax, [ SrcStart ]
    mov  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ srcX ]
    xor  eax, eax
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    inc  esi
    cmp  eax, [ TransparentColour ]
    je   .skipPixel
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, edx
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteReverse (No Colour, NoFlip, Reverse Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteReverse:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  eax, [ SrcStart ]
    mov  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ srcX ]
    xor  eax, eax
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    mov  al, [ edi ]
    cmp  al, [ TransparentColour ]
    je   .skipPixel
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    mov  [ edi ], al
.skipPixel
    inc  edi
    inc  esi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, edx
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteY (No Colour, Y Flip, No Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteY:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    add  ebp, ecx
    mov  esi, [ eax + 4 * esi ]
    xor  eax, eax
    add  esi, [ srcX ]
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    inc  esi
    mov  [ edi ], al
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, edx
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteYRaw (No Colour, Y Flip, Raw Trans)
;;  JumpPoint: blitSpriteYMap (No Colour, Y Flip, Map Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteYRaw:
blitSpriteYMap:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    add  ebp, ecx
    mov  esi, [ eax + 4 * esi ]
    xor  eax, eax
    add  esi, [ srcX ]
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    inc  esi
    cmp  eax, [ TransparentColour ]
    je   .skipPixel
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, edx
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteYReverse (No Colour, Y Flip, Reverse Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteYReverse:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    add  ebp, ecx
    mov  esi, [ eax + 4 * esi ]
    xor  eax, eax
    add  esi, [ srcX ]
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ edi ]
    cmp  al, [ TransparentColour ]
    je   .skipPixel
    mov  al, [ esi ]
    mov  [ edi ], al
.skipPixel
    inc  edi
    inc  esi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, edx
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteX (No Colour, X Flip, No Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteX:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ srcY ]
    mov  eax, [ SrcStart ]
    mov  esi, [ eax + 4 * esi ]
    mov  ebp, [ SrcWidth ]
    add  esi, ebp
    add  ebp, ecx
    sub  esi, [ srcX ]
    xor  eax, eax
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    dec  esi
    mov  [ edi ], al
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, edx
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteXRaw (No Colour, X Flip, Raw Trans)
;;  JumpPoint: blitSpriteXMap (No Colour, X Flip, Map Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteXRaw:
blitSpriteXMap:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ srcY ]
    mov  eax, [ SrcStart ]
    mov  esi, [ eax + 4 * esi ]
    mov  ebp, [ SrcWidth ]
    add  esi, ebp
    add  ebp, ecx
    sub  esi, [ srcX ]
    xor  eax, eax
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    dec  esi
    cmp  eax, [ TransparentColour ]
    je   .skipPixel
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, edx
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteXReverse (No Colour, X Flip, Reverse Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteXReverse:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ srcY ]
    mov  eax, [ SrcStart ]
    mov  esi, [ eax + 4 * esi ]
    mov  ebp, [ SrcWidth ]
    add  esi, ebp
    add  ebp, ecx
    sub  esi, [ srcX ]
    xor  eax, eax
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ edi ]
    cmp  al, [ TransparentColour ]
    je   .skipPixel
    mov  al, [ esi ]
    mov  [ edi ], al
.skipPixel
    inc  edi
    inc  esi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, edx
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteXY (No Colour, XY Flip, No Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteXY:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ SrcWidth ]
    xor  eax, eax
    sub  esi, [ srcX ]
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    dec  esi
    mov  [ edi ], al
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, edx
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteXYRaw (No Colour, XY Flip, Raw Trans)
;;  JumpPoint: blitSpriteXYMap (No Colour, XY Flip, Map Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteXYRaw:
blitSpriteXYMap:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ SrcWidth ]
    xor  eax, eax
    sub  esi, [ srcX ]
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    dec  esi
    cmp  eax, [ TransparentColour ]
    je   .skipPixel
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, edx
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteXYReverse (No Colour, XY Flip, Reverse Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteXYReverse:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ SrcWidth ]
    xor  eax, eax
    sub  esi, [ srcX ]
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ edi ]
    cmp  al, [ TransparentColour ]
    je   .skipPixel
    mov  al, [ esi ]
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  esi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, edx
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColour (Colour, No Flip, No Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColour:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  eax, [ SrcStart ]
    mov  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ srcX ]
    xor  eax, eax
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    mov  al, [ edx + eax * 4 ]
    inc  esi
    mov  [ edi ], al
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourRaw (Colour, No Flip, Raw Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourRaw:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  eax, [ SrcStart ]
    mov  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ srcX ]
    xor  eax, eax
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    inc  esi
    cmp  eax, [ TransparentColour ]
    je   .skipPixel
    mov  al, [ edx + eax * 4 ]
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourMap (Colour, No Flip, Map Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourMap:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  eax, [ SrcStart ]
    mov  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ srcX ]
    xor  eax, eax
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    mov  al, [ edx + eax * 4 ]
    inc  esi
    cmp  eax, [ TransparentColour ]
    je   .skipPixel
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourReverse (Colour, No Flip, Reverse Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourReverse:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  eax, [ SrcStart ]
    mov  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ srcX ]
    xor  eax, eax
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    mov  al, [ edi ]
    cmp  al, [ TransparentColour ]
    je   .skipPixel
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    mov  al, [ edx + eax * 4 ]
    mov  [ edi ], al
.skipPixel
    inc  edi
    inc  esi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourY (Colour, Y Flip, No Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourY:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    add  ebp, ecx
    mov  esi, [ eax + 4 * esi ]
    xor  eax, eax
    add  esi, [ srcX ]
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    mov  al, [ edx + eax * 4 ]
    inc  esi
    mov  [ edi ], al
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourYRaw (Colour, Y Flip, Raw Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourYRaw:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    add  ebp, ecx
    mov  esi, [ eax + 4 * esi ]
    xor  eax, eax
    add  esi, [ srcX ]
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    inc  esi
    cmp  eax, [ TransparentColour ]
    je   .skipPixel
    mov  al, [ edx + eax * 4 ]
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourYMap (Colour, Y Flip, Map Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourYMap:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    add  ebp, ecx
    mov  esi, [ eax + 4 * esi ]
    xor  eax, eax
    add  esi, [ srcX ]
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    mov  al, [ edx + eax * 4 ]
    inc  esi
    cmp  eax, [ TransparentColour ]
    je   .skipPixel
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourYReverse (Colour, Y Flip, Reverse Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourYReverse:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    add  ebp, ecx
    mov  esi, [ eax + 4 * esi ]
    xor  eax, eax
    add  esi, [ srcX ]
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ edi ]
    cmp  al, [ TransparentColour ]
    je   .skipPixel
    mov  al, [ esi ]
    mov  al, [ edx + eax * 4 ]
    mov  [ edi ], al
.skipPixel
    inc  edi
    inc  esi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourX (Colour, X Flip, No Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourX:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ srcY ]
    mov  eax, [ SrcStart ]
    mov  esi, [ eax + 4 * esi ]
    mov  ebp, [ SrcWidth ]
    add  esi, ebp
    add  ebp, ecx
    sub  esi, [ srcX ]
    xor  eax, eax
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    mov  al, [ edx + eax * 4 ]
    dec  esi
    mov  [ edi ], al
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourXRaw (Colour, X Flip, Raw Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourXRaw:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ srcY ]
    mov  eax, [ SrcStart ]
    mov  esi, [ eax + 4 * esi ]
    mov  ebp, [ SrcWidth ]
    add  esi, ebp
    add  ebp, ecx
    sub  esi, [ srcX ]
    xor  eax, eax
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    dec  esi
    cmp  eax, [ TransparentColour ]
    je   .skipPixel
    mov  al, [ edx + eax * 4 ]
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourXMap (Colour, X Flip, Map Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourXMap:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ srcY ]
    mov  eax, [ SrcStart ]
    mov  esi, [ eax + 4 * esi ]
    mov  ebp, [ SrcWidth ]
    add  esi, ebp
    add  ebp, ecx
    sub  esi, [ srcX ]
    xor  eax, eax
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    mov  al, [ edx + eax * 4 ]
    dec  esi
    cmp  eax, [ TransparentColour ]
    je   .skipPixel
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourXReverse (Colour, X Flip, Reverse Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourXReverse:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ srcY ]
    mov  eax, [ SrcStart ]
    mov  esi, [ eax + 4 * esi ]
    mov  ebp, [ SrcWidth ]
    add  esi, ebp
    add  ebp, ecx
    sub  esi, [ srcX ]
    xor  eax, eax
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ edi ]
    cmp  al, [ TransparentColour ]
    je   .skipPixel
    mov  al, [ esi ]
    mov  al, [ edx + eax * 4 ]
    mov  [ edi ], al
.skipPixel
    inc  edi
    inc  esi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    add  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourXY (Colour, XY Flip, No Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourXY:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ SrcWidth ]
    xor  eax, eax
    sub  esi, [ srcX ]
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    mov  al, [ edx + eax * 4 ]
    dec  esi
    mov  [ edi ], al
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourXYRaw (Colour, XY Flip, Raw Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourXYRaw:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ SrcWidth ]
    xor  eax, eax
    sub  esi, [ srcX ]
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    dec  esi
    cmp  eax, [ TransparentColour ]
    je   .skipPixel
    mov  al, [ edx + eax * 4 ]
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourXYMap (Colour, XY Flip, Map Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourXYMap:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ SrcWidth ]
    xor  eax, eax
    sub  esi, [ srcX ]
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ esi ]
    mov  al, [ edx + eax * 4 ]
    dec  esi
    cmp  eax, [ TransparentColour ]
    je   .skipPixel
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  JumpPoint: blitSpriteColourXYReverse (Colour, XY Flip, Reverse Trans)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
align
blitSpriteColourXYReverse:
    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  esi, [ SrcHeight ]
    mov  eax, [ SrcStart ]
    sub  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    dec  esi
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ SrcWidth ]
    xor  eax, eax
    sub  esi, [ srcX ]
    dec  esi
    ;  Loop for each row of the source and each column within each row.
.pixelLoop
    ;  Get the source byte and store it in the destination.
    mov  al, [ edi ]
    cmp  al, [ TransparentColour ]
    je   .skipPixel
    mov  al, [ esi ]
    mov  al, [ edx + eax * 4 ]
    mov  [ edi ], al
.skipPixel
    inc  edi
    dec  esi
    dec  ecx
    jnz  .pixelLoop
    mov  ecx, [ numColumns ]
    add  edi, [ destIncrement ]
    sub  esi, ebp
    dec  ebx
    jnz  .pixelLoop 
    ExitFunction


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  Function: bitmapBlitScroll
;;
;;  Description:
;;
;;      This is an external function which is used to blit a large scrollable
;;      bitmap onto another.
;;
;;  Parameters:
;;
;;      None (the parameters are passed via our global blit info structure).
;;
;;  Returns:
;;
;;      Nothing.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
global	_bitmapBlitScroll
global	 bitmapBlitScroll
global	 bitmapBlitScroll_

align

_bitmapBlitScroll:
 bitmapBlitScroll:
 bitmapBlitScroll_:

    EnterFunction

    ;  Here we attack the X, Y coordinates of the source and destination 
    ;  and keep a running total of the amount clipped in the X and Y
    ;  directions.
    ;  The following registers will be used:
    ;      esi - X clipped
    ;      edi - Y clipped
    xor  esi, esi
    xor  edi, edi
    mov  [ srcX ], esi
    mov  [ srcY ], edi

    ;  Clip all four points.
    mov  eax, [ DestX ]
    mov  ecx, [ SrcWidth ]
    mov  ebx, [ ClipMaxX ]
    add  ecx, eax
    inc  ebx
    mov  edx, [ ClipMinX ]
    sub  ecx, ebx
    jle  .noClipMaxXAdjust
    add  esi, ecx
.noClipMaxXAdjust
    sub  edx, eax
    jle  .noClipMinXAdjust
    add  esi, edx
    add  [ DestX ], edx
    add  [ srcX ], edx
.noClipMinXAdjust
    mov  eax, [ DestY ]
    mov  ecx, [ SrcHeight ]
    mov  ebx, [ ClipMaxY ]
    add  ecx, eax
    inc  ebx
    mov  edx, [ ClipMinY ]
    sub  ecx, ebx
    jle  .noClipMaxYAdjust
    add  edi, ecx
.noClipMaxYAdjust
    sub  edx, eax
    jle  .noClipMinYAdjust
    add  edi, edx
    add  [ DestY ], edx
    add  [ srcY ], edx
.noClipMinYAdjust
     
    ;  Calculate the number of rows (ebx) and columns (ecx) that are going 
    ;  to be blitted.
    mov  ebx, [ SrcHeight ]
    mov  ecx, [ SrcWidth ]
    sub  ebx, edi
    jle  near .nothingToBlit
    sub  ecx, esi
    jle  near .nothingToBlit

    ;  Calculate the destination buffer pointer (edi) and the row 
    ;  increment (edx).
    mov  eax, [ DestStart ] 
    mov  esi, [ DestY ]
    mov  edx, [ DestWidth ]
    mov  edi, [ eax + esi * 4 ]
    sub  edx, ecx
    add  edi, [ DestX ]

    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  eax, [ SrcStart ]
    mov  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ srcX ]

    ;  Calculate the pillars and the columns.  Pillars are simply groups
    ;  of 4 columns so that we can move 32 bits at a time.
    mov  eax, ecx
    shr  eax, 3
    and  ecx, 07h
    mov  [ numPillars ], eax
    mov  [ numColumns ], ecx
    xor  eax, eax

    ;  Loop for each row of the source and each column within each row.
.rowLoop
    cmp  dword [ numPillars ], 0h
    je   .noPillars
    mov  ecx, [ numPillars ]
.pillarLoop
    mov  eax, dword [ esi ]
    add  esi, 4
    mov  dword [ edi ], eax
    add  edi, 4
    mov  eax, dword [ esi ]
    add  esi, 4
    mov  dword [ edi ], eax
    add  edi, 4
    dec  ecx
    jnz  .pillarLoop
.noPillars
    cmp  dword [ numColumns ], 0h
    je   .noColumns 
    mov  ecx, [ numColumns ]
.columnLoop
    mov  al, [ esi ]
    inc  esi
    mov  [ edi ], al
    inc  edi
    dec  ecx
    jnz  .columnLoop
.noColumns
    add  edi, edx
    add  esi, ebp
    dec  ebx
    jnz  .rowLoop 
    ExitFunction
    
.nothingToBlit
    ExitFunction


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  Function: bitmapBlitScrollTransparent
;;
;;  Description:
;;
;;      This is an external function which is used to blit a large scrollable
;;      bitmap onto another.
;;
;;  Parameters:
;;
;;      None (the parameters are passed via our global blit info structure).
;;
;;  Returns:
;;
;;      Nothing.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
global	_bitmapBlitScrollTransparent
global	 bitmapBlitScrollTransparent
global	 bitmapBlitScrollTransparent_

align

_bitmapBlitScrollTransparent:
 bitmapBlitScrollTransparent:
 bitmapBlitScrollTransparent_:

    EnterFunction

    ;  Here we attack the X, Y coordinates of the source and destination 
    ;  and keep a running total of the amount clipped in the X and Y
    ;  directions.
    ;  The following registers will be used:
    ;      esi - X clipped
    ;      edi - Y clipped
    xor  esi, esi
    xor  edi, edi
    mov  [ srcX ], edi
    mov  [ srcY ], edi

    ;  Clip all four points.
    mov  eax, [ DestX ]
    mov  ecx, [ SrcWidth ]
    mov  ebx, [ ClipMaxX ]
    add  ecx, eax
    inc  ebx
    mov  edx, [ ClipMinX ]
    sub  ecx, ebx
    jle  .noClipMaxXAdjust
    add  esi, ecx
.noClipMaxXAdjust
    sub  edx, eax
    jle  .noClipMinXAdjust
    add  esi, edx
    add  [ DestX ], edx
    add  [ srcX ], edx
.noClipMinXAdjust
    mov  eax, [ DestY ]
    mov  ecx, [ SrcHeight ]
    mov  ebx, [ ClipMaxY ]
    add  ecx, eax
    inc  ebx
    mov  edx, [ ClipMinY ]
    sub  ecx, ebx
    jle  .noClipMaxYAdjust
    add  edi, ecx
.noClipMaxYAdjust
    sub  edx, eax
    jle  .noClipMinYAdjust
    add  edi, edx
    add  [ DestY ], edx
    add  [ srcY ], edx
.noClipMinYAdjust
     
    ;  Calculate the number of rows (ebx) and columns (ecx) that are going 
    ;  to be blitted.
    mov  ebx, [ SrcHeight ]
    mov  ecx, [ SrcWidth ]
    sub  ebx, edi
    jle  near .nothingToBlit
    sub  ecx, esi
    jle  near .nothingToBlit

    ;  Calculate the destination buffer pointer (edi) and the row 
    ;  increment (edx).
    mov  eax, [ DestStart ] 
    mov  esi, [ DestY ]
    mov  edx, [ DestWidth ]
    mov  edi, [ eax + esi * 4 ]
    sub  edx, ecx
    add  edi, [ DestX ]
    mov  [ destIncrement ], edx

    ;  Calculate the source buffer pointer (esi) and row increment (ebp).
    mov  eax, [ SrcStart ]
    mov  esi, [ srcY ]
    mov  ebp, [ SrcWidth ]
    mov  esi, [ eax + 4 * esi ]
    sub  ebp, ecx
    add  esi, [ srcX ]

    ;  Calculate the pillars and the columns.
    mov  eax, ecx
    shr  eax, 3
    and  ecx, 07h
    mov  [ numPillars ], eax
    mov  [ numColumns ], ecx
    xor  eax, eax

    ;  Move the transparent colour into a register.
    mov  edx, [ TransparentColour ]

    ;  Loop for each row of the source and each column within each row.
.rowLoop
    cmp  dword [ numPillars ], 0h
    je   .noPillars
    mov  ecx, [ numPillars ]
.pillarLoop
    mov  al, [ esi ]
    inc  esi
    cmp  eax, edx
    je   .skipPixel0
    mov  [ edi ], al
.skipPixel0
    inc  edi
    mov  al, [ esi ]
    inc  esi
    cmp  eax, edx
    je   .skipPixel1
    mov  [ edi ], al
.skipPixel1
    inc  edi
    mov  al, [ esi ]
    inc  esi
    cmp  eax, edx
    je   .skipPixel2
    mov  [ edi ], al
.skipPixel2
    inc  edi
    mov  al, [ esi ]
    inc  esi
    cmp  eax, edx
    je   .skipPixel3
    mov  [ edi ], al
.skipPixel3
    inc  edi
    mov  al, [ esi ]
    inc  esi
    cmp  eax, edx
    je   .skipPixel4
    mov  [ edi ], al
.skipPixel4
    inc  edi
    mov  al, [ esi ]
    inc  esi
    cmp  eax, edx
    je   .skipPixel5
    mov  [ edi ], al
.skipPixel5
    inc  edi
    mov  al, [ esi ]
    inc  esi
    cmp  eax, edx
    je   .skipPixel6
    mov  [ edi ], al
.skipPixel6
    inc  edi
    mov  al, [ esi ]
    inc  esi
    cmp  eax, edx
    je   .skipPixel7
    mov  [ edi ], al
.skipPixel7
    inc  edi
    dec  ecx
    jnz  .pillarLoop
.noPillars
    cmp  dword [ numColumns ], 0h
    je   .noColumns 
    mov  ecx, [ numColumns ]
.columnLoop
    mov  al, [ esi ]
    inc  esi
    cmp  eax, edx
    je   .skipPixel8
    mov  [ edi ], al
.skipPixel8
    inc  edi
    dec  ecx
    jnz  .columnLoop
.noColumns
    add  edi, [ destIncrement ]
    add  esi, ebp
    dec  ebx
    jnz  near .rowLoop 
    ExitFunction
    
.nothingToBlit
    ExitFunction

end
