/* 
 * Rock-ulator (aka Vanguard Emulator) v.86 copyright 1997 by Brian Levine
 * with apologies to Moose for the new name
 *
 * The goal of this emulator is to faithfully reproduce the experience of playing
 * the Vanguard arcade machine, and to share the resulting code with whoever is 
 * interested in it.  No one is allowed to redistribute either the source code or 
 * the executable program for profit, nor is it to be distributed with Vanguard ROMs.
 *
 * Brian Levine (blevine@systems.titan.com)
 *
 * 0.05 	Continuing the work...Game is now (barely) playable
 * 0.051	20% faster, quick start hack (yuck!), and setup for real colors
 * 0.06  Now using Multi-6502 Emulation Engine by Neil Bradley
 *       This fixes the game start problem!  Should be faster, too
 *       Score no longer displays as Hex.  Beginning origional color support
 * 0.65  Modified m6502.o so that no changes to header required.  Thanks, Allard!
 *       Joystick support added.  High score saved.  Screen dump added. All 
 *       colors are now correct.  Also, I finally adjusted version number to 
 *       reflect true status of the emulator.
 * 0.7   Custom blit function increases speed 25%.
 *  		Preliminary sound support added.  Preliminary gamepad support added.
 * 0.71  Fixed bug in custom blit and added swtich to disable it
 * 0.72  Sped up custom blit by another 20%.  Continued modifications to music.
 *       Fixed bug in screen dump.  Assorted other fixes.
 * 0.80  Major modifications to music (after discovering multi6502 bug).
 *       Background color enabled.  Various bug fixes.
 * 0.85  Added support for Fantasy and Nibbler.  
 * 0.86  Numerous fixes for multigame mode.  Game save feature.  Other bug fixes.
 *
 * Vanguard Emulator 0.02 by Allard van der Bas (avdbas@wi.leidenuniv.nl)
 * Ported to GCC / Allegro.
 *
 * Original Vanguard Emulator 0.01 code (Borland C) by Ivan.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <conio.h>
#include <allegro.h>  
#include <audio.h>
#include <dos.h>
#include <time.h>
#include <sys\movedata.h>
#include "m6502.h"

//-- comment out for normal compile
//#define SOUNDTEST    
//#define DUMPDATA
//---------------------------------

//-- prototypes 
void xblit(BITMAP *source, BITMAP *dest, int source_x, int source_y, 
           int dest_x, int dest_y, int width, int height, int topline);
void xblitvsync(BITMAP *source, BITMAP *dest, int source_x, int source_y, 
           int dest_x, int dest_y, int width, int height, int topline);
void LoadPalettes();
void SetupVideo();
int  LoadRom(int Address, char * Name, size_t Size);
void DrawSprite (int c, int col, int row, BYTE color);
void DrawCharacter (int c, int col, int row, BYTE color);
void DisplayForegroundChars(void);
void DisplayBackgroundChars(void);
void Exec();
int  SetupAudio();
void ShutdownAudio();
BYTE Interrupt(void);
int  SoundOut0(register WORD A, register BYTE B);
int  SoundOut1(register WORD A, register BYTE B);
int  SoundOut2(register WORD A, register BYTE B);
int  ColorOut(register WORD A, register BYTE B);
int  SpeechOut(register WORD A, register BYTE B);
int  BackgroundUpdate(register WORD A, register BYTE B);
int  ForegroundUpdate(register WORD A, register BYTE B);
WORD ReadCoin(register WORD A);
WORD ReadPlayer1(register WORD A);
WORD ReadPlayer2(register WORD A);
void Help();
void LoadGame();
void SaveGame();
void LoadHiScore();
void SaveHiScore();
void PlaySample (int channel, BYTE *data, int len, int freq, int vol, int loop);
void AdjustSample (int channel, int freq, int vol);
void PlayAudio();
WORD GetBit(int startbit, unsigned int off);
void PrintVal(int add, int row, int col);
void PrintNum(int add, int row, int col);
void PrintStr(char *str, int row, int col);
void cBlit (int x, int y, int col, int row, int w, int h);
WORD ReadRom(register WORD A);

void (*xb)(BITMAP *, BITMAP *, int, int, int, int, int, int, int);

#define INTCOUNT     12000

#define NUMVOICES    5        
#define SAMPLE_RATE  44100
#define SAMPLE_LEN   50000

enum interrupt_types { INT_NONE=0, INT_IRQ, INT_NMI, INT_QUIT };

typedef struct
{
   WORD loadAddr;
   char *fileName;
   WORD romSize;
   char *desc;
} ROMS;

enum Games { VANGUARD=0, FANTASY, NIBBLER };

typedef struct 
{
   WORD GameID;
   ROMS *Roms;
   WORD BChar1;
   WORD BChar2;
   WORD Sound1;
   WORD Sound2;
   WORD Port1;
   WORD Port2;
   WORD Port3;
   WORD Port4;
   WORD Player1;
   WORD Player2;
   WORD DipSwitch;
   WORD Coin;
   WORD VScroll;
   WORD HScroll;
   WORD Speech;
   char *Path;
} GameAddresses;

ROMS vanguardRoms[] = 
{
   { 0x2000, "sk5_ic50.bin", 2048, "background character graphics - plane 1" },
   { 0x2800, "sk5_ic51.bin", 2048, "background character graphics - plane 2" },
   { 0x4000, "sk4_ic07.bin", 4096, "code" },
   { 0x5000, "sk4_ic08.bin", 4096, "code" },
   { 0x6000, "sk4_ic09.bin", 4096, "code" },
   { 0x7000, "sk4_ic10.bin", 4096, "code, sprite graphics, tables" },
   { 0x8000, "sk4_ic13.bin", 4096, "foreground character graphics" },
   { 0x9000, "sk4_ic14.bin", 4096, "tables (sound?)" },
   { 0xA000, "sk4_ic15.bin", 4096, "more tables?" },
   { 0xB000, "sk4_ic16.bin", 4096, "data (1K), code" },
   { 0xC000, "sk4_ic51.bin", 2048, "sound - part 1" },
   { 0xC800, "sk4_ic52.bin", 2048, "sound - part 2" },
   { 0xF000, "sk4_ic13.bin", 4096, "vectors" },
   { 0, NULL, 0, NULL }	
};

ROMS fantasyRoms[] = 
{
   { 0x3000, "ic12.cpu",	4096, "code" },
   { 0x4000, "ic07.cpu", 	4096, "code" },
   { 0x5000, "ic08.cpu", 	4096, "code" },
   { 0x6000, "ic09.cpu", 	4096, "code" },
   { 0x7000, "ic10.cpu",  	4096, "code / graphics" },
   { 0x8000, "ic14.cpu",  	4096, "graphics" },
   { 0x9000, "ic15.cpu",  	4096, "data" },
   { 0xa000, "ic16.cpu",  	4096, "code - graphics" },
   { 0xb000, "ic17.cpu",  	4096, "graphics - code" },
   { 0xC000, "ic50.vid",   4096, "graphics" },
   { 0xD000, "ic51.vid",   4096, "graphics" },
   { 0xF000, "ic14.cpu",  	4096, "vectors" },
   { 0xE000, "ic51.cpu",   2048, "sound" },
   { 0xE800, "ic52.cpu",   2048, "sound" },
   { 0xF000, "ic53.cpu",   2048, "sound" },
   { 0, NULL, 0, NULL }	
};

ROMS nibblerRoms[] = 
{
   { 0x3000, "ic12",	4096, "code" },
   { 0x4000, "ic07", 	4096, "code" },
   { 0x5000, "ic08", 	4096, "code" },
   { 0x6000, "ic09", 	4096, "code" },
   { 0x7000, "ic10",  	4096, "code / graphics" },
   { 0x8000, "ic14",  	4096, "graphics" },
   { 0x9000, "ic15",  	4096, "data" },
   { 0xa000, "ic16",  	4096, "code - graphics" },
   { 0xb000, "ic17",  	4096, "graphics - code" },
   { 0xC000, "ic50",   4096, "graphics" },
   { 0xD000, "ic51",   4096, "graphics" },
   { 0xF000, "ic14",  	4096, "vectors" },
   { 0xE000, "ic52",   2048, "sound" },
   { 0xE800, "ic53",   2048, "sound" },
   { 0, NULL, 0, NULL }	
};

GameAddresses gaVanguard = 
{
   VANGUARD, vanguardRoms,
   0x2000, 0x2800,
   0xC000, 0xC800,
   0x3100, 0x3101, 0x3102, 0x3103,
   0x3104, 0x3105, 0x3106, 0x3107,
   0x3200, 0x3300, 0x3400,
   "Vanguard"
};

GameAddresses gaFantasy = 
{
   FANTASY, fantasyRoms,
   0xC800, 0xD800,
   0xE000, 0xE800,
   0x2100, 0x2101, 0x2102, 0x2103,
   0x2104, 0x2105, 0x2106, 0x2107,
   0x2200, 0x2300, 0x2400,
   "Fantasy"
};

GameAddresses gaNibbler = 
{
   NIBBLER, nibblerRoms,
   0xC800, 0xD800,
   0xE000, 0xE800,
   0x2100, 0x2101, 0x2102, 0x2103,
   0x2104, 0x2105, 0x2106, 0x2107,
   0x2200, 0x2300, 0x2400,
   "Nibbler"
};

struct m6502MemoryWrite VanguardWrite[6];
struct m6502MemoryRead VanguardRead[4];

BYTE Foreground[0x400];
BYTE Background[0x400];

// Global variables
GameAddresses *cga;
BYTE *RAM;
BITMAP *piccie;
BITMAP *fchars;
BITMAP *bchars;

//BYTE pal[3*256];
PALETTE pal;
WORD ExitAdress;

int sw_dumpgraphics = 0;
int sw_men = 0x00;
int sw_nojoystick = 0;
int sw_gravis = 0;
int sw_nosound = 0;
int sw_noxblit = 0;
int sw_useX = 0;
int sw_v360 = 0;
int sw_sb16 = 0;
int sw_sb = 0;
int sw_vsync = 0;

int joy_next_shot = 0;
int score_loaded = 0;
int sound_test = 0;

BOOL bNextSound = FALSE;
int NoSound = TRUE;
int NoSound0;
int Sound0;
int Sound0Base;
int oSound0Base;
int Sound0Count;
int NoSound1;
int Sound1;
int Sound1Base;
int oSound1Base;
int Sound1Count;
HAC hVoice[NUMVOICES];
LPAUDIOWAVE lpWave[NUMVOICES];

int LoopCounter = 0;
BYTE LastPort1 = 0;
BYTE LastPort2 = 0;
BYTE LastPort3 = 0;
BYTE LastPort4 = 0;
FILE *fptr;

int ActivePage = 0;
BOOL pause = FALSE;

int SoundSwap[] = { 0, 4, 2, 6, 1, 5, 3, 7 };

//-- sine wave sample
static BYTE samples[32] = 
{
   0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x22, 
   0x44, 0x44, 0x44, 0x44, 0x22, 0x22, 0x22, 0x22, 
   0x00, 0x00, 0x00, 0x00, 0xdd, 0xdd, 0xdd, 0xdd, 
   0xbb, 0xbb, 0xbb, 0xbb, 0xdd, 0xdd, 0xdd, 0xdd
};

//-- Background color table - decoded from color ROMs
BYTE bPal[8][4];

//-- Foreground color table - decoded from color ROMs
BYTE fPal[8][4];

//-- color intensity based on 2 bit definition in color ROMs
BYTE intensity4[] = { 0, 31, 47, 63 };
BYTE intensity8[] = { 0, 15, 23, 31, 39, 47, 55, 63 };

void LoadPalettes()
{
   int i;
   FILE * fp;
   BYTE buffer[32];
   char *bpal = "vanguard\\sk5_ic6.bin";
   char *fpal = "vanguard\\sk5_ic7.bin";

   //-- open the background palette ROM
   if ((fp = fopen(bpal, "rb")) != NULL)
      {
      fread(buffer, 32, 1, fp);
      for (i=0; i<8; i++)
         {
         bPal[i][0] = 0;
         bPal[i][2] = buffer[i*4+1];
         bPal[i][1] = buffer[i*4+2];
         bPal[i][3] = buffer[i*4+3];
         }
      }
   fclose(fp);

   //-- open the foreground palette ROM
   if ((fp = fopen(fpal, "rb")) != NULL)
      {
      fread(buffer, 32, 1, fp);
      for (i=0; i<8; i++)
         {
         fPal[i][0] = 0;
         fPal[i][2] = buffer[i*4+1];
         fPal[i][1] = buffer[i*4+2];
         fPal[i][3] = buffer[i*4+3];
         }
      }
   fclose(fp);
}

void SetupVideo () 
{
   WORD i;
   BYTE b=0;	

   if (sw_v360)
      set_gfx_mode(GFX_MODEX, 360,360, 0,720);
   else
      set_gfx_mode(GFX_MODEX, 256,256, 0,512);

   for (i=0; i<256; i++, b++)
      {
      pal[i].r = intensity8[b&0x7];
      pal[i].g = intensity8[(b&0x38)>>3];
      pal[i].b = intensity4[(b&0xc0)>>6];
      }
   LoadPalettes();
   set_palette(pal);

   piccie = create_bitmap(256,256);
   fchars = create_bitmap(256, 64);
   bchars = create_bitmap(256, 64);

   clear (piccie);
   clear (fchars);
}

void SaveGame()
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\savegame", cga->Path);
   fp = fopen (file, "wb");
   fwrite (RAM, 0x10000, 1, fp);
	fwrite (&m6502clockticks, 4, 1, fp);
	fwrite (&m6502pc, 4, 1, fp);
	fwrite (&m6502af, 4, 1, fp);
	fwrite (&m6502x, 4, 1, fp);
	fwrite (&m6502y, 4, 1, fp);
	fwrite (&m6502s, 4, 1, fp);
	fwrite (&cyclesRemaining, 4, 1, fp);
	fwrite (&syncCycle, 4, 1, fp);
   fclose (fp);
}

void LoadGame()
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\savegame", cga->Path);
   fp = fopen (file, "rb");
   if (fp != NULL)
      {
      fread (RAM, 0x10000, 1, fp);
   	fread (&m6502clockticks, 4, 1, fp);
	   fread (&m6502pc, 4, 1, fp);
   	fread (&m6502af, 4, 1, fp);
	   fread (&m6502x, 4, 1, fp);
   	fread (&m6502y, 4, 1, fp);
	   fread (&m6502s, 4, 1, fp);
   	fread (&cyclesRemaining, 4, 1, fp);
	   fread (&syncCycle, 4, 1, fp);
      fclose (fp);
      }
}

void SaveHiScore()
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\hiscore", cga->Path);
   fp = fopen (file, "wb");
   fwrite (RAM + 0x220, 0x70, 1, fp);
   fclose (fp);
}

void LoadHiScore()
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\hiscore", cga->Path);
   fp = fopen (file, "rb");
   if (fp == NULL)
      return;
   fread (RAM + 0x220, 0x70, 1, fp);
   fclose (fp);

   //-- now copy high score from table to current high score 
   RAM[0x25] = RAM[0x22b];
   RAM[0x26] = RAM[0x22c];
   RAM[0x27] = RAM[0x22d];
}

void Help()
{
   clrscr();
	printf ("Vanguard/Fantasy/Nibbler Emulator by Brian Levine\n\n");
   printf ("   Multi-6502 Emulation Engine by Neil Bradley\n");
   printf ("   Allegro Library by Shawn Hargreaves\n");
   printf ("   SEAL Synthetic Audio Library API Interface Copyright (C) 1995, 1996\n");
   printf ("      Carlos Hasan. All Rights Reserved\n\n");
   printf ("Command Line Switches:\n");
   printf ("   -d         display ROM graphics before starting emulation\n");
   printf ("   -fantasy   play Fantasy\n");
   printf ("   -m3/4/5/6  3/4/5 ships per game (default 3)\n");
   printf ("   -nibbler   play Nibbler\n");
   printf ("   -nojoy     disable joystick support\n");
   printf ("   -nosound   disable sound support\n");
   printf ("   -noxblit   disable custom blit\n");
   printf ("   -sb/sb16   force sound support to Soundblaster/Soundblaster16\n");
   printf ("   -v360      use 360 x 360 video mode\n");
   printf ("   -vsync     forces custom blit to wait for vsync\n");
   printf ("   -x         use X key to fire down\n\n");
   printf ("<Hit any key to begin emulation>\n");
   getch();
}

int SetupAudio()
{
	int i;
   unsigned int ndevice;
   AUDIOINFO info;

   if (AInitialize())
      return 1;

	APingAudio(&ndevice);

   info.wFormat = AUDIO_FORMAT_8BITS | AUDIO_FORMAT_MONO;
   info.nSampleRate = SAMPLE_RATE;

   if (sw_sb16)
   	info.nDeviceId = AUDIO_PRODUCT_SB16;
	else if (sw_sb)
   	info.nDeviceId = AUDIO_PRODUCT_SB;
   else
      info.nDeviceId = AUDIO_DEVICE_MAPPER;


   if (AOpenAudio(&info))
      return 1;

	AOpenVoices(NUMVOICES);
   //-- allocate voices and waveforms
   for(i=0; i<NUMVOICES; i++)
      {
      ACreateAudioVoice(&hVoice[i]);
      ASetVoicePanning(hVoice[i], 0x80);

      if ((lpWave[i] = (LPAUDIOWAVE)malloc(sizeof(AUDIOWAVE))) == 0)
         {
         //-- ERROR - free audio memory and exit
         int j;
         for(j=0; j<i; j++)
            {
            ADestroyAudioVoice(hVoice[j]);
            ADestroyAudioData(lpWave[j]);
            free(lpWave[j]);
            }
         ADestroyAudioVoice(hVoice[i]);
         return 1;
         }

      if (i < NUMVOICES-2)
         {
         lpWave[i]->wFormat = AUDIO_FORMAT_8BITS | AUDIO_FORMAT_MONO;
         lpWave[i]->nSampleRate = SAMPLE_RATE;
         lpWave[i]->dwLength = SAMPLE_LEN;
         lpWave[i]->dwLoopStart = lpWave[i]->dwLoopEnd = 0;

         if (ACreateAudioData(lpWave[i]) != AUDIO_ERROR_NONE)
            {
            //-- ERROR - free audio memory
            int j;
            for(j=0; j<i; j++)
               {
               ADestroyAudioVoice(hVoice[j]);
               ADestroyAudioData(lpWave[j]);
               free(lpWave[j]);
               }
            ADestroyAudioVoice(hVoice[i]);
            free(lpWave[i]);
            return 1;
            }
         }
      }

   ALoadWaveFile("explsion.wav", &(lpWave[3]), 0L);
   ALoadWaveFile("fire.wav", &(lpWave[4]), 0L);
   ALoadWaveFile("fire.wav", &(lpWave[2]), 0L);

   NoSound0 = TRUE;
   Sound0 = cga->Sound1;
   Sound0Base = cga->Sound1;
   oSound0Base = cga->Sound1;
   Sound0Count = 0;
   NoSound1 = TRUE;
   Sound1 = cga->Sound2;
   Sound1Base = cga->Sound2;
   oSound1Base = cga->Sound2;
   Sound1Count = 0;

   return 0;
}

void ShutdownAudio()
{
	int i;

   for(i=0; i<NUMVOICES; i++)
      {
      AStopVoice(hVoice[i]);
      ADestroyAudioVoice(hVoice[i]);
      ADestroyAudioData(lpWave[i]);
      free(lpWave[i]);
      }

   //-- stop audio
   ACloseVoices();
   ACloseAudio();
}

int htoi(char c)
{
   if (c > '9')
      {
      c &= 0xDF;         
      return (c - 'A' + 10);
      }
   return (c - '0');
}

int main(int argc, char **argv) 
{

   int i;

   if (argc < 2)
      Help();

   //-- check arguments
   cga = &gaVanguard;
   for (i=1; i<argc; i++)
      {
      if (strcmp ("-fantasy", argv[i]) == 0)
         {
         cga = &gaFantasy;
         break;
         }
      if (strcmp ("-nibbler", argv[i]) == 0)
         {
         cga = &gaNibbler;
         break;
         }
      }

   for (i=1; i<argc; i++)
      {
      if (memcmp ("-ds", argv[i], 3) == 0)
         {
         sw_men = (htoi(argv[i][3])<<4)+htoi(argv[i][4]);
         break;
         }
      if (strcmp ("-m3", argv[i]) == 0)
         {
         break;
         }
      if (strcmp ("-m4", argv[i]) == 0)
         {
         switch (cga->GameID)
            {
            case VANGUARD:
               sw_men |= 0x10;
               break;
            case FANTASY:
               sw_men |= 0x20;
               break;
            case NIBBLER:
               sw_men |= 0x01;
               break;
            }
         break;
         }
      if (strcmp ("-m5", argv[i]) == 0)
         {
         switch (cga->GameID)
            {
            case VANGUARD:
               sw_men |= 0x20;
               break;
            case FANTASY:
               sw_men |= 0x40;
               break;
            case NIBBLER:
               sw_men |= 0x02;
               break;
            }
         break;
         }
      if (strcmp ("-m6", argv[i]) == 0)
         {
         switch (cga->GameID)
            {
            case VANGUARD:
               sw_men |= 0x20;
               break;
            case FANTASY:
               sw_men |= 0x40;
               break;
            case NIBBLER:
               sw_men |= 0x03;
               break;
            }
         break;
         }
      }

   //-- check for display graphics
   for (i=1; i<argc; i++)
      {
      if (strcmp ("-d", argv[i]) == 0)
         {
         sw_dumpgraphics = 1;
         break;
         }
      }
      
   //-- check for use x key
   for (i=1; i<argc; i++)
      {
      if (strcmp ("-x", argv[i]) == 0)
         {
         sw_useX = 1;
         break;
         }
      }
      
   //-- check for nosound
   for (i=1; i<argc; i++)
      {
      if (strcmp ("-nosound", argv[i]) == 0)
         {
         sw_nosound = 1;
         break;
         }
      }

   //-- check for soundblaster
   if (!sw_nosound)
      {
      for (i=1; i<argc; i++)
         {
         if (strcmp ("-sb16", argv[i]) == 0)
            {
            sw_sb16 = 1;
            break;
            }
         if (strcmp ("-sb", argv[i]) == 0)
            {
            sw_sb = 1;
            break;
            }
         }
      }

   //-- check for 360 x 360 video
   for (i=1; i<argc; i++)
      {
      if (strcmp ("-v360", argv[i]) == 0)
         {
         sw_v360 = 1;
         break;
         }
      }
      
   //-- check for no xblit
   xb = xblit;
   for (i=1; i<argc; i++)
      {
      if (strcmp ("-vsync", argv[i]) == 0)
         {
         sw_vsync = 1;
         xb = xblitvsync;
         break;
         }
      if (strcmp ("-noxblit", argv[i]) == 0)
         {
         sw_noxblit = 1;
         break;
         }
      }

   //-- check for joystick
   for (i=1; i<argc; i++)
      {
      if (strcmp ("-nojoy", argv[i]) == 0)
         {
         sw_nojoystick = 1;
         break;
         }
      }

   //-- check for gravis gamepad
   for (i=1; i<argc; i++)
      {
      if (strcmp ("-gravis", argv[i]) == 0)
         {
         sw_gravis = 1;
         break;
         }
      }

   //-- calibrate the joystick
   if (!sw_nojoystick)
      {        
      printf ("\nMake sure the joystick is centered...\n<hit any key>\n\n");
      getch();      
      if (initialise_joystick())
         sw_nojoystick = 0;
      else
         {
         if (sw_gravis)
            joy_type = JOY_TYPE_4BUTTON;
         else
            joy_type = JOY_TYPE_STANDARD;
         }
      }

   //-- initialize audio library
   if (!sw_nosound)
      {
      if (SetupAudio())
         {
         sw_nosound = 1;
         printf ("Error initializing audio.  Disabling sound.\n<Hit any key>\n");         	
         getch();
         }
      }

   printf("Allocating 64K memory\n");
   if(!(RAM=malloc(0x10000)))
      { 
      puts("FAILED");
      exit(1); 
      }

   printf("Loading Roms...\n");
   i = 0;
   while (cga->Roms[i].fileName)
      {
      if (!LoadRom(cga->Roms[i].loadAddr, cga->Roms[i].fileName, cga->Roms[i].romSize)) 
         {
         //-- free memory
         free(RAM);
         exit(1);
         }
      i++;
      }

   //-- this is a kludge - rework
   VanguardWrite[0].lowAddr = cga->Port1;
   VanguardWrite[0].highAddr = cga->Port1;
   VanguardWrite[0].memoryCall = SoundOut0;
   VanguardWrite[1].lowAddr = cga->Port2;
   VanguardWrite[1].highAddr = cga->Port2;
   VanguardWrite[1].memoryCall = SoundOut1;
   VanguardWrite[2].lowAddr = cga->Port3;
   VanguardWrite[2].highAddr = cga->Port3;
   VanguardWrite[2].memoryCall = SoundOut2;
   VanguardWrite[3].lowAddr = cga->Port4;
   VanguardWrite[3].highAddr = cga->Port4;
   VanguardWrite[3].memoryCall = ColorOut;
   VanguardWrite[4].lowAddr = cga->Speech;
   VanguardWrite[4].highAddr = cga->Speech;
   VanguardWrite[4].memoryCall = SpeechOut;
   VanguardWrite[5].lowAddr = (WORD)-1;
   VanguardWrite[5].highAddr = (WORD)-1;
   VanguardWrite[5].memoryCall = NULL; 

   VanguardRead[0].lowAddr = cga->Player1;
   VanguardRead[0].highAddr = cga->Player1;
   VanguardRead[0].memoryCall = ReadPlayer1;
   VanguardRead[1].lowAddr = cga->Player2;
   VanguardRead[1].highAddr = cga->Player2;
   VanguardRead[1].memoryCall = ReadPlayer2;
   VanguardRead[2].lowAddr = cga->Coin;
   VanguardRead[2].highAddr = cga->Coin;
   VanguardRead[2].memoryCall = ReadCoin;
   VanguardRead[3].lowAddr = (WORD)-1;
   VanguardRead[3].highAddr = (WORD)-1;
   VanguardRead[3].memoryCall = NULL;

   m6502Base = RAM;
   m6502reset();
   m6502MemoryRead = VanguardRead;
   m6502MemoryWrite = VanguardWrite;

   install_timer();
   install_keyboard();

   SetupVideo();

   //-- check for high refresh rate
   for (i=1; i<argc; i++)
      {
      if (strcmp ("-hr", argv[i]) == 0)
         {
         outportb(0x3c2, 0xef);
         break;
         }
      }
      
   DisplayForegroundChars();   

   // if 360 x 360 mode, display the backdrop - future feature
/*   if (sw_v360)
      {
      BITMAP *bmp;
      PALLETE pal;
      get_pallete(pal);
      bmp = load_pcx("logo.pcx", pal);
      if (bmp)
         {
         blit (bmp, screen, 0,0, 0,0, 360,360);
         blit (bmp, screen, 0,0, 0,360, 360,360);
         destroy_bitmap(bmp);
         }
      } */

   //-- start audio
   if (!sw_nosound)
      {
      PlaySample(0, samples, 32, 1000, 0, 1);
      PlaySample(1, samples, 32, 1000, 0, 1);
      APlayVoice(hVoice[2], lpWave[2]);
      APlayVoice(hVoice[3], lpWave[3]);
      APlayVoice(hVoice[4], lpWave[4]);
      }

#ifdef SOUNDTEST
   fptr = fopen ("sound.txt", "w");
#endif

   // run 
   RAM[cga->DipSwitch] = sw_men;
#ifndef SOUNDTEST
   if (!sw_nosound)
#endif
      install_int(PlayAudio, 30);
   Exec();

#ifdef SOUNDTEST
   fclose (fptr);
#endif

   SaveHiScore();

   set_gfx_mode(GFX_TEXT,80,25,0,0);   

   //-- free audio memory
   if (!sw_nosound)
      {
      ShutdownAudio();
      }

   //-- destroy the bitmaps
   destroy_bitmap(bchars);
   destroy_bitmap(fchars);
   destroy_bitmap(piccie);
   allegro_exit();

   //-- free memory and exit
   free(RAM);
   exit(0);
}

int LoadRom(int Address, char * Name, size_t Size) 
{                                     
   FILE * fp;
   char file[80];
   sprintf (file, "%s\\%s", cga->Path, Name);
   
   //-- open the ROM file
   printf("Loading %s ...", file);
   if ((fp = fopen(file, "rb")) == NULL)
   {
      printf("failed on open\n");
      return FALSE;
   }

   //-- read it into RAM image
   if (fread(RAM + Address, 1, Size, fp) != Size) {
         printf("failed on read\n");
         return FALSE;
   }
   fclose(fp);

   printf("\n");
   return TRUE;
}

//-- run emulation with masked interrupt every 12000 cycles
void Exec()
{
	BYTE rtn = INT_IRQ;
   while (rtn != INT_QUIT)
      {
      m6502exec(INTCOUNT);
      rtn = Interrupt();
      if (rtn == INT_NMI)
         {
         //-- force a non-masked interrupt
         m6502nmi();
         }
      else if (rtn == INT_IRQ)
         {
         //-- force a masked interrupt
         m6502int();
         if (score_loaded == 0)
            {
            LoadHiScore();
            score_loaded++;
            }
         }
      }
}

void hexdump()
{
#ifdef SOUNDTEST
   fprintf (fptr, "b%01d", (Data&0x80)>>7);	
   fprintf (fptr, "%01d", (Data&0x40)>>6);	
   fprintf (fptr, "%01d", (Data&0x20)>>5);	
   fprintf (fptr, "%01d", (Data&0x10)>>4);	
   fprintf (fptr, "%01d", (Data&0x08)>>3);	
   fprintf (fptr, "%01d", (Data&0x04)>>2);	
   fprintf (fptr, "%01d", (Data&0x02)>>1);	
   fprintf (fptr, "%01d - %d\n", (Data&0x01), LoopCounter);	
#endif
}

void PlayAudio()
{
   if (!sw_nosound) 
   {
   // output music
   if (NoSound == FALSE)
      {
      if (!NoSound0)
         AdjustSample (0, (32000 / (256-RAM[Sound0++])) * 16, 255);
      else
         AdjustSample (0, 1000, 0);
      if (!NoSound1)
         AdjustSample (1, (32000 / (256-RAM[Sound1++])) * 16, 255);
      else
         AdjustSample (1, 1000, 0);
      }
   else
      {
      AdjustSample (0, 1000, 0);
      AdjustSample (1, 1000, 0);
      }
   AUpdateAudio();

   // check for overflow
   if (Sound0 > Sound0Base + 255)
      Sound0 = Sound0Base;

   if (Sound1 > Sound1Base + 255)
      Sound1 = Sound1Base;
   }

   LoopCounter++; 
}

//-- out to sound port 0
int  SoundOut0(register WORD A, register BYTE B)
{
#ifdef SOUNDTEST
   if (sound_test)
      {
      fprintf (fptr, "%X: %02X - ", A, B);
      hexdump();
      }
#endif

   if (!sw_nosound)                                                
      {
      Sound0Base = ((B & 0x07) << 8) + cga->Sound1;
      if (Sound0Base != oSound0Base)
         {
         oSound0Base = Sound0Base;
         Sound0 = Sound0Base;
         }

      if (B & 0x20 && !(LastPort1 & 0x20))
         {
         APrimeVoice(hVoice[4], lpWave[4]);
         ASetVoiceVolume(hVoice[4], 64);
         AStartVoice(hVoice[4]);            
         }
      else if (!(B & 0x20) && LastPort1 & 0x20)
         AStopVoice(hVoice[4]);

      if (B & 0x40 && !(LastPort1 & 0x40))
         {
         APrimeVoice(hVoice[2], lpWave[2]);
         ASetVoiceVolume(hVoice[2], 64);
         AStartVoice(hVoice[2]);            
         }
      else if (!(B & 0x20) && LastPort1 & 0x20)
         AStopVoice(hVoice[4]);

      if (B & 0x80 && !(LastPort1 & 0x80))
         {
         APrimeVoice(hVoice[3], lpWave[3]);
         ASetVoiceVolume(hVoice[3], 64);
         AStartVoice(hVoice[3]);            
         }

      if (B & 0x10)
         {
         NoSound0 = FALSE;
         }
      else
         {
         NoSound0 = TRUE;
         }


      if (B & 0x08)
         {
         NoSound = FALSE;
         }
      }

   LastPort1 = B;
   RAM[cga->Port1] = B;
   return 0;
}

//-- out to sound port 1
int  SoundOut1(register WORD A, register BYTE B)
{
#ifdef SOUNDTEST
   if (sound_test)
      {
      fprintf (fptr, "%X: %02X - ", A, B);
      hexdump();
      }
#endif

   if (!sw_nosound)                                                
      {
      Sound1Base = ((B & 0x07) << 8) + cga->Sound2;
      if (Sound1Base != oSound1Base)
         {
         oSound1Base = Sound1Base;
         Sound1 = Sound1Base;
         }

      if (B & 0x08)
         {
         NoSound1 = FALSE;
         }
      }

   LastPort2 = B;
   RAM[cga->Port2] = B;
   return 0;
}

//-- out to sound port 2 ??
int  SoundOut2(register WORD A, register BYTE B)
{
#ifdef SOUNDTEST
   if (sound_test)
      {
      fprintf (fptr, "%X: %02X - ", A, B);
      hexdump();
      }
#endif

   if (!sw_nosound)
      {
      }

  	LastPort3 = B;
   RAM[cga->Port3] = B;
   return 0;
}

//-- out to sound port 3 ??
int  ColorOut(register WORD A, register BYTE B)
{
 	LastPort4 = B;
   RAM[cga->Port4] = B;
   return 0;
}

//-- out to speech port ??
int  SpeechOut(register WORD A, register BYTE B)
{
#ifdef SOUNDTEST
   if (sound_test)
      {
//      fprintf (fptr, "%X: %02X - ", A, B);
//      hexdump();
      }
#endif

   if (!sw_nosound)
      {
	   //-- out to speech port
      }

   RAM[cga->Speech] = B;
   return 0;
}

int  BackgroundUpdate(register WORD A, register BYTE B)
{
   if (RAM[A] != B)
    	Background[A-0x800] = 1;
   else
    	Background[A-0x800] = 0; 
   RAM[A] = B;
   return 0;
}

int  ForegroundUpdate(register WORD A, register BYTE B)
{
   if (RAM[A] != B)
    	Foreground[A-0x400] = 1;
   else
    	Foreground[A-0x400] = 0;
   RAM[A] = B;
   return 0;
}

//-- attempting to read coins, start port
WORD ReadCoin(register WORD A)
{
   BYTE valCoin = 0;

   if (key[KEY_1])
      {
      //-- start 1 player game
      valCoin |= 0x80;
      if (!score_loaded)
         {
         score_loaded = 1;
         }
      }
   else if (key[KEY_2])
      {
      //-- start 2 player game
      valCoin |= 0x40;
      if (!score_loaded)
         {
         score_loaded = 1;
         }
      }  
      	
   return (WORD)valCoin;
}

//-- attempting to read player 1 controls
WORD ReadPlayer1(register WORD A)
{
   BYTE valP1 = 0;
   if (key[KEY_UP])
      valP1 |= 0x20;  // player 1 move up
      
   if (key[KEY_DOWN])
      valP1 |= 0x10;  // player 1 move down
      
   if (key[KEY_LEFT])
      valP1 |= 0x80;  // player 1 move left
      
   if (key[KEY_RIGHT])
      valP1 |= 0x40;  // player 1 move right
      
   if (key[KEY_W])
      valP1 |= 0x02;  // player 1 shoot up
      
   if ((sw_useX && key[KEY_X]) || (!sw_useX && key[KEY_S]))
      valP1 |= 0x01;  // player 1 shoot down
      
   if (key[KEY_A])
      valP1 |= 0x08;  // player 1 shoot left

   if (key[KEY_D])
      valP1 |= 0x04;  // player 1 shoot right

   if (!sw_nojoystick)
      {
      if (joy_up)
         valP1 |= 0x20;  // player 1 move up
      
      if (joy_down)
         valP1 |= 0x10;  // player 1 move down
      
      if (joy_left)
         valP1 |= 0x80;  // player 1 move left
      
      if (joy_right)
         valP1 |= 0x40;  // player 1 move right

      if (!sw_gravis && joy_next_shot)
         {
         valP1 |= joy_next_shot;
         joy_next_shot = 0;
         }
      else
         {
         if (joy_b1)
            {
            valP1 |= 0x08;  // player 1 shoot left
            if (!sw_gravis)
               joy_next_shot = 0x04;   // shoot right next
            }
         if (joy_b2)
            {
            if (!sw_gravis)
               {
               valP1 |= 0x02;  // player 1 shoot up and down
               joy_next_shot = 0x01;
               }
            else
               valP1 |= 0x02;  // player 1 shoot up
            }
         if (sw_gravis && joy_b3)
            valP1 |= 0x01;  // player 1 shoot down
         if (sw_gravis && joy_b4)
            valP1 |= 0x04;  // player 1 shoot right
         }
      }      

   return (WORD)valP1;
}
              
//-- attempting to read player 2 controls
WORD ReadPlayer2(register WORD A)
{
   BYTE valP2 = 0;
   if (key[KEY_UP])
      valP2 |= 0x20;  // player 2 move up
      
   if (key[KEY_DOWN])
      valP2 |= 0x10;  // player 2 move down
      
   if (key[KEY_LEFT])
      valP2 |= 0x80;  // player 2 move left
      
   if (key[KEY_RIGHT])
      valP2 |= 0x40;  // player 2 move right
      
   if (key[KEY_W])
      valP2 |= 0x02;  // player 2 shoot up
      
   if ((sw_useX && key[KEY_X]) || (!sw_useX && key[KEY_S]))
      valP2 |= 0x01;  // player 2 shoot down
      
   if (key[KEY_A])
      valP2 |= 0x08;  // player 2 shoot left
      
   if (key[KEY_D])
      valP2 |= 0x04;  // player 2 shoot right

   if (!sw_nojoystick)
      {
      if (joy_up)
         valP2 |= 0x20;  // player 2 move up
      
      if (joy_down)
         valP2 |= 0x10;  // player 2 move down
      
      if (joy_left)
         valP2 |= 0x80;  // player 2 move left
      
      if (joy_right)
         valP2 |= 0x40;  // player 2 move right
      
      if (!sw_gravis && joy_next_shot)
         {
         valP2 |= joy_next_shot;
         joy_next_shot = 0;
         }
      else
         {
         if (joy_b1)
            {
            valP2 |= 0x08;  // player 2 shoot left 
            if (!sw_gravis)
               joy_next_shot = 0x04;   // next shot right
            }
         
         if (joy_b2)
            {
            if (!sw_gravis)
               {
               valP2 |= 0x02;  // player 2 shoot up and down
               joy_next_shot = 0x01;
               }
            else
               valP2 |= 0x02;  // player 2 shoot up
            }
         if (sw_gravis && joy_b3)
            valP2 |= 0x01;  // player 2 shoot down
         if (sw_gravis && joy_b4)
            valP2 |= 0x04;  // player 2 shoot right
         }
      }      

   return (WORD)valP2;
}

void DumpRom(char *nm, BYTE *bp, WORD len)
{
   FILE *f5;
   f5 = fopen (nm, "wb");
   fwrite (bp, len, 1, f5);
   fclose (f5);
}

//-- masked interrupt --- most of the emulation takes place here
BYTE Interrupt(void)
{
   char *start, *ptr;
   unsigned char c;
   int i, j; 
   int rtn = INT_IRQ;
   
   //-- always clear the background
   clear_to_color(piccie, bPal[0][LastPort4 & 7]);

   // display characters, accounting for rotation
   start = RAM + 0x800 + 31 * 32;
   for (i=0; i<32; i++)
      {
   	ptr = start;  
      for (j=0; j<32; j++)
         {
         c = *ptr;
         if (c != 48)
            {
            DrawCharacter (c, j<<3, i<<3, (BYTE)*(ptr+0x400));
            }
   	   ptr -= 32;
  	      }
  	   start++;
     	}

   //-- display the forground characters as sprites
   start = RAM + 0x400 + 31 * 32;
   for (i=0; i<32; i++)
      {
     	ptr = start;  
      for (j=0; j<32; j++)
         {
         c = *ptr;
         if (c != 48)
 		      {
            //-- updated sprites are read from 0x1000-0x1fff
            //-- sprite color palette is from 0xc000-0xcfff
            DrawSprite (c, j<<3, i<<3, (BYTE)*(ptr+0x800)); 
 		      }

         ptr -= 32;
   	   }
	   start++;
   	}

   //-- check for user exit
   if (key[KEY_ESC])
      rtn = INT_QUIT;

   //-- check for pause
   if (key[KEY_P])
      {
      pause = TRUE;

      //-- TODO: if sound, need to turn volume off
      if (!sw_nosound)
         {
         }

      //-- display a pause message
      PrintStr("HIT P TO CONTINUE", 128, 0x50);
      }
   //-- check for coin 1 - requires nmi
   else if (key[KEY_3])
      {
      RAM[0x3107] |= 0x01;
      clear_keybuf();
      rtn = INT_NMI;
      }
   //-- check for coin 2 - requires nmi
   else if (key[KEY_4])
      {
      RAM[0x3107] |= 0x02;
      clear_keybuf();
      rtn = INT_NMI;
      }   
   //-- screen dump
   else if (key[KEY_F2])
      {
      BITMAP *bmp;
      PALLETE pal;
      char file[80];
      sprintf (file, "%s\\game.pcx", cga->Path);
      get_pallete(pal);
      bmp = create_sub_bitmap(screen, 0,0, 224,256);
      save_pcx(file, bmp, pal);
      destroy_bitmap(bmp);
      }
   else if (key[KEY_F3])
      {
      SaveGame();
      }
   else if (key[KEY_F4])
      {
      LoadGame();
      }
   else if (key[KEY_F5])
      {
      DumpRom ("rom.b", RAM, 0x400);
      }
   else if (key[KEY_F6])
      {
      DumpRom ("rom.a", RAM, 0x400);
      }
   //-- sound test
   else if (key[KEY_F12])
      {
      sound_test ^= 1;
      }
   //-- check the joystick
   if (!sw_nojoystick)
      {
      poll_joystick();	
      }

#ifdef DUMPDATA
   PrintVal (0x2f, 248, 0x20);
   PrintVal (0x52, 248, 0x40);
   PrintVal (0x9c, 248, 0x60);
   PrintVal (0xb0, 248, 0x80);
   PrintVal (0x1f4, 248, 0xa0);
   PrintVal (0x1f4, 248, 0xc0);
#endif

#ifdef SOUNDTEST
   if (sound_test)
      PrintStr("ON", 248, 0x20);
   else
      PrintStr("OFF", 248, 0x20);
#endif

   // blast the video work buffer to screen
   if (sw_noxblit)
      {
      if (sw_v360)
         blit(piccie, screen, 32,0, 68,84, 224,256);
      else
         blit(piccie, screen, 32,0, 0,0, 224,256);
      }
   else      
      {  
      if (sw_v360)
         {
         if (ActivePage)
            xb(piccie, screen, 32,0, 68,84, 224,256, 0);
         else
            xb(piccie, screen, 32,0, 68,444, 224,256, 360);
         ActivePage ^= 1;
         }
      else
         {
         if (ActivePage)
            xb(piccie, screen, 32,0, 0,0, 224,256, 0);
         else
            xb(piccie, screen, 32,0, 0,256, 224,256, 256);
         ActivePage ^= 1;
         }
      }

   if (pause)
      {
      clear_keybuf();
      while (!key[KEY_P])
         continue;
      clear_keybuf();

      //-- TODO: if sound, turn on
      if (!sw_nosound)
         {
         }

      pause = FALSE;
      }

   return rtn;
}

//-- draw rotated character at specified position
void DrawCharacter (int c, int col, int row, BYTE color)
{
	int x, y;
   int sr, sc;
   int sb1 = 0;
   int sb2 = 0;
   BYTE result = 0;
   BYTE res1, res2;
   BYTE mask;
   int hpos;
   int vpos;
   int ROM1, ROM2;

   //-- background character color palette selected using bits 4,5,6
   BYTE cPal = (color & 0x38) >> 3;

   ROM1 = cga->BChar1;
   ROM2 = cga->BChar2;

   if (cga->GameID == FANTASY)
      {
//      ROM1 = (RAM[0x4D] ? cga->BChar1 - 0x800 : cga->BChar1);
//      ROM2 = (RAM[0x4D] ? cga->BChar2 - 0x800 : cga->BChar2);
      }

   // get the offset to the correct bit for each bitplane
   sb1 = c * 64;

   // for each bit in character
	sc  = 7;
   for (y = 0; y < 8; y ++)
      {
      // get a BYTE at a time
	   sr = 0;
      res1 = RAM[(sb1>>3)+ROM1];
      res2 = RAM[(sb1>>3)+ROM2];

      //-- adjust horizontal scrolling
      hpos = sc+col+RAM[cga->HScroll];
      if (hpos >= 256) hpos -= 256;
      else if (hpos < 0) hpos += 256;

      // for each bit in BYTE
      for (x=0; x<8; x++)
         {
         //-- adjust vertical scrolling
         vpos = sr+row-RAM[cga->VScroll];
         if (vpos >= 256) vpos -= 256;
         else if (vpos < 0) vpos += 256;

         // find the color by adding the bitplane values
         mask = 0x80 >> x; 
         if ((res1 & mask) == mask) result = 1;
         if ((res2 & mask) == mask) result ^= 2;
         if (result)
            {
            if (sw_noxblit)
               piccie->line[vpos][hpos] = bPal[cPal][result];
            else
               // the custom blit expects to find data packed by plane per line
               piccie->line[vpos][(hpos>>2)+((hpos%4)<<6)] = bPal[cPal][result];
            }
         sr++;
         sb1++;
         sb2++;
         result = 0;
         }

	   sc--;
      }
}

//-- draw rotated sprite at specified position
void DrawSprite (int c, int col, int row, BYTE color)
{
	int x, y;
   int sr, sc;
   int sb1 = 0;
   int sb2 = 0;
   BYTE result = 0;
   BYTE res1, res2;
   BYTE mask;

   //-- sprite color palette selected using bits 0,1,2
   BYTE cPal = color & 0x07;

   // get the offset to the correct bit for each bitplane
   sb1 = c * 64;

   // for each bit in sprite
	sc  = 7;
   for (y = 0; y < 8; y ++)
      {
      // get a BYTE at a time
	   sr = 0;
      res1 = RAM[(sb1>>3)+0x1000];
      res2 = RAM[(sb1>>3)+0x1800];

      // for each bit in BYTE
      for (x=0; x<8; x++)
         {
         // find the color by adding the bitplane values
         mask = 0x80 >> x; 
         if ((res1 & mask) == mask) result = 1;
         if ((res2 & mask) == mask) result ^= 2;
         if (result)
            {
            if (sw_noxblit)
               piccie->line[sr+row][sc+col] = fPal[cPal][result];
            else
               // the custom blit expects to find data packed by plane per line
               piccie->line[sr+row][((sc+col)>>2)+(((sc+col)%4)<<6)] = fPal[cPal][result];
            }
         sr++;
         sb1++;
         sb2++;
         result = 0;
         }

	   sc--;
      }
}

//-- play a sample
void PlaySample (int channel, BYTE *data, int len, int freq, int vol, int loop)
{
   //-- sanity check
	if (channel >= NUMVOICES)
      return;

   memcpy(lpWave[channel]->lpData, data, len);
   lpWave[channel]->wFormat = AUDIO_FORMAT_8BITS | AUDIO_FORMAT_MONO;
   if (loop)
      lpWave[channel]->wFormat |= AUDIO_FORMAT_LOOP;
   lpWave[channel]->dwLoopStart = 0;
   lpWave[channel]->dwLoopEnd = len;
   lpWave[channel]->dwLength = len;

   AWriteAudioData(lpWave[channel], 0, len);
   APlayVoice(hVoice[channel], lpWave[channel]);
   ASetVoiceFrequency(hVoice[channel], freq);
   ASetVoiceVolume(hVoice[channel], vol>>2);
}

void AdjustSample (int channel, int freq, int vol)
{
   //-- sanity check
	if (channel >= NUMVOICES)
      return;

   ASetVoiceFrequency(hVoice[channel], freq);
   ASetVoiceVolume(hVoice[channel], vol>>2);
}

// capture and prerotate all the background characters to a bitmap
void DisplayBackgroundChars(void)
{
   int x,y,cx,cy;
   int sr, sc;
   int sb1 = 0;
   int sb2 = 0;

   BYTE result = 0;

   for (cy = 0; cy < 8; cy ++)
   {
      for (cx = 0; cx < 32; cx ++)
      {
	   sc  = 7;
         for (y = 0; y < 8; y ++)
         {
	      sr = 0;
            for (x = 0; x < 8; x ++)
            {
               if ( GetBit(sb1, cga->BChar1) )
                 result = 1; 
               if ( GetBit(sb2, cga->BChar2) )
                 result ^= 2;

               //-- get the correct palette color
               bchars->line[cy*8 + sr][cx*8 + sc] = result;

               result = 0;
               sb1++;
               sb2++;
		         sr++;
            }
	   sc--;
         }
      }
   }
}

// capture and prerotate all the foreground characters to a bitmap
void DisplayForegroundChars(void)
{
   int x,y,cx,cy;
   int sr, sc;
   int sb1 = 0;
   int sb2 = 16384;

   BYTE result = 0;
 
   for (cy = 0; cy < 8; cy ++)
   {
      for (cx = 0; cx < 32; cx ++)
      {
	      sc  = 7;
         for (y = 0; y < 8; y ++)
         {
	         sr = 0;
            for (x = 0; x < 8; x ++)
            {
               if ( GetBit(sb1, 0x8000) )
                 result = 1;
               if ( GetBit(sb2, 0x8000) )
                 result ^= 2;
               
               fchars->line[cy*8 + sr][cx*8 + sc] = result; 

               result = 0;
               sb1++;
               sb2++;
		         sr++;
            }
	      sc--;            
         }
      }
   }

   DisplayBackgroundChars();

   // display them
   if (sw_dumpgraphics)
      {
      blit(fchars, screen, 0,0, 0,72, 256,64);
      blit(bchars, screen, 0,0, 0,0, 256,64);
      readkey();
      }
}


// return specific bit from memory location
WORD GetBit(int  startbit, unsigned int off)
{
   WORD nBYTE = startbit / 8 + off;
   WORD nbit = startbit%8;
          
   switch (nbit)
   {
      case 0:
         return ((RAM[nBYTE] & 0x80) == 0x80);
         break;

      case 1:
         return ((RAM[nBYTE] & 0x40) == 0x40);
         break;

      case 2:
         return ((RAM[nBYTE] & 0x20) == 0x20);
         break;

      case 3:
         return ((RAM[nBYTE] & 0x10) == 0x10);
         break;

      case 4:
         return ((RAM[nBYTE] & 0x08) == 0x08);
         break;

      case 5:
         return ((RAM[nBYTE] & 0x04) == 0x04);
         break;

      case 6:
         return ((RAM[nBYTE] & 0x02) == 0x02);
         break;

      case 7:
         return ((RAM[nBYTE] & 0x01) == 0x01);
         break;

   }

   return 0;
}

// blit data by plane per line
void cBlit (int x, int y, int col, int row, int w, int h)
{
   int i, j;
   for (i=0; i<h; i++)
      for (j=0; j<w; j++)
         piccie->line[row+i][((col+j)>>2)+(((col+j)%4)<<6)] = fchars->line[y+i][x+j];
}

// display a value from a memory location
void PrintVal(int add, int row, int col)
{
   char buf[20];
   char *ptr;
   int val;

   val = RAM[add];
   sprintf (buf, "%u", val);
   ptr = buf;

   while (*ptr)
      {
      if (sw_noxblit)
      	blit (fchars, piccie, (*ptr-'0')*8,0, col,row, 8,8);
      else
      	cBlit ((*ptr-'0')*8,0, col,row, 8,8);
      col += 8;
      ptr++;
   	}
}

// display a number
void PrintNum(int num, int row, int col)
{
   char buf[20];
   char *ptr;

   sprintf (buf, "%u", num);
   ptr = buf;

   while (*ptr)
      {
      if (sw_noxblit)
   	   blit (fchars, piccie, (*ptr-'0')*8,0, col,row, 8,8);
      else
	      cBlit ((*ptr-'0')*8,0, col,row, 8,8);
      col += 8;
      ptr++;
   	}
}

// display a string
void PrintStr(char *str, int row, int col)
{
   int x, y;
   BYTE b;
   while (*str)
      {
      b = *str;
      if (b >= '0' && b <= '9')
         {
         y = 0;
         x = (b-'0')*8;
         }
      else if ((b >= 'A' && b <= 'Z') || (b >= 'a' && b <= 'z'))
         {
         b &= 0xdf;
         y = 0;
         x = (b-'A'+10)*8;
         }
      else if (b == '.')
         {
         y = 8;
         x = 48;
         }
      else
         {
         y = 8;
         x = 128;
         }
      if (sw_noxblit)
      	blit (fchars, piccie, x, y, col,row, 8,8);
      else
      	cBlit (x, y, col,row, 8,8);
      col += 8;
      str++;
	   }
}







