#include <stdio.h>
#include <string.h>
#include "frogger.h"

Frogger * pFrogger;

BYTE froggerColors[] = 
{
   0x00, 0xC0, 0x07, 0xF6, 0x00, 0x17, 0x39, 0xF8,   
   0x00, 0x5E, 0xF6, 0x5C, 0x00, 0xC7, 0x39, 0x3F,   
   0x00, 0x3C, 0xF8, 0xD7, 0x00, 0x07, 0xF6, 0x39,   
   0x00, 0xC4, 0xC0, 0x07, 0x00, 0x07, 0x3F, 0xC4,   
   0x40, 0xC0, 0x07, 0xF6, 0x40, 0x17, 0x39, 0xF8,   
   0x40, 0x5E, 0xF6, 0x5C, 0x40, 0xC7, 0x39, 0x3F,   
   0x40, 0x3C, 0xF8, 0xD7, 0x40, 0x07, 0xF6, 0x39,   
   0x40, 0xC4, 0xC0, 0x07, 0x40, 0x07, 0x3F, 0xC4,   
};

ROMS froggerRoms[] = 
{
   { 0x0000,  "frogger.ic5", 4096, "code" },
   { 0x1000,  "frogger.ic6", 4096, "code" },
   { 0x2000,  "frogger.ic7", 4096, "code" },
   { 0x3000,  "frogger.ic8", 4096, "code" },
   { 0x7000,  "frogger.606", 2048, "graphics" },
   { 0x7800,  "frogger.607", 2048, "graphics" },
   { 0, NULL, 0, NULL }	
};

ROMS froggerAudioRoms[] = 
{
   { 0x0000,  "frogger.608", 2048, "sound" },
   { 0x0800,  "frogger.609", 2048, "sound" },
   { 0x1000,  "frogger.610", 2048, "sound" },
   { 0, NULL, 0, NULL }	
};

MemoryWrite FroggerWrite[] = 
{
   { 0xA800, 0xABFF, FroggerOut },
   { 0xB000, 0xB03F, FroggerOut },
   { 0xD000, 0xD000, FroggerSoundCommand },
   { 0xFFFF, 0xFFFF, NULL }
};

MemoryRead FroggerRead[] = 
{
   { 0x83F1, 0x83FA, FroggerReadScores },
   { 0xE000, 0xE000, FroggerReadPlayer0 },
   { 0xE002, 0xE002, FroggerReadPlayer1 },
   { 0xE004, 0xE004, FroggerReadDipSwitch },
   { 0xFFFF, 0xFFFF, NULL }
};

MemoryWrite FroggerAudioWrite[] = 
{
   { 0xFFFF, 0xFFFF, NULL }
};

MemoryRead FroggerAudioRead[] = 
{
   { 0xFFFF, 0xFFFF, NULL }
};

Frogger::Frogger() : Game (froggerRoms, 0x10000, "frogger")
{
   pFrogger = this;
   pProcZ80 = new ProcZ80(2);
   pProcessor = (Processor *) pProcZ80;

   // setup audio hardware
   LoadAudio(froggerAudioRoms, 0x10000);

   pProcZ80->SetMemoryRW(FroggerWrite, FroggerRead, pRWTable, FroggerInProc,
      FroggerOutProc, 30000, pRAM, 1);
   pProcZ80->SetMemoryRW(FroggerAudioWrite, FroggerAudioRead, pAudioRWTable, 
     FroggerAudioInProc, FroggerAudioOutProc, 20000, pAudioRAM, 2);

   strcpy(Name, "frogger");
   strcpy(Descript, "Frogger");

   // input ports
   Player0 = new IOPort(0xE000, 0xFF);
   Player0->SetKey(KEY_3, KEY_4, KEY_LEFT, KEY_RIGHT, 0, 0, 0, 0);
   Player0->SetJoy(JOY_BUTTON2, 0, JOY_LEFT, JOY_RIGHT, 0, 0, 0, 0);
   Player1 = new DipSwitch(0xE002, 0xFF);
   Player1->SetKey(KEY_1, KEY_2, 0, 0, 0, 0, 0, 0);
   Player1->SetJoy(JOY_BUTTON1, 0, 0, 0, 0, 0, 0, 0);
   Player1->SetMen(0, 2, "256 Frogs", "7 Frogs", "5 Frogs", "3 Frogs");
   Dipswitch = new DipSwitch(0xE004, 0xFF);
   Dipswitch->SetKey(0, KEY_DOWN, 0, KEY_UP, 0, 0, 0, 0);
   Dipswitch->SetJoy(0, JOY_DOWN, 0, JOY_UP, 0, 0, 0, 0);
   Dipswitch->SetCoins(1, 0, "1 Coin 1 Play", "2 Coins 1 Play", "2 Coins 1 Play", 
      "1 Coin 1 Play");

   pMenDSW = Player1;
   pCoinsDSW = Dipswitch;
   pDisplay = 0;
   pAYPSG = 0;
}

Frogger::~Frogger()
{
   delete Player0;
   delete Player1;
   delete Dipswitch;
   delete pProcZ80;   
   deleteAYPSG(&pAYPSG);
   deleteDisplay(&pDisplay);
}   

void Frogger::Reset()
{
   // reset memory
   Game::Reset();

   // setup the display
   pDisplay = new Display(256, 256, 256, 256, 0, 0, 0, 0, scr256x256, 200);
   pDisplay->SetClip(16, 0, 240, 256);
   pColors = froggerColors;
   BuildText(0x7800, 0x7888, 0x7880, 0x7880);
   memset(BackBuffer, 1, 0x400);

   // setup sound
   if (bSound)
      {
      pAYPSG = new AYPSG(1, 1000, 33);
      if (!pAYPSG->Status())
         {
         bSound = 0;
         }
      }

   // reset the processors
   pProcZ80->Reset(1);
   pProcZ80->Reset(2);
   pProcZ80->EnableInterrupts(1, OFF);
}

void Frogger::Execute()
{
   int rtn = INT_NMI;

   // main execution loop
   while (rtn != INT_QUIT)
      {
      // main processor
      pProcZ80->Exec(1);
      rtn = Interrupt();
      if (pRAM[0xB808])
         {
         pProcZ80->NMI();
         }


      if(rtn != INT_QUIT)
         {
         // audio processor
         pProcZ80->Exec(2);
         pProcZ80->NMI();
         }
      }

   // save the high score
   SaveScore();

   // close the display and audio
   deleteAYPSG(&pAYPSG);
   deleteDisplay(&pDisplay);
}

int Frogger::Interrupt()
{
   int rtn = INT_NMI;

   Video();
   if (bSound)
      Audio();

   if (Game::Interrupt() == INT_QUIT)
      rtn = INT_QUIT;

   pDisplay->Draw(); 

   return rtn;
}

void Frogger::Video()
{
   unsigned char *start, *ptr;
   int soff, scur;
   unsigned char c;
   int i, j; 

   // display characters, accounting for rotation
   start = pRAM + 0xA800 + 31 * 32;
   soff = 31 * 32;
   for (i=0; i<32; i++)
      {
   	ptr = start;
   	scur = soff;
      for (j=0; j<32; j++)
         {
         c = *ptr;
         if (BackBuffer[scur])
            {
            DrawCharacter (c, j<<3, i<<3, pRAM[0xB000+(i<<1)+1] + (i<17?8:0));
            BackBuffer[scur] = 0;
            }
   	   ptr -= 32;
         scur -= 32;
  	      }
  	   start++;
      soff++;
     	}

	for (i=0; i<256; i+=8)
		{
		int scroll = pRAM[0xB000 + (i>>2)];
		scroll = ((scroll << 4) & 0xf0) | ((scroll >> 4) & 0x0f);

      blit(pDisplay->Background(), pDisplay->Picture(), 
         0, i, scroll, i, 256, 8);
      if (scroll)
         blit(pDisplay->Background(), pDisplay->Picture(), 
            0, i, scroll-256, i, 256, 8);
      }

   for (i=7*4; i>=0; i-=4)
     	{
   	if (pRAM[0xB040+i+3] != 0)
	   	{
         DrawSprite(0xB040+i);
   		}
	   }
}

void Frogger::Audio()                     
{
   pAYPSG->Update();
}

void Frogger::Load()                     
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\%s.sav", Path, Name);
   fp = fopen (file, "rb");
   if (fp)
      {
      fread(pRAM+0x8000, 0x4000, 1, fp);
      fread(pAudioRAM+0x4000, 0x0400, 1, fp);
      pProcZ80->Load(fp);
      fclose(fp);
      memset(BackBuffer, 1, 0x400);
      }
}

void Frogger::Save()                     
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\%s.sav", Path, Name);
   fp = fopen (file, "wb");
   if (fp)
      {
      fwrite(pRAM+0x8000, 0x4000, 1, fp);
      fwrite(pAudioRAM+0x4000, 0x0400, 1, fp);
      pProcZ80->Save(fp);
      fclose(fp);
      }
}

void Frogger::LoadScore()
{
   if (!ScoreLoaded && memcmp (pRAM + 0x83F1, "\x63\x04", 2) == 0 &&
      memcmp (pRAM + 0x83F9, "\x27\x01", 2) == 0)
      {
      ReadScore(0x83F1, 0x0A);
      memcpy (pRAM + 0x83EF, pRAM + 0x83F1, 2);
      ScoreLoaded = 1;
      }
}

void Frogger::SaveScore()
{
   WriteScore(0x83F1, 0x0A);
}

void Frogger::DrawCharacter (int c, int col, int row, BYTE color)
{
	int x, y;
   int sr, sc;
   int sb1 = 0;
   BYTE result = 0;
   BYTE res1, res2;
   BYTE mask;
   int ROM1 = 0x7000;
   int ROM2 = 0x7800;

   // get the offset to the correct bit for each bitplane
   sb1 = c * 64;

   // for each bit in character
	sc  = 7;
   for (y=0; y<8; y++)
      {
      // get a BYTE at a time
	   sr = 0;
      res1 = pRAM[(sb1>>3)+ROM1];
      res2 = pRAM[(sb1>>3)+ROM2];

      // for each bit in BYTE
      for (x=0; x<8; x++)
         {
         // find the color by adding the bitplane values
         mask = 0x80 >> x; 
         if ((res1 & mask) == mask) result = 2;
         if ((res2 & mask) == mask) result |= 1;
         pDisplay->Background()->line[sr+row][sc+col] = pColors[(color<<2)+result];
         sr++;
         sb1++;
         result = 0;
         }

	   sc--;
      }
}
                         
void Frogger::DrawSprite (WORD off)
{
   int col = ((pRAM[off] << 4) & 0xf0) | ((pRAM[off] >> 4) & 0x0f);
   int row = pRAM[off+3];
   int c = (pRAM[off+1] & 0x3F)<<2;
   int color = pRAM[off+2];
   int fx = pRAM[off+1] & 0x80;
   int fy = pRAM[off+1] & 0x40;
	int x, y;
   int sr, sc;
   int sb1 = 0;
   BYTE result = 0;
   BYTE res1, res2, res3, res4, res5, res6, res7, res8;
   BYTE mask;
   int ROM1 = 0x7000;
   int ROM2 = 0x7800;
   int hpos;
   int vpos;

   // get the offset to the correct bit for each bitplane
   sb1 = c * 64;

   // for each bit in character
	sc = (fx?0:7);
   for (y = 0; y < 8; y ++)
      {
      //-- adjust horizontal scrolling
      hpos = sc+col;

      // get a BYTE at a time
	   sr = (fy?7:0);

	   res1 = pRAM[(sb1>>3)+ROM1+(!fy?(fx?0:16):(fx?8:24))];
      res2 = pRAM[(sb1>>3)+ROM2+(!fy?(fx?0:16):(fx?8:24))];
      res3 = pRAM[(sb1>>3)+ROM1+(!fy?(fx?8:24):(fx?0:16))];
      res4 = pRAM[(sb1>>3)+ROM2+(!fy?(fx?8:24):(fx?0:16))];
	   res5 = pRAM[(sb1>>3)+ROM1+(!fy?(fx?16:0):(fx?24:8))];
      res6 = pRAM[(sb1>>3)+ROM2+(!fy?(fx?16:0):(fx?24:8))];
      res7 = pRAM[(sb1>>3)+ROM1+(!fy?(fx?24:8):(fx?16:0))];
      res8 = pRAM[(sb1>>3)+ROM2+(!fy?(fx?24:8):(fx?16:0))];

      // for each bit in BYTE
      for (x=0; x<8; x++)
         {
         //-- adjust vertical scrolling
         vpos = sr+row;

         // find the color by adding the bitplane values
         mask = 0x80 >> x; 
         result = 0;
         if ((res1 & mask) == mask) result = 2;
         if ((res2 & mask) == mask) result |= 1;
         if (result && hpos > 15 && hpos < 240)
            pDisplay->Picture()->line[vpos][hpos] = pColors[(color<<2)+result];
         result = 0;
         if ((res3 & mask) == mask) result = 2;
         if ((res4 & mask) == mask) result |= 1;
         if (result && hpos > 15 && hpos < 240)
            pDisplay->Picture()->line[vpos+8][hpos] = pColors[(color<<2)+result];
         result = 0;
         if ((res5 & mask) == mask) result = 2;
         if ((res6 & mask) == mask) result |= 1;
         if (result && hpos > 7 && hpos < 232)
            pDisplay->Picture()->line[vpos][hpos+8] = pColors[(color<<2)+result];
         result = 0;
         if ((res7 & mask) == mask) result = 2;
         if ((res8 & mask) == mask) result |= 1;
         if (result && hpos > 7 && hpos < 232)
            pDisplay->Picture()->line[vpos+8][hpos+8] = pColors[(color<<2)+result];
         if(fy) sr--; else sr++;
         sb1++;
         }

	   if(fx) sc++; else sc--;
      }
}
                         
BYTE FroggerReadPlayer0(WORD A)
{
   return pFrogger->Player0->Get();
}

BYTE FroggerReadPlayer1(WORD A)
{
   return pFrogger->Player1->Get();
}

BYTE FroggerReadDipSwitch(WORD A)
{
   return pFrogger->Dipswitch->Get();
}

BYTE FroggerTest(WORD A)
{
   return pGame->RAM()[A];
}

BYTE FroggerReadScores(WORD A)
{
   pFrogger->LoadScore();
   return pGame->RAM()[A];
}

int FroggerOut(WORD A,  BYTE B)
{
   if (A >= 0xA800 && A < 0xAC00 && pGame->RAM()[A] != B)
      pFrogger->BackBuffer[A-0xA800] = 1;
	else if ((A & 1) && pGame->RAM()[A] != B)
		for (int i=(A-0xB000)>>1; i<0x400; i+=0x20)
         pFrogger->BackBuffer[i] = 1;
   pGame->RAM()[A] = B;
   return 0;
}

int FroggerIntEnable(WORD A,  BYTE B)
{
   pFrogger->pProcZ80->EnableInterrupts(1, B);
   return 0;   
}

int FroggerSoundCommand(WORD A,  BYTE B)
{
   return 0;   
}

BYTE FroggerInProc(WORD port)
{
   return 0;
}

void FroggerOutProc(BYTE data, BYTE port)
{
}

BYTE FroggerAudioInProc(WORD port)
{
   return 0;
}

void FroggerAudioOutProc(BYTE data, BYTE port)
{
}




