#include <stdio.h>
#include <string.h>
#include "galaxian.h"

Galax * pGalax;

BYTE starColors[] = 
{   
   0x00,0xC0,0x38,0x07,0xF8,0x3F,0xC7,0xFF
};

Galax::Galax(ROMS * roms, char *dir) : Game (roms, 0x10000, dir)
{
   pGalax = this;
   pProcZ80 = new ProcZ80(1);
   pProcessor = (Processor *) pProcZ80;

   Player0 = new IOPort(0x6000, 0x00);
   Player1 = new DipSwitch(0x6800, 0x00);
   Player1->SetKey(0, 0, 0, 0, 0, 0, KEY_2, KEY_1);
   Player1->SetJoy(0, 0, 0, 0, 0, 0, 0, JOY_BUTTON2);
   Dipswitch = new DipSwitch(0x7000, 0x00);

   pDisplay = 0;
   pAudioDev = 0;
}

Galax::~Galax()
{
   delete Player0;
   delete Player1;
   delete Dipswitch;
   delete pProcZ80;   
   deleteAudioDev(&pAudioDev);
   deleteDisplay(&pDisplay);
}   

void Galax::Reset()
{
   // reset memory
   Game::Reset();

   // reset the processors
   pProcZ80->Reset(1);

   // setup the display
   pDisplay->SetClip(16, 0, 240, 256);
   memset(BackBuffer, 1, 0x400);

   SetupStars();

   // setup sound
   if (bSound)
      {
      // initialize sound
      pAudioDev = new AudioDev(4, 2);
      if (!pAudioDev->Status())
         {
         bSound = 0;
         }
      else
         {
         ExplosionWave = pAudioDev->LoadWaveFile("samples\\galexpld.wav");
         FireWave = pAudioDev->LoadWaveFile("samples\\galshot.wav");
         pAudioDev->PlaySample(0, SineWaveSample, 32, 1000, 0, 1);
         pAudioDev->PlayVoice(2);
         pAudioDev->PlayVoice(3);
         SoundBase = 0;
         }
      }
}

void Galax::Execute()
{
   int rtn = INT_NMI;

   // reset the timer
   if (bSound)
      aTimer.setHandler(GalaxAudioHandler, 33);

   // main execution loop
   while (rtn != INT_QUIT)
      {
      // main processor
      pProcZ80->Exec(1);
      rtn = Interrupt();
      if(pRAM[InterruptEnable])
         pProcZ80->NMI();
      }

   // save the high score
   SaveScore();

   // close the display and audio
   deleteAudioDev(&pAudioDev);
   deleteDisplay(&pDisplay);
}

int Galax::Interrupt()
{
   int rtn = INT_NMI;

   Video();

   if (Game::Interrupt() == INT_QUIT)
      rtn = INT_QUIT;

//   pDisplay->PrintNum(FrameCount, 248, 32);
   pDisplay->Draw(); 

   return rtn;
}

void Galax::Video()
{
   unsigned char *start, *ptr;
   int soff, scur;
   unsigned char c;
   int i, j; 

   // display characters, accounting for rotation
   start = pRAM + VideoAddress + 31 * 32;
   soff = 31 * 32;
   for (i=0; i<32; i++)
      {
   	ptr = start;
   	scur = soff;
      for (j=0; j<32; j++)
         {
         c = *ptr;
			if ((ROMSet & 4) && (c & 0xc0) == 0x80)
				c = (c & 0x3f) | (ROMSet << 6);
         c += (ROMSet2 << 8);
         if (BackBuffer[scur])
            {
            DrawCharacter (c, j<<3, i<<3, pRAM[AttributeAddress+(i<<1)+1]);
            BackBuffer[scur] = 0;
            }
   	   ptr -= 32;
         scur -= 32;
  	      }
  	   start++;
      soff++;
     	}
   
   // scroll the background
	for (i=0; i<256; i+=8)
		{
		int scroll = pRAM[AttributeAddress + (i>>2)];

      blit(pDisplay->Background(), pDisplay->Picture(), 
         0, i, scroll, i, 256, 8);
      if (scroll)
         blit(pDisplay->Background(), pDisplay->Picture(), 
            0, i, scroll-256, i, 256, 8);
      }
   
   // draw the bullets
	for (i=0; i<0x20; i += 4)
	   {
		int x, y;
      x = pRAM[BulletAddress+i+1];
		if (x >= 16 && x < 240)
   		{
			y = 256 - pRAM[BulletAddress+i+3] - 4;

			if (y >= 0)
				for (j=0; j<3; j++)
               pDisplay->Picture()->line[y+j][x] = (i==28?0x3F:0xFF);
			}
		}

   // draw stars
	if (bStars)
	   {
		for (i=0; i<StarCount; i++)
   		{
			int y = (StarY[i] + StarScroll/2) % 256;
			if (((StarX[i] & 1) ^ ((y >> 4) & 1)) &&
			   pDisplay->Picture()->line[y][StarX[i]] == 0)
				pDisplay->Picture()->line[y][StarX[i]] = StarColor[i];
		   }

		StarScroll++;
	   }
   
   // draw sprites
   for (i=0; i<0x20; i+=4)
     	{
   	if (pRAM[SpriteAddress+i+3] > 8)
	   	{
         DrawSprite(SpriteAddress+i);
   		}
	   }
}

void Galax::Audio()                     
{
   if (bSound)
      {
      if (bPause)
         {
         pAudioDev->AdjustSample (0, 1000, 0);
         pAudioDev->AdjustSample (1, 1000, 0);
         pAudioDev->AdjustSample (2, 1000, 0);
         pAudioDev->Update();
         }
      else
         {
      	if (SoundBase && SoundBase != 0xFF) 
	         pAudioDev->AdjustSample (0, (1536000/16) / (256-SoundBase) * 16, 255);
         else
            pAudioDev->AdjustSample (0, 1000, 0);
         pAudioDev->Update();
         }
      }
}

void Galax::Load()                     
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\%s.sav", Path, Name);
   fp = fopen (file, "rb");
   if (fp)
      {
      fread(pRAM+MemBase, MemSize, 1, fp);
      pProcZ80->Load(fp);
      fclose(fp);
      memset(BackBuffer, 1, 0x400);
      }
}

void Galax::Save()                     
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\%s.sav", Path, Name);
   fp = fopen (file, "wb");
   if (fp)
      {
      fwrite(pRAM+MemBase, MemSize, 1, fp);
      pProcZ80->Save(fp);
      fclose(fp);
      }
}

void Galax::LoadScore()
{
}

void Galax::SaveScore()
{
}

void Galax::DrawCharacter (int c, int col, int row, BYTE color)
{
	int x, y;
   int sr, sc;
   int sb1 = 0;
   BYTE result = 0;
   BYTE res1, res2;
   BYTE mask;
   int hpos, vpos;

   // get the offset to the correct bit for each bitplane
   sb1 = c * 64;

   // for each bit in character
	sc  = 7;
   for (y=0; y<8; y++)
      {
      // get a BYTE at a time
	   sr = 0;
      res1 = pRAM[(sb1>>3)+ROM1];
      res2 = pRAM[(sb1>>3)+ROM2];

      //-- adjust horizontal scrolling
      hpos = sc+col;

      // for each bit in BYTE
      for (x=0; x<8; x++)
         {
         //-- adjust vertical scrolling
         vpos = sr+row;

         // find the color by adding the bitplane values
         mask = 0x80 >> x; 
         if ((res1 & mask) == mask) result = 2;
         if ((res2 & mask) == mask) result |= 1;
//         pDisplay->Background()->line[vpos][hpos] = pColors[(color<<2)+result];
         pDisplay->SetPixel(vpos, hpos, pColors[(color<<2)+result], pDisplay->Background());
         sr++;
         sb1++;
         result = 0;
         }

	   sc--;
      }
}
                         
void Galax::DrawSprite (WORD off)
{
   int col = pRAM[off];
   int row = pRAM[off+3];
   int c = (pRAM[off+1] & 0x3F);
   int color = pRAM[off+2];
   int fx = pRAM[off+1] & 0x80;
   int fy = pRAM[off+1] & 0x40;
	int x, y;
   int sr, sc;
   int sb1 = 0;
   BYTE result = 0;
   BYTE res1, res2, res3, res4, res5, res6, res7, res8;
   BYTE mask;
   int hpos;
   int vpos;

	if ((ROMSet & 4) && (c & 0x30) == 0x20)
		c = (c & 0x0F) | (ROMSet << 4);
   c += (ROMSet2 << 6);
   c <<= 2;

   // get the offset to the correct bit for each bitplane
   sb1 = c * 64;

   // for each bit in character
	sc = (fx?0:7);
   for (y = 0; y < 8; y ++)
      {
      //-- adjust horizontal scrolling
      hpos = sc+col;

      // get a BYTE at a time
	   sr = (fy?7:0);

	   res1 = pRAM[(sb1>>3)+ROM1+(!fy?(fx?0:16):(fx?8:24))];
      res2 = pRAM[(sb1>>3)+ROM2+(!fy?(fx?0:16):(fx?8:24))];
      res3 = pRAM[(sb1>>3)+ROM1+(!fy?(fx?8:24):(fx?0:16))];
      res4 = pRAM[(sb1>>3)+ROM2+(!fy?(fx?8:24):(fx?0:16))];
	   res5 = pRAM[(sb1>>3)+ROM1+(!fy?(fx?16:0):(fx?24:8))];
      res6 = pRAM[(sb1>>3)+ROM2+(!fy?(fx?16:0):(fx?24:8))];
      res7 = pRAM[(sb1>>3)+ROM1+(!fy?(fx?24:8):(fx?16:0))];
      res8 = pRAM[(sb1>>3)+ROM2+(!fy?(fx?24:8):(fx?16:0))];

      // for each bit in BYTE
      for (x=0; x<8; x++)
         {
         //-- adjust vertical scrolling
         vpos = sr+row;

         // find the color by adding the bitplane values
         mask = 0x80 >> x; 
         result = 0;
         if ((res1 & mask) == mask) result = 2;
         if ((res2 & mask) == mask) result |= 1;
         if (result && hpos > 15 && hpos < 240 && vpos < 256)
            pDisplay->Picture()->line[vpos][hpos] = pColors[(color<<2)+result];
         result = 0;
         if ((res3 & mask) == mask) result = 2;
         if ((res4 & mask) == mask) result |= 1;
         if (result && hpos > 15 && hpos < 240 && vpos < 248)
            pDisplay->Picture()->line[vpos+8][hpos] = pColors[(color<<2)+result];
         result = 0;
         if ((res5 & mask) == mask) result = 2;
         if ((res6 & mask) == mask) result |= 1;
         if (result && hpos > 7 && hpos < 232 && vpos < 256)
            pDisplay->Picture()->line[vpos][hpos+8] = pColors[(color<<2)+result];
         result = 0;
         if ((res7 & mask) == mask) result = 2;
         if ((res8 & mask) == mask) result |= 1;
         if (result && hpos > 7 && hpos < 232 && vpos < 248)
            pDisplay->Picture()->line[vpos+8][hpos+8] = pColors[(color<<2)+result];
         if(fy) sr--; else sr++;
         sb1++;
         }
         
	   if(fx) sc++; else sc--;
      }
}

void Galax::SetupStars(void)
{
	int generator = 0;

	// precalculate the star background 
   bStars = 0;
   ROMSet = 0;
   ROMSet2 = 0;
	StarCount = 0;

	for (int i=255; i>=0; i--)
   	{
		for (int j=511; j>=0; j--)
		   {
			int bit1,bit2;

			generator <<= 1;
			bit1 = (~generator >> 17) & 1;
			bit2 = (generator >> 5) & 1;

			if (bit1 ^ bit2) generator |= 1;

			if (i >= 16 && i < 240 && ((~generator >> 16) & 1) && 
			   (generator & 0xff) == 0xff)
			   {
				int color;

				color = (~(generator >> 8)) & 0x07;
				if (color && StarCount < MAX_STARS)
				   {
					StarX[StarCount] = i;
					StarY[StarCount] = j;
					StarColor[StarCount++] = starColors[color];
   				}
			   }
		   }
	   }
}

BYTE GalaxReadPlayer0(WORD A)
{
   return pGalax->Player0->Get();
}

BYTE GalaxReadPlayer1(WORD A)
{
   return pGalax->Player1->Get();
}

BYTE GalaxReadDipSwitch(WORD A)
{
   return pGalax->Dipswitch->Get();
}

BYTE GalaxReadScores(WORD A)
{
   pGalax->LoadScore();
   return pGame->RAM()[A];
}

int GalaxOut(WORD A,  BYTE B)
{
   if (A >= pGalax->VideoAddress && A < pGalax->VideoAddress+0x400 && pGame->RAM()[A] != B)
      pGalax->BackBuffer[A-pGalax->VideoAddress] = 1;
	else if ((A & 1) && pGame->RAM()[A] != B)
		for (int i=(A-pGalax->AttributeAddress)>>1; i<0x400; i+=0x20)
         pGalax->BackBuffer[i] = 1;
   pGame->RAM()[A] = B;
   return 0;
}

int GalaxStarsEnable(WORD A,  BYTE B)
{
	pGalax->bStars = B & 1;
	pGalax->StarScroll = 0;
   pGame->RAM()[A] = B;
   return 0;
}

int GalaxSound(WORD A, BYTE B)
{
   pGalax->SoundBase = B;
   return 0;
}

int GalaxShoot(WORD A, BYTE B)
{
   if (pGalax->bSound && B & 1)
      {
      pGalax->pAudioDev->PrimeVoice(pGalax->FireWave);
      pGalax->pAudioDev->SetVoiceVolume(pGalax->FireWave, 64);
      pGalax->pAudioDev->StartVoice(pGalax->FireWave);            
      }

   return 0;
}

int GalaxExplode(WORD A, BYTE B)
{
   if (pGalax->bSound && B & 1)
      {
      pGalax->pAudioDev->PrimeVoice(pGalax->ExplosionWave);
      pGalax->pAudioDev->SetVoiceVolume(pGalax->ExplosionWave, 64);
      pGalax->pAudioDev->StartVoice(pGalax->ExplosionWave);            
      }

   return 0;
}

BYTE GalaxInProc(WORD port)
{
   return 0;
}

void GalaxOutProc(BYTE data, BYTE port)
{
}

void GalaxAudioHandler(...)
{
   pGalax->Audio();
}



