/*
** File: psg.cpp -- software implementation of AY-3-8910 Programmable
**		sound generator.
**
** Based on: Sound.c, (C) Ville Hallik (ville@physic.ut.ee) 1996
**
** SCC emulation removed.  Made modular and capable of multiple PSG
** emulation.
**
** Modifications (C) 1996 Michael Cuddy, Fen's Ende Software.
** http://www.fensende.com/Users/mcuddy
**
** Encapsulation Modifications (C) 1997 Brian Levine, BJ Software.
**
*/

#include <stdio.h>
#include <string.h>
#include "psg.h"

static unsigned char AYEnvForms[16][32] = {
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	  15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	   0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15 },
	{ 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0,
	  15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	   0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	  15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	  15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0 },
	{  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
	   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 }
};

AYPSG * pAYPSG;

AY8910::AY8910(AudioDev *pAudio, int voice, AYSAMPLE *buf, int bufSize, int rate)
{
   UserBuffer = 0;
   if (buf) 
      {
	   Buf = buf;
	   UserBuffer = 1;
      } 
   else 
      {
      Buf = new AYSAMPLE[bufSize];
      }

   AYBufSize = bufSize;
   AYSoundRate = rate;

   pAudioDev = pAudio;
   for (int i=0; i<3; i++)
      AudioVoices[i] = voice+i;

   Port[0] = Port[1] = NULL;
   Reset();
}

AY8910::~AY8910()
{
   if (Buf && !UserBuffer) 
      delete []Buf; 
   Buf = NULL;
}

void AY8910::Reset()
{
   int i;

   if (!UserBuffer)
      memset(Buf,'\0', AYBufSize);

   // initialize hardware registers 
   for( i=0; i<16; i++ ) 
      Regs[i] = AUDIO_CONV(0);

   Regs[AY_ENABLE] = 077;
   Regs[AY_AVOL] = 8;
   Regs[AY_BVOL] = 8;
   Regs[AY_CVOL] = 8;
   NoiseGen = 1;
   Envelope = 15;
   StateNoise = 0;
   Incr0 = Incr1 = Incr2 = Increnv = Incrnoise = 0;
}

// 'num' is the number of virtual AY8910's to allocate
// 'rate' is sampling rate and 'bufsiz' is the size of the
// buffer that should be updated at each interval
AYPSG::AYPSG(int num, int rate, int bufsiz, ... )
{
   int i;
   va_list ap;
   AYSAMPLE *userbuffer = NULL;
   int moreargs = 1;

   va_start(ap,bufsiz);

   AYNumChips = num;
   AYSoundRate = rate;
   AYBufSize = bufsiz;

   pAudioDev = new AudioDev(AYNumChips*3, AYNumChips*3);
   pAY8910s = new AY8910*[AYNumChips];
   for (i=0; i<AYNumChips; i++) 
      {
   	if (moreargs) userbuffer = va_arg(ap,AYSAMPLE*);
	   if (userbuffer == NULL) moreargs = 0;
      pAY8910s[i] = new AY8910(pAudioDev, i*3, userbuffer, bufsiz, rate);
      }
}

AYPSG::~AYPSG()
{
   for(int i=0; i<AYNumChips; i++)
      delete pAY8910s[i];
   delete []pAY8910s;
   delete pAudioDev;
}


// write a register on AY8910 chip number 'n' 
void AY8910::WriteReg(int r, int v)
{
   Regs[r] = v;
		
   switch( r ) 
      {
	   case AY_AVOL:
	   case AY_BVOL:
	   case AY_CVOL:
	      Regs[r] &= 0x1F;	// mask volume 
	      break;
   	case AY_EFINE:
	   case AY_ECOARSE:
	      Countenv=0;
	      // fall through 
	   case AY_ESHAPE:
	      Regs[AY_ESHAPE] &= 0xF;
	      break;
	   case AY_PORTA:
	      if (Port[0]) 
	         {
		      (Port[0])(this,AY_PORTA,1,v);
	         }
	      break;
	   case AY_PORTB:
	      if (Port[1]) 
	         {
		      (Port[1])(this,AY_PORTB,1,v);
	         }
	      break;
      }
}

BYTE AY8910::ReadReg(int r)
{
   switch (r) 
      {
	   case AY_PORTA:
	      if (Port[0]) 
	         {
		      (Port[0])(this,AY_PORTA,0,0);
	         }
	      break;
	   case AY_PORTB:
	      if (Port[1]) 
	         {
		      (Port[1])(this,AY_PORTB,0,0);
	         }
	      break;
      }

   return Regs[r];
}
	
void AY8910::Update()
{
   int i, x;
   int c0, c1, l0, l1, l2;
    
   x = (Regs[AY_AFINE]+((unsigned)(Regs[AY_ACOARSE]&0xF)<<8));
   Incr0 = x ? AY8910_CLOCK / AYSoundRate * 4 / x : 0;

   x = (Regs[AY_BFINE]+((unsigned)(Regs[AY_BCOARSE]&0xF)<<8));
   Incr1 = x ? AY8910_CLOCK / AYSoundRate * 4 / x : 0;

   x = (Regs[AY_CFINE]+((unsigned)(Regs[AY_CCOARSE]&0xF)<<8));
   Incr2 = x ? AY8910_CLOCK / AYSoundRate * 4 / x : 0;

   x = Regs[AY_NOISEPER]&0x1F;
   Incrnoise = AY8910_CLOCK / AYSoundRate * 4 / ( x ? x : 1 );

   x = (Regs[AY_EFINE]+((unsigned)Regs[AY_ECOARSE]<<8));
   Increnv = x ? AY8910_CLOCK / AYSoundRate * 4 / x * AYBufSize : 0;
	  
   Envelope = AYEnvForms[Regs[AY_ESHAPE]][(Countenv>>16)&0x1F];

   if((Countenv += Increnv) & 0xFFE00000 ) 
      {
      switch(Regs[AY_ESHAPE]) 
         {
	      case 8:
	      case 10:
	      case 12:
	      case 14:
	         Countenv -= 0x200000;
	         break;
	      default:
	         Countenv = 0x100000;
	         Increnv = 0;
	      }
      }
	
   Vol0 = (Regs[AY_AVOL] < 16) ? Regs[AY_AVOL] : Envelope;
   Vol1 = (Regs[AY_BVOL] < 16) ? Regs[AY_BVOL] : Envelope;
   Vol2 = (Regs[AY_CVOL] < 16) ? Regs[AY_CVOL] : Envelope;
	
   Volnoise = (((Regs[AY_ENABLE] & 010) ? 0 : Vol0) +
	   ((Regs[AY_ENABLE] & 020) ? 0 : Vol1) +
	   ((Regs[AY_ENABLE] & 040) ? 0 : Vol2)) / 2;
	
   Vol0 = (Regs[AY_ENABLE] & 001) ? 0 : Vol0;
   Vol1 = (Regs[AY_ENABLE] & 002) ? 0 : Vol1;
   Vol2 = (Regs[AY_ENABLE] & 004) ? 0 : Vol2;
		    
   for( i=0; i<AYBufSize; ++i ) 
      {
	   // These strange tricks are needed for getting rid
	   // of nasty interferences between sampling frequency
	   // and "rectangular sound" (which is also the output
	   // of real AY-3-8910) we produce.
	
	   c0 = Counter0;
	   c1 = Counter0 + Incr0;
	   l0 = ((c0&0x8000)?-16:16);
	   if((c0^c1)&0x8000) 
	      {
	      l0=l0*(0x8000-(c0&0x7FFF)-(c1&0x7FFF))/Incr0;
	      }
	   Counter0 = c1 & 0xFFFF;
			    
	   c0 = Counter1;
	   c1 = Counter1 + Incr1;
	   l1 = ((c0&0x8000)?-16:16);
	   if((c0^c1)&0x8000) 
	      {
	      l1=l1*(0x8000-(c0&0x7FFF)-(c1&0x7FFF))/Incr1;
	      }
	   Counter1 = c1 & 0xFFFF;
			    
	   c0 = Counter2;
	   c1 = Counter2 + Incr2;
	   l2 = ((c0&0x8000)?-16:16);
	   if((c0^c1)&0x8000) 
	      {
	      l2=l2*(0x8000-(c0&0x7FFF)-(c1&0x7FFF))/Incr2;
	      }
	   Counter2 = c1 & 0xFFFF;
			    
	   Countnoise &= 0xFFFF;
	   if((Countnoise += Incrnoise) & 0xFFFF0000) 
	      {
	      // The following code is a random bit generator :)
	      StateNoise = ((NoiseGen <<= 1) & 0x80000000
	         ? NoiseGen ^= 0x00040001 : NoiseGen) & 1;
      	}
	
   	Buf[i] = AUDIO_CONV ((l0 * Vol0 + l1 * Vol1 + l2 * Vol2 ) / 6 +
	      (StateNoise ? Volnoise : -Volnoise));
      }
}

void AYPSG::Update()
{
   for (int i=0; i<AYNumChips; i++)
	   pAY8910s[i]->Update();
}

// set a port handler function to be called when AYWriteReg() or AYReadReg()
// is called for register AY_PORTA or AY_PORTB.
void AY8910::SetPortHandler(int port, AYPortHandler func)
{
   port -= AY_PORTA;
   if (port > 1 || port < 0 ) return;

   Port[port] = func;
}

BYTE AY8910ReadPort0(WORD A)
{
	return pAYPSG->ReadReg(0,pAYPSG->GetReg(0));
}

BYTE AY8910ReadPort1(WORD A)
{
	return pAYPSG->ReadReg(1,pAYPSG->GetReg(1));
}

BYTE AY8910ReadPort2(WORD A)
{
	return pAYPSG->ReadReg(2,pAYPSG->GetReg(2));
}

BYTE AY8910ReadPort3(WORD A)
{
	return pAYPSG->ReadReg(3,pAYPSG->GetReg(3));
}

BYTE AY8910ReadPort4(WORD A)
{
	return pAYPSG->ReadReg(4,pAYPSG->GetReg(4));
}



void AY8910ControlPort0(BYTE A, BYTE B)
{
	pAYPSG->SetReg(0, B);
}

void AY8910ControlPort1(BYTE A, BYTE B)
{
	pAYPSG->SetReg(1, B);
}

void AY8910ControlPort2(BYTE A, BYTE B)
{
	pAYPSG->SetReg(2, B);
}

void AY8910ControlPort3(BYTE A, BYTE B)
{
	pAYPSG->SetReg(3, B);
}

void AY8910ControlPort4(BYTE A, BYTE B)
{
	pAYPSG->SetReg(4, B);
}

void AY8910WritePort0(BYTE A, BYTE B)
{
	pAYPSG->WriteReg(0, pAYPSG->GetReg(0), B);
}

void AY8910WritePort1(BYTE A, BYTE B)
{
	pAYPSG->WriteReg(1, pAYPSG->GetReg(1), B);
}

void AY8910WritePort2(BYTE A, BYTE B)
{
	pAYPSG->WriteReg(2, pAYPSG->GetReg(2), B);
}

void AY8910WritePort3(BYTE A, BYTE B)
{
	pAYPSG->WriteReg(3, pAYPSG->GetReg(3), B);
}

void AY8910WritePort4(BYTE A, BYTE B)
{
	pAYPSG->WriteReg(4, pAYPSG->GetReg(4), B);
}





