/*
** File: psg.h -- header file for software implementation of AY-3-8910
**		Programmable sound generator.  This module is used in my
**		Rockulator++ emulator.
**
** Based on: Sound.c, (C) Ville Hallik (ville@physic.ut.ee) 1996
**
** SCC emulation removed.  Made modular and capable of multiple PSG
** emulation.
**
** Modifications (C) 1996 Michael Cuddy, Fen's Ende Software.
** http://www.fensende.com/Users/mcuddy
**
** Encapsulation Modifications (C) 1997 Brian Levine, BJ Software.
**
*/
#ifndef __PSG_H__
#define __PSG_H__

#include <stdarg.h>
#include <stdlib.h>
#include "rockdefs.h"
#include "audiodev.h"

// register id's 
#define AY_AFINE	   (0)
#define AY_ACOARSE	(1)
#define AY_BFINE	   (2)
#define AY_BCOARSE	(3)
#define AY_CFINE	   (4)
#define AY_CCOARSE	(5)
#define AY_NOISEPER	(6)
#define AY_ENABLE	   (7)
#define AY_AVOL		(8)
#define AY_BVOL		(9)
#define AY_CVOL		(10)
#define AY_EFINE	   (11)
#define AY_ECOARSE	(12)
#define AY_ESHAPE	   (13)

#define AY_PORTA	(14)
#define AY_PORTB	(15)

// default clock frequency, frequency in MHz * 100 
#ifndef AY8910_CLOCK 
#define AY8910_CLOCK (1832727040)	// 1.832727040 MHZ 
#endif 

typedef unsigned char AYSAMPLE;	
#define AUDIO_CONV(A) (A)		// use this macro for unsigned samples 

// port callback functions 
BYTE AY8910ReadPort0(WORD A);
BYTE AY8910ReadPort1(WORD A);
BYTE AY8910ReadPort2(WORD A);
BYTE AY8910ReadPort3(WORD A);
BYTE AY8910ReadPort4(WORD A);
void AY8910ControlPort0(BYTE A, BYTE B);
void AY8910ControlPort1(BYTE A, BYTE B);
void AY8910ControlPort2(BYTE A, BYTE B);
void AY8910ControlPort3(BYTE A, BYTE B);
void AY8910ControlPort4(BYTE A, BYTE B);
void AY8910WritePort0(BYTE A, BYTE B);
void AY8910WritePort1(BYTE A, BYTE B);
void AY8910WritePort2(BYTE A, BYTE B);
void AY8910WritePort3(BYTE A, BYTE B);
void AY8910WritePort4(BYTE A, BYTE B);

// forward declaration of typedef, so we can use this in next typedef ... 
class AY8910;

// this is a handler for the AY8910's I/O ports -- called when
// AYWriteReg(AY_PORTA) or AYWriteReg(AY_PORTB) is called.
typedef BYTE (*AYPortHandler)(AY8910 *, int port, int iswrite, BYTE val);

// here's the AY8910 class
class AY8910 
{
private:
protected:
   AYSAMPLE *Buf;	         // sound buffer 
   int UserBuffer;	      // if user provided buffers 
   AYPortHandler Port[2];	// 'A' and 'B' port 
   int AYBufSize;		      // size of sound buffer, in samples 
   int AYSoundRate; 	      // Output rate (Hz) 
   BYTE Regs[16];
   AudioDev * pAudioDev;
   int AudioVoices[3];

   // state variables 
   int LastRegister;
   int Incr0, Incr1, Incr2;
   int Increnv, Incrnoise;
   int StateNoise, NoiseGen;
   int Counter0, Counter1, Counter2, Countenv, Countnoise;
   int Vol0, Vol1, Vol2, Volnoise, Envelope;
public:
   AY8910(AudioDev *pAudio, int voice, AYSAMPLE *buf, int bufSize, int rate);
   virtual ~AY8910();
   void Reset();
   void Update();
   void WriteReg(int r, int v);
   BYTE ReadReg(int r);
   AYSAMPLE *Buffer() { return Buf; };
   void SetBuffer(AYSAMPLE *buf) { Buf = buf; };
   void SetPortHandler(int port, AYPortHandler func);
   int & LastReg() { return LastRegister; };
};

class AYPSG
{
private:
protected:
   int AYSoundRate; 	      // Output rate (Hz) 
   int AYBufSize;		      // size of sound buffer, in samples 
   int AYNumChips;		   // total # of PSG's emulated 
   AY8910 **  pAY8910s;
   AudioDev * pAudioDev;
public:
   AYPSG(int num, int rate, int bufsiz, ...);
   virtual ~AYPSG();
   void Reset() { for (int i=0; i<AYNumChips; i++) pAY8910s[i]->Reset(); };
   void Reset(int num) { pAY8910s[num]->Reset(); };
   void Update();
   void Update(int num) { pAY8910s[num]->Update(); };
   void WriteReg(int n, int r, int v) { pAY8910s[n]->WriteReg(r, v); };
   BYTE ReadReg(int n, int r) { return pAY8910s[n]->ReadReg(r); };
   AYSAMPLE *Buffer(int n) { return pAY8910s[n]->Buffer(); };
   void SetBuffer(int n, AYSAMPLE *buf) { pAY8910s[n]->SetBuffer(buf); };
   void SetPortHandler(int n, int port, AYPortHandler func)
      { pAY8910s[n]->SetPortHandler(port, func); };
   int GetReg(int x) { return pAY8910s[x]->LastReg(); };
   void SetReg(int x, int v) { pAY8910s[x]->LastReg() = v; };
   int Status() { return pAudioDev->Status(); };
};

extern AYPSG *pAYPSG;

// helper function to ensure proper destruction
inline void deleteAYPSG(AYPSG**p)
{ 
   if (*p) { delete *p; *p = 0; }
}

#endif 

