#include <stdio.h>
#include "rockola.h"
#include "time.h"

RockOla * pRockOla;

RockOla::RockOla(ROMS *pR, char *path) : Game (pR, 0x10000, path)
{
   pRockOla = this;
   pProc6502 = new Proc6502;
   pProcessor = (Processor *) pProc6502;

   Coin = 0x2107;

   // setup ioports
   Player0 = new IOPort(0x2104, 0x00);
   Player0->SetKey(KEY_LEFT, KEY_RIGHT, KEY_UP, KEY_DOWN, KEY_A, KEY_D, KEY_W, KEY_S);
   Player0->SetJoy(JOY_LEFT, JOY_RIGHT, JOY_UP, JOY_DOWN, JOY_BUTTON1, JOY_BUTTON1, JOY_BUTTON2, JOY_BUTTON2);
   Player1 = new IOPort(0x2105, 0x00);
   Player1->SetKey(KEY_LEFT, KEY_RIGHT, KEY_UP, KEY_DOWN, KEY_A, KEY_D, KEY_W, KEY_S);
   Player1->SetJoy(JOY_LEFT, JOY_RIGHT, JOY_UP, JOY_DOWN, JOY_BUTTON1, JOY_BUTTON1, JOY_BUTTON2, JOY_BUTTON2);
   CoinSlot = new IOPort(0x2107, 0x00);
   CoinSlot->SetKey(KEY_1, KEY_2, 0, 0, 0, 0, 0, 0);
   Dipswitch = new DipSwitch(0x2106, 0x00);
   pMenDSW = Dipswitch;

   pDisplay = (Display*)NULL;
   pAudioDev = (AudioDev*)NULL;
}

RockOla::~RockOla()
{
   delete Player0;
   delete Player1;
   delete CoinSlot;
   delete Dipswitch;
   deleteAudioDev(&pAudioDev);
   deleteDisplay(&pDisplay);
   delete pProc6502;   
}

void RockOla::Reset()
{
   Game::Reset();
   pProc6502->Reset();

   // initialize sound
   if (bSound)
      {
      pAudioDev = new AudioDev(6, 3);
      if (!pAudioDev->Status())
         {
         bSound = 0;
         }
      else
         {
         ExplosionWave = pAudioDev->LoadWaveFile("samples\\vanexpld.wav");
         Fire1Wave = pAudioDev->LoadWaveFile("samples\\vanshot.wav");
         Fire2Wave = pAudioDev->LoadWaveFile("samples\\vanshot.wav");

         SoundOff = FALSE;
         for (int i=0; i<3; i++)
            NoSound[i] = TRUE;

         pAudioDev->PlaySample(0, SineWaveSample, 32, 1000, 0, 1);
         pAudioDev->PlaySample(1, SineWaveSample, 32, 1000, 0, 1);
         pAudioDev->PlaySample(2, SineWaveSample, 32, 1000, 0, 1);
         pAudioDev->PlayVoice(3);
         pAudioDev->PlayVoice(4);
         pAudioDev->PlayVoice(5);
         }
      }
}

void RockOla::Execute()
{
   int rtn = INT_IRQ;
 
   // reset the timer
   if (bSound)
      aTimer.setHandler(RockOlaAudioHandler, 33);

   // main program loop
   while (rtn != INT_QUIT)
      {
      pProc6502->Exec();
      rtn = Interrupt();
      if (rtn == INT_NMI)
         //-- force a non-masked interrupt
         pProc6502->NMI();
      else if (rtn == INT_IRQ)
         //-- force a masked interrupt
         pProc6502->IRQ();
      }

   // save the high score
   SaveScore();

   //clean up after execution
//   deleteAudioDev(&pAudioDev);
   delete pAudioDev; pAudioDev = 0;
   deleteDisplay(&pDisplay);
}

int RockOla::Interrupt()
{
   static int key3 = 0;
   static int key4 = 0;
   int rtn = INT_IRQ;

   Video();

   if (Game::Interrupt() == INT_QUIT)
      rtn = INT_QUIT;
   //-- check for coin 1 - requires nmi
   else if (aKeyboard[KEY_3] && !key3)
      {
      key3++;
      pRAM[Coin] |= 0x01;
      rtn = INT_NMI;
      }
   else if (!aKeyboard[KEY_3] && key3)
      key3 = 0;

   //-- check for coin 2 - requires nmi
   else if (KB()[KEY_4] && !key4)
      {
      key4++;
      pRAM[Coin] |= 0x01;
      rtn = INT_NMI;
      }
   else if (!KB()[KEY_4] && key4)
      key4 = 0;

   pDisplay->Draw(); 

   return rtn;
}

void RockOla::Video()
{
   unsigned char *start, *ptr;
   unsigned char c;
   int i, j; 

   //-- always clear the background
   clear_to_color(pDisplay->Picture(), pColors[LastPort[3] & 7]);

   // display characters, accounting for rotation
   start = pRAM + 0x0800 + 31 * 32;
   for (i=0; i<32; i++)
      {
   	ptr = start;  
      for (j=0; j<32; j++)
         {
         c = *ptr;
         if (c != 48)
            DrawCharacter (c, j<<3, i<<3, (BYTE)*(ptr+(0x0400)));
   	   ptr -= 32;
  	      }
  	   start++;
     	}

   //-- display the forground characters as sprites
   start = pRAM + 0x0400 + 31 * 32;
   for (i=0; i<32; i++)
      {
     	ptr = start;  
      for (j=0; j<32; j++)
         {
         c = *ptr;
         if (c != 48)
 		      {
            //-- updated sprites are read from 0x1000-0x1fff
            //-- sprite color palette is from 0xc000-0xcfff
            DrawSprite (c, j<<3, i<<3, (BYTE)*(ptr+(0x0800))); 
 		      }

         ptr -= 32;
   	   }
	   start++;
   	}
}

void RockOla::Audio()
{
   if (bSound)
      {
      if (bPause)
         {
         for (int i=0; i<3; i++)
            pAudioDev->AdjustSample (i, 1000, 0);
         pAudioDev->Update();
         }
      else
         {
         // output music
         if (SoundOff == FALSE)
            {
            for (int i=0; i<3; i++)
               {
               if (!NoSound[i] && Sound[i] != 0xFF)
                  pAudioDev->AdjustSample (i, (32000 / (256-pRAM[Sound[i]++])) * 16, 255);
               else
                  pAudioDev->AdjustSample (i, 1000, 0);
               }
            }
         else
            {
            for (int i=0; i<3; i++)
               pAudioDev->AdjustSample (i, 1000, 0);
            }
         pAudioDev->Update();
         }

      // check for overflow
      for (int i=0; i<3; i++)
         if (Sound[i] > SoundBase[i] + 255)
            Sound[i] = SoundBase[i];
      }
}

void RockOla::Load()
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\%s.sav", Path, Name);
   fp = fopen (file, "rb");
   if (fp)
      {
      fread(pRAM, 0x4000, 1, fp);
      fread(NoSound, 4*3, 1, fp);
      fread(Sound, 4*3, 1, fp);
      fread(SoundBase, 4*3, 1, fp);
      fread(oSoundBase, 4*3, 1, fp);
      pProc6502->Load(fp);
      fclose(fp);
      }
}

void RockOla::Save()
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\%s.sav", Path, Name);
   fp = fopen (file, "wb");
   if (fp)
      {
      fwrite(pRAM, 0x4000, 1, fp);
      fwrite(NoSound, 4*3, 1, fp);
      fwrite(Sound, 4*3, 1, fp);
      fwrite(SoundBase, 4*3, 1, fp);
      fwrite(oSoundBase, 4*3, 1, fp);
      pProc6502->Save(fp);
      fclose(fp);
      }
}

//-- draw rotated character at specified position
void RockOla::DrawCharacter (int c, int col, int row, BYTE color)
{
	int x, y;
   int sr, sc;
   int sb1 = 0;
   int sb2 = 0;
   BYTE result = 0;
   BYTE res1, res2;
   BYTE mask;
   int hpos;
   int vpos;

   //-- background character color palette selected using bits 4,5,6
   BYTE cPal = (color & 0x38) >> 3;

   // get the offset to the correct bit for each bitplane
   sb1 = c * 64;

   // for each bit in character
	sc  = 7;
   for (y = 0; y < 8; y ++)
      {
      // get a BYTE at a time
	   sr = 0;
      res1 = pRAM[(sb1>>3)+ROM1];
      res2 = pRAM[(sb1>>3)+ROM2];

      //-- adjust horizontal scrolling
      hpos = sc+col+pRAM[HScroll];
      if (hpos >= 256) hpos -= 256;
      else if (hpos < 0) hpos += 256;

      // for each bit in BYTE
      for (x=0; x<8; x++)
         {
         //-- adjust vertical scrolling
         vpos = sr+row-pRAM[VScroll];
         if (vpos >= 256) vpos -= 256;
         else if (vpos < 0) vpos += 256;

         // find the color by adding the bitplane values
         mask = 0x80 >> x; 
         if ((res1 & mask) == mask) result = 2;
         if ((res2 & mask) == mask) result |= 1;
         if (result)
               pDisplay->Picture()->line[vpos][hpos] = pColors[(cPal<<2)+result];
         sr++;
         sb1++;
         sb2++;
         result = 0;
         }

	   sc--;
      }
}

//-- draw rotated sprite at specified position
void RockOla::DrawSprite (int c, int col, int row, BYTE color)
{
	int x, y;
   int sr, sc;
   int sb1 = 0;
   int sb2 = 0;
   BYTE result = 0;
   BYTE res1, res2;
   BYTE mask;

   //-- sprite color palette selected using bits 0,1,2
   BYTE cPal = color & 0x07;

   // get the offset to the correct bit for each bitplane
   sb1 = c * 64;

   // for each bit in sprite
	sc  = 7;
   for (y = 0; y < 8; y ++)
      {
      // get a BYTE at a time
	   sr = 0;
      res1 = pRAM[(sb1>>3)+0x1000];
      res2 = pRAM[(sb1>>3)+0x1800];

      // for each bit in BYTE
      for (x=0; x<8; x++)
         {
         // find the color by adding the bitplane values
         mask = 0x80 >> x; 
         if ((res1 & mask) == mask) result = 2;
         if ((res2 & mask) == mask) result |= 1;
         if (result)
            pDisplay->Picture()->line[sr+row][sc+col] = pColors[32+(cPal<<2)+result];
         sr++;
         sb1++;
         sb2++;
         result = 0;
         }

	   sc--;
      }
}

void RockOla::CheckSoundReset(int voice)
{
   if (SoundBase[voice] != oSoundBase[voice])
      {
      oSoundBase[voice] = SoundBase[voice];
      Sound[voice] = SoundBase[voice];
      }
}

void RockOla::CheckWave(BYTE data, BYTE bit, int port, int voice, int vol, int stopFlag)
{
   if (data & bit && !(LastPort[port] & bit))
      {
      pAudioDev->PrimeVoice(voice);
      pAudioDev->SetVoiceVolume(voice, vol);
      pAudioDev->StartVoice(voice);            
      }
   else if (stopFlag && !(data & bit) && LastPort[port] & bit)
      pAudioDev->StopVoice(voice);
}

BYTE RockOlaReadPlayer(WORD A)
{
   BYTE valP1 = 0;

   switch (A & 0x07)
      {
      // player controls
      case 4:
         valP1 = pRockOla->Player0->Get();
         break;
      case 5:
         valP1 = pRockOla->Player1->Get();
         break;
      // dip switches
      case 6:
         valP1 = pRockOla->Dipswitch->Get();
         break;
      // coin slots
      case 7:
         valP1 = pGame->RAM()[A] | pRockOla->CoinSlot->Get();
         break;
      }

   return valP1;
}

int RockOlaSpeechOut(WORD A, BYTE B)
{
   pGame->RAM()[A] = B;
   return 0;
}

void RockOlaAudioHandler(...)
{
   pRockOla->Audio();
}

