#ifndef __GAME_H__
#define __GAME_H__

#include <allegro.h>
#include <audio.h>
#include "rockdefs.h"
#include "m6502.h"
#include "maze.h"
#include "display.h"
#include "keyboard.h"
#include "joystick.h"
#include "ioport.h"
#include "timer.h"
#include "proc.h"

typedef struct
{
   int  loadAddr;
   char *fileName;
   int  romSize;
   char *desc;
} ROMS;

class Game
{
public:
   enum GameError { ERR_NONE=0, ERR_NO_DIR, ERR_NO_ROM, ERR_MEMORY };
private:
protected:
   int         Debug;         // turns on debugging
   WORD *      pReadBuffer;	// for debugging only - counts memory read accesses
   WORD *      pWriteBuffer;	// for debugging only - counts memory write accesses
   GameError   ErrorCode;		// last error code
   Processor * pProcessor;		// pointer to game processor object
   Display *   pDisplay;		// pointer to display object
   Keyboard    aKeyboard;		// reference to keyboard object
   Joystick *  pJoystick;		// pointer to joystick object
   int         bSound;		   // audio flag (on/off)
   int         bJoystick;     // joystick flag (on/off)
   int         bThrottle;     // speed throttling flag (on/off)
   int         bPause;        // pause glag (on/off)
   BYTE *      pColors;		   // pointer to game color table
   BYTE *      pRAM;			   // pointer to game ram
   BYTE *      pAudioRAM;		// pointer to audio ram
   BYTE *      pRAMSave;		// game ram image saved for easy reset
   BYTE *      pAudioRAMSave;	// audio ram image saved for easy reset
   int         RAMSize;		   // game ram size
   int         AudioRAMSize;	// audio ram size
   char        Path[80];		// rom image path
   char        Name[9];       // name of game
   char        Descript[80];  // description of game
   ROMS *      pROMs;		   // game rom table
   ROMS *      pAudioROMs;		// audio rom table
   int         ScoreLoaded;	// score loaded flag
   DipSwitch * pMenDSW;		   // pointer to men per game dipswitch object
   DipSwitch * pCoinsDSW;		// pointer to coins per game dipswitch object
   DipSwitch * pBonusDSW;		// pointer to game bonus dipswitch object
   DipSwitch * pLevelDSW;		// pointer to difficulty level dipswitch object
   DipSwitch * pExtraDSW;		// pointer to extra dipswitch object
   int         FrameCount;
   int         VsyncCount;
   int         VsyncFlag;
public:
   Game(ROMS *pRom, int ramSize, char * path);
   virtual ~Game();
   virtual void Reset();
   virtual void Execute() = 0;
   virtual void Video() = 0;
   virtual void Audio() = 0;
   virtual void Load() = 0;
   virtual void Save() = 0;
   virtual void LoadScore() = 0;
   virtual void SaveScore() = 0;
   virtual void ReadScore(int table, int length, int flag=0);
   virtual void WriteScore(int table, int length, int flag=0);
   virtual int  Interrupt();
   virtual GameError LoadRoms(ROMS *roms, BYTE *ram);
   virtual GameError LoadRom(int address, char * name, size_t size, BYTE *ram);
   virtual void LoadAudio(ROMS *pRom, int ramSize);
   virtual void BuildText(WORD numbers, WORD characters, WORD period, WORD space);
   virtual void GetTextChar(WORD sAdd, WORD dOff);
   BYTE * RAM() { return pRAM; };
   char * Desc() { return Descript; };
   Display * Disp() { return pDisplay; };
   Keyboard & KB() { return aKeyboard; };
   Joystick * Joy() { return pJoystick; };
   int Sound() { return bSound; };
   void SetSound(int flag) { bSound = flag; };
   void SetJoy(int flag) { bJoystick = flag; };
   int Throttle() { return bThrottle; };
   void SetThrottle(int flag) { bThrottle = flag; };
   GameError errorCode() { return ErrorCode; };
   void GetVsync();
   int Frames() { return FrameCount; };
   int Vsyncs() { return VsyncCount; };
   WORD * ReadBuffer() { return pReadBuffer; };
   WORD * WriteBuffer() { return pWriteBuffer; };
   DipSwitch * menDSW() { return pMenDSW; };
   DipSwitch * coinsDSW() { return pCoinsDSW; };
   DipSwitch * bonusDSW() { return pBonusDSW; };
   DipSwitch * levelDSW() { return pLevelDSW; };
   DipSwitch * extraDSW() { return pExtraDSW; };
};

extern Game * pGame;

#endif




