#include "sample.h"

//-- global pointer used by callback functions
SampleGame * pSampleGame;

// color tables from color proms
BYTE sampleColors[] = 
{
   0x00, 0xC0, 0x07, 0xF6, 0x00, 0x17, 0x39, 0xF8,   
   0x00, 0x5E, 0xF6, 0x5C, 0x00, 0xC7, 0x39, 0x3F,   
   0x00, 0x3C, 0xF8, 0xD7, 0x00, 0x07, 0xF6, 0x39,   
   0x00, 0xC4, 0xC0, 0x07, 0x00, 0x07, 0x3F, 0xC4,   
   0x40, 0xC0, 0x07, 0xF6, 0x40, 0x17, 0x39, 0xF8,   
   0x40, 0x5E, 0xF6, 0x5C, 0x40, 0xC7, 0x39, 0x3F,   
   0x40, 0x3C, 0xF8, 0xD7, 0x40, 0x07, 0xF6, 0x39,   
   0x40, 0xC4, 0xC0, 0x07, 0x40, 0x07, 0x3F, 0xC4,   
};

// rom images table
ROMS sampleRoms[] = 
{
//   address  name        size    desc
   { 0x0000,  "sample.1", 0x1000, "code" },
   { 0x1000,  "sample.2", 0x1000, "code" },
   { 0x2000,  "sample.3", 0x1000, "code" },
   { 0x3000,  "sample.4", 0x1000, "code" },
   { 0xE000,  "sample.5", 0x0800, "graphics" },
   { 0xE800,  "sample.6", 0x0800, "graphics" },
   { 0, NULL, 0, NULL }	
};

//-- memory write callback table
MemoryWrite SampleGameWrite[] = 
{
   { 0xA800, 0xA8FF, SampleGameVideoOut },
   { 0xFFFF, 0xFFFF, NULL }
};

//-- memory read callback table
MemoryRead SampleGameRead[] = 
{
   { 0x8000, 0x8000, SampleGameReadPlayer0 },
   { 0x8040, 0x8040, SampleGameReadPlayer1 },
   { 0x8080, 0x8080, SampleGameReadDipswitch },
   { 0x8120, 0x813F, SampleGameReadScores },
   { 0xFFFF, 0xFFFF, NULL }
};

//-- constructor
SampleGame::SampleGame() : Game(sampleRoms, 0x10000, "sample")
{
   // setup global pointer
   pFrogger = this;

   // setup name and description
   strcpy(Name, "sample");
   strcpy(Descript, "A Sample Game");

   // create processor object(s)
   pProcZ80 = new ProcZ80(1);
   pProcessor = (Processor *) pProcZ80;

   // setup processor info
   pProcZ80->SetMemoryRW(SampleGameWrite, SampleGameRead, pRWTable, 
      SampleGameInProc, SampleGameOutProc, 30000, pRAM, 1);

   // setup input ports - note that SetKey and SetJoy start with bit 7
   Player0 = new IOPort(0x8000, 0xFF);
   Player0->SetKey(KEY_LEFT, KEY_RIGHT, KEY_UP, KEY_DOWN, 0, 0, KEY_4, KEY_3);
   Player0->SetJoy(JOY_LEFT, JOY_RIGHT, JOY_UP, JOY_DOWN, 0, 0, 0, JOY_BUTTON2);

   Player1 = new IOPort(0x8040, 0xFF);
   Player1->SetKey(0, 0, 0, 0, 0, 0, KEY_2, KEY_1);
   Player1->SetJoy(0, 0, 0, 0, 0, 0, 0, JOY_BUTTON1);

   // setup dipswitch - DipSwitch is derived from IOPort
   Dipswitch = new DipSwitch(0x8080, 0x00);
   Dipswitch->SetMen(0, 0, "3 Men", "4 Men", "5 Men", "Not Used");
   Dipswitch->SetCoins(2, 0, "10000 Points", "15000 Points");

   // setup base class pointers
   pMenDSW = Dipswitch;
   pBonusDSW = Dipswitch;
   pDisplay = 0;
   pAudioDev = 0;
}

//-- destructor
SampleGame::~SampleGame()
{
   delete Player0;
   delete Player1;
   delete Dipswitch;
   delete pProcZ80;   

   // use helper functions to ensure objects aren't already destroyed
   deleteDisplay(&pDisplay);
   deleteAudioDev(&pAudioDev);
}

//-- reset the game
void SampleGame::Reset()
{
   // call base class reset 
   Game::Reset();

   // setup the display
   pDisplay = new Display(256, 256, 256, 256, 0, 0, 0, 0, scr256x256, 200);
   pDisplay->SetClip(16, 0, 240, 256);
   pColors = sampleColors;

   // define where text is in memory
   BuildText(0xE800, 0xE888, 0xE880, 0xE880);

   // clear background buffer
   memset(BackBuffer, 1, 0x400);

   // setup sound
   if (bSound)
      {
      // create audio object(s)
      pAudioDev = new AudioDev(6, 3);
      if (!pAudioDev->Status())
         bSound = 0;
      }

   // reset the processors
   pProcZ80->Reset(1);
}

//-- begine game execution
void SampleGame::Execute()
{
   int rtn = INT_NMI;

   // main execution loop
   while (rtn != INT_QUIT)
      {
      // main processor
      pProcZ80->Exec(1);
      rtn = Interrupt();

      // check for interrupt enabled
      if (pRAM[0xA000])
         pProcZ80->NMI();
      }

   // save the high score
   SaveScore();

   // close the display and audio
   deleteAudioDev(&pAudioDev);
   deleteDisplay(&pDisplay);
}

int SampleGame::Interrupt()
{
   int rtn = INT_NMI;

   // display video
   Video();

   // actually, its better to do sound from within a timer event ;)
   if (bSound)
      Audio();

   // call base class interrupt function
   if (Game::Interrupt() == INT_QUIT)
      rtn = INT_QUIT;

   // display the image
   pDisplay->Draw(); 

   return rtn;
}

//-- draw all graphics - see other source files for specific examples
void SampleGame::Video()
{
   // draw background characters
   
   // do any scrolling

   // do any special effects

   // draw sprites
}

//-- update the audio
void SampleGame::Audio()
{
   // update any audio data
   pAudioDev->Update();
}

//-- load a saved game
void SampleGame::Load()
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\%s.sav", Path, Name);
   fp = fopen (file, "rb");
   if (fp)
      {
      // read the saved ram area
      fread(pRAM+0x8000, 0x4000, 1, fp);

      // read the saved processor info
      pProcZ80->Load(fp);
      fclose(fp);

      // force a video redraw
      memset(BackBuffer, 1, 0x400);
      }
}

//-- save the current game
void SampleGame::Save()
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\%s.sav", Path, Name);
   fp = fopen (file, "wb");
   if (fp)
      {
      // save significant ram area
      fwrite(pRAM+0x8000, 0x4000, 1, fp);

      // save processor info
      pProcZ80->Save(fp);
      fclose(fp);
      }
}

//-- load highscore info
void SampleGame::LoadScore()
{
   if (!ScoreLoaded)
      {
      ReadScore(0x8120, 0x20);
      ScoreLoaded = 1;
      }
}

void SampleGame::SaveScore()
{
   WriteScore(0x8120, 0x20);
}

//-- memory read - read player 0
BYTE SampleGameReadPlayer0(WORD A)
{
   return pSampleGame->Player0->Get();
}

//-- memory read - read player 1
BYTE SampleGameReadPlayer1(WORD A)
{
   return pSampleGame->Player1->Get();
}

//-- memory read - read dipswitch
BYTE SampleGameReadDipSwitch(WORD A)
{
   return pSampleGame->Dipswitch->Get();
}

//-- memory read - read high score
BYTE SampleGameReadScores(WORD A)
{
   pSampleGame->LoadScore();
   return pGame->RAM()[A];
}

//-- memory write - updating video ram
int SampleGameVideoOut(WORD A,  BYTE B)
{
   if (A >= 0xA800 && A < 0xAC00 && pGame->RAM()[A] != B)
      pSampleGame->BackBuffer[A-0xA800] = 1;
   pGame->RAM()[A] = B;
   return 0;
}

//-- handle an IN instruction
BYTE SampleGameInProc(WORD port)
{
   return 0;
}

//-- handle an OUT instruction
void SampleGameOutProc(BYTE data, BYTE port)
{
}



  
