#include <stdio.h>
#include <string.h>
#include "scramble.h"

ScramBase * pScramble;

BYTE scrambleStarColors[] = 
{   
   0x00,0xC0,0x38,0x07,0xF8,0x3F,0xC7,0xFF
};

BYTE scrambleColors[] = 
{
	0x00,0x17,0xC7,0xF6,0x00,0x17,0xC0,0x3F,
	0x00,0x07,0xC0,0x3F,0x00,0xC0,0xC4,0x07,
	0x00,0xC7,0x31,0x17,0x00,0x31,0xC7,0x3F,
	0x00,0xF6,0x07,0xF0,0x00,0x3F,0x07,0xC4
};

ROMS scrambleRoms[] = 
{
   { 0x0000,  "2d", 2048, "code" },
   { 0x0800,  "2e", 2048, "code" },
   { 0x1000,  "2f", 2048, "code" },
   { 0x1800,  "2h", 2048, "code" },
   { 0x2000,  "2j", 2048, "code" },
   { 0x2800,  "2l", 2048, "code" },
   { 0x3000,  "2m", 2048, "code" },
   { 0x3800,  "2p", 2048, "code" },
   { 0xE000,  "5f", 2048, "graphics" },
   { 0xE800,  "5h", 2048, "graphics" },
   { 0, NULL, 0, NULL }	
};

ROMS scrambleAudioRoms[] = 
{
   { 0x0000,  "5c", 2048, "sound" },
   { 0x0800,  "5d", 2048, "sound" },
   { 0x1000,  "5e", 2048, "sound" },
   { 0, NULL, 0, NULL }	
};

ROMS atlantisRoms[] = 
{
   { 0x0000,  "2c", 2048, "code" },
   { 0x0800,  "2e", 2048, "code" },
   { 0x1000,  "2f", 2048, "code" },
   { 0x1800,  "2h", 2048, "code" },
   { 0x2000,  "2j", 2048, "code" },
   { 0x2800,  "2l", 2048, "code" },
   { 0xE000,  "5f", 2048, "graphics" },
   { 0xE800,  "5h", 2048, "graphics" },
   { 0, NULL, 0, NULL }	
};

ROMS atlantisAudioRoms[] = 
{
   { 0x0000,  "5c", 2048, "sound" },
   { 0x0800,  "5d", 2048, "sound" },
   { 0x1000,  "5e", 2048, "sound" },
   { 0, NULL, 0, NULL }	
};

ROMS theendRoms[] = 
{
   { 0x0000,  "ic13", 2048, "code" },
   { 0x0800,  "ic14", 2048, "code" },
   { 0x1000,  "ic15", 2048, "code" },
   { 0x1800,  "ic16", 2048, "code" },
   { 0x2000,  "ic17", 2048, "code" },
   { 0x2800,  "ic18", 2048, "code" },
   { 0x3000,  "ic56", 2048, "code" },
   { 0x3800,  "ic55", 2048, "code" },
   { 0xE000,  "ic30", 2048, "graphics" },
   { 0xE800,  "ic31", 2048, "graphics" },
   { 0, NULL, 0, NULL }	
};

ROMS theendAudioRoms[] = 
{
   { 0x0000,  "ic56", 2048, "sound" },
   { 0x0800,  "ic55", 2048, "sound" },
   { 0, NULL, 0, NULL }	
};

ROMS scobraRoms[] = 
{
   { 0x0000,  "scobra2c.bin", 4096, "code" },
   { 0x1000,  "scobra2e.bin", 4096, "code" },
   { 0x2000,  "scobra2f.bin", 4096, "code" },
   { 0x3000,  "scobra2h.bin", 4096, "code" },
   { 0x4000,  "scobra2j.bin", 4096, "code" },
   { 0x5000,  "scobra2l.bin", 4096, "code" },
   { 0xE000,  "scobra5f.bin", 2048, "graphics" },
   { 0xE800,  "scobra5h.bin", 2048, "graphics" },
   { 0, NULL, 0, NULL }	
};

ROMS scobraAudioRoms[] = 
{
   { 0x0000,  "scobra5c.bin", 2048, "sound" },
   { 0x0800,  "scobra5d.bin", 2048, "sound" },
   { 0x1000,  "scobra5e.bin", 2048, "sound" },
   { 0, NULL, 0, NULL }	
};

MemoryWrite ScrambleWrite[] = 
{
   { 0x4800, 0x4BFF, ScrambleOut },
   { 0x5000, 0x503F, ScrambleOut },
   { 0x6804, 0x6804, ScrambleStars },
   { 0x8200, 0x8200, ScrambleSoundCommand },
   { 0xFFFF, 0xFFFF, NULL }
};

MemoryRead ScrambleRead[] = 
{
//   { 0x83F1, 0x83FA, ScrambleReadScores },
   { 0x8100, 0x8100, ScrambleReadPlayer0 },
   { 0x8101, 0x8101, ScrambleReadPlayer1 },
   { 0x8102, 0x8102, ScrambleReadDipSwitch },
   { 0x8202, 0x8202, ScrambleSecurity },
   { 0xFFFF, 0xFFFF, NULL }
};

MemoryWrite SCobraWrite[] = 
{
   { 0x8800, 0x8BFF, ScrambleOut },
   { 0x9000, 0x803F, ScrambleOut },
   { 0xA000, 0xA000, ScrambleSoundCommand },
   { 0xA804, 0xA804, ScrambleStars },
   { 0xFFFF, 0xFFFF, NULL }
};

MemoryRead SCobraRead[] = 
{
//   { 0x83F1, 0x83FA, ScrambleReadScores },
   { 0x9800, 0x9800, ScrambleReadPlayer0 },
   { 0x9801, 0x9801, ScrambleReadPlayer1 },
   { 0x9802, 0x9802, ScrambleReadDipSwitch },
   { 0xA002, 0xA002, ScrambleSecurity },
   { 0xFFFF, 0xFFFF, NULL }
};

MemoryWrite ScrambleAudioWrite[] = 
{
   { 0xFFFF, 0xFFFF, NULL }
};

MemoryRead ScrambleAudioRead[] = 
{
   { 0xFFFF, 0xFFFF, NULL }
};

ScramBase::ScramBase(ROMS * roms, ROMS * audioRoms, char *path) 
   : Game (roms, 0x10000, path)
{
   pScramble = this;
   pProcZ80 = new ProcZ80(2);
   pProcessor = (Processor *) pProcZ80;

   // setup audio hardware
   LoadAudio(audioRoms, 0x10000);

   // input ports
   Player0 = new IOPort(0x8100, 0xFF);
   Player0->SetKey(KEY_3, KEY_4, KEY_LEFT, KEY_RIGHT, KEY_CONTROL, 0, KEY_ALT, KEY_UP);
   Player0->SetJoy(JOY_BUTTON4, 0, JOY_LEFT, JOY_RIGHT, JOY_BUTTON1, 0, JOY_BUTTON2, JOY_UP);
   Player1 = new DipSwitch(0x8101, 0xFD);
   Player1->SetKey(KEY_1, KEY_2, KEY_LEFT, KEY_RIGHT, KEY_CONTROL, KEY_ALT, 0, 0);
   Player1->SetJoy(JOY_BUTTON1, 0, JOY_LEFT, JOY_RIGHT, JOY_BUTTON1, JOY_BUTTON2, 0, 0);
   Dipswitch = new DipSwitch(0x8102, 0xF1);
   Dipswitch->SetKey(0, KEY_DOWN, 0, KEY_UP, 0, 0, 0, KEY_DOWN);
   Dipswitch->SetJoy(0, JOY_DOWN, 0, JOY_UP, 0, 0, 0, JOY_DOWN);

   SaveRAMStart = 0x4000;
   SaveRAMSize = 0x1100;
   VideoAddress = 0x4800;
   SpriteAddress = 0x5000;
   IntEnableAddress = 0x6801;

   pMenDSW = Player1;
   pCoinsDSW = Player1;
   pDisplay = 0;
   pAYPSG = 0;
   ShotDir = 0;
}

ScramBase::~ScramBase()
{
   delete Player0;
   delete Player1;
   delete Dipswitch;
   delete pProcZ80;   
   deleteAYPSG(&pAYPSG);
   deleteDisplay(&pDisplay);
}   

void ScramBase::Reset()
{
   // reset memory
   Game::Reset();

   // setup the display
   pDisplay = new Display(256, 256, 256, 256, 0, 0, 0, 0, scr256x256, 200);
   pDisplay->SetClip(16, 0, 240, 256);
   pColors = scrambleColors;
   BuildText(0xE800, 0xE888, 0xE880, 0xE880);
   memset(BackBuffer, 1, 0x400);

   SetupStars();

   // setup sound
   if (bSound)
      {
      pAYPSG = new AYPSG(1, 1000, 33);
      if (!pAYPSG->Status())
         {
         bSound = 0;
         }
      }

   // reset the processors
   pProcZ80->Reset(1);
   pProcZ80->Reset(2);
//   pProcZ80->SetInterruptMode(1, 0);
}

void ScramBase::Execute()
{
   int rtn = INT_NMI;

   // main execution loop
   while (rtn != INT_QUIT)
      {
      // main processor
      pProcZ80->Exec(1);
      rtn = Interrupt();
      if (pRAM[IntEnableAddress])
         {
         pProcZ80->NMI();
         }


      if(rtn != INT_QUIT)
         {
         // audio processor
//         pProcZ80->Exec(2);
//         pProcZ80->NMI();
         }
      }

   // save the high score
   SaveScore();

   // close the display and audio
   deleteAYPSG(&pAYPSG);
   deleteDisplay(&pDisplay);
}

int ScramBase::Interrupt()
{
   int rtn = INT_NMI;

   Video();
   if (bSound)
      Audio();

   if (Game::Interrupt() == INT_QUIT)
      rtn = INT_QUIT;

   pDisplay->Draw(); 

   return rtn;
}

void ScramBase::Video()
{
   unsigned char *start, *ptr;
   int soff, scur;
   unsigned char c;
   int i, j; 


   // display characters, accounting for rotation
   start = pRAM + VideoAddress + 31 * 32;
   soff = 31 * 32;
   for (i=0; i<32; i++)
      {
   	ptr = start;
   	scur = soff;
      for (j=0; j<32; j++)
         {
         c = *ptr;
         if (BackBuffer[scur])
            {
            DrawCharacter (c, j<<3, i<<3, pRAM[SpriteAddress+(i<<1)+1]);
            BackBuffer[scur] = 0;
            }
   	   ptr -= 32;
         scur -= 32;
  	      }
  	   start++;
      soff++;
     	}

	for (i=0; i<256; i+=8)
		{
		int scroll = pRAM[SpriteAddress + (i>>2)];

      blit(pDisplay->Background(), pDisplay->Picture(), 
         0, i, scroll, i, 256, 8);
      if (scroll)
         blit(pDisplay->Background(), pDisplay->Picture(), 
            0, i, scroll-256, i, 256, 8);
      }

   // draw the bullets
	for (i=0; i<0x20; i += 4)
	   {
		int x, y;
      x = pRAM[SpriteAddress+0x60+i+1];
		if (x >= 16 && x < 240)
   		{
			y = 256 - pRAM[SpriteAddress+0x60+i+3] - 4;

			if (y >= 0)
				for (j=0; j<3; j++)
               if (ShotDir)
                  pDisplay->Picture()->line[y-j][x] = 0xFF;
               else
                  pDisplay->Picture()->line[y][x+j] = 0xFF;
			}
		}

   // draw stars
	if (bStars)
	   {
		BlinkCount++;
		if (BlinkCount > 42)
   		{
			BlinkCount = 0;
			StarsBlink = (StarsBlink++) % 4;
	   	}
		for (i=0; i<StarCount; i++)
   		{
			int y = StarY[i];
			if (((StarX[i] & 1) ^ ((y >> 4) & 1)) &&
			   pDisplay->Picture()->line[y][StarX[i]] == 0)
            {
				pDisplay->Picture()->line[y][StarX[i]] = StarColor[i];
            if ((StarsBlink == 0 && StarCode[i] & 0x01) ||
               (StarsBlink == 1 && StarCode[i] & 0x04) ||
               (StarsBlink == 2 && StarX[i] & 0x02) ||
               (StarsBlink == 3))
   				pDisplay->Picture()->line[y][StarX[i]] = StarColor[i];
            }
		   }
	   }
   
   // draw sprites
   for (i=7*4; i>=0; i-=4)
     	{
   	if (pRAM[SpriteAddress+0x40+i+3] > 8)
	   	{
         DrawSprite(SpriteAddress+0x40+i);
   		}
	   }
}

void ScramBase::Audio()                     
{
   pAYPSG->Update();
}

void ScramBase::Load()                     
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\%s.sav", Path, Name);
   fp = fopen (file, "rb");
   if (fp)
      {
      fread(pRAM+SaveRAMStart, SaveRAMSize, 1, fp);
      fread(pAudioRAM+0x8000, 0x0400, 1, fp);
      pProcZ80->Load(fp);
      fclose(fp);
      memset(BackBuffer, 1, 0x400);
      }
}

void ScramBase::Save()                     
{
   char file[80];
   FILE *fp;
   sprintf (file, "%s\\%s.sav", Path, Name);
   fp = fopen (file, "wb");
   if (fp)
      {
      fwrite(pRAM+SaveRAMStart, SaveRAMSize, 1, fp);
      fwrite(pAudioRAM+0x8000, 0x0400, 1, fp);
      pProcZ80->Save(fp);
      fclose(fp);
      }
}

void ScramBase::LoadScore()
{
   if (!ScoreLoaded && memcmp (pRAM + 0x83F1, "\x63\x04", 2) == 0 &&
      memcmp (pRAM + 0x83F9, "\x27\x01", 2) == 0)
      {
      ReadScore(0x83F1, 0x0A);
      memcpy (pRAM + 0x83EF, pRAM + 0x83F1, 2);
      ScoreLoaded = 1;
      }
}

void ScramBase::SaveScore()
{
   WriteScore(0x83F1, 0x0A);
}

void ScramBase::DrawCharacter (int c, int col, int row, BYTE color)
{
	int x, y;
   int sr, sc;
   int sb1 = 0;
   BYTE result = 0;
   BYTE res1, res2;
   BYTE mask;
   int ROM1 = 0xE000;
   int ROM2 = 0xE800;

   // get the offset to the correct bit for each bitplane
   sb1 = c * 64;

   // for each bit in character
	sc  = 7;
   for (y=0; y<8; y++)
      {
      // get a BYTE at a time
	   sr = 0;
      res1 = pRAM[(sb1>>3)+ROM1];
      res2 = pRAM[(sb1>>3)+ROM2];

      // for each bit in BYTE
      for (x=0; x<8; x++)
         {
         // find the color by adding the bitplane values
         mask = 0x80 >> x; 
         if ((res1 & mask) == mask) result = 2;
         if ((res2 & mask) == mask) result |= 1;
         pDisplay->Background()->line[sr+row][sc+col] = pColors[(color<<2)+result];
         sr++;
         sb1++;
         result = 0;
         }

	   sc--;
      }
}
                         
void ScramBase::DrawSprite (WORD off)
{
   int col = pRAM[off];
   int row = pRAM[off+3];
   int c = (pRAM[off+1] & 0x3F)<<2;
   int color = pRAM[off+2];
   int fx = pRAM[off+1] & 0x80;
   int fy = pRAM[off+1] & 0x40;
	int x, y;
   int sr, sc;
   int sb1 = 0;
   BYTE result = 0;
   BYTE res1, res2, res3, res4, res5, res6, res7, res8;
   BYTE mask;
   int ROM1 = 0xE000;
   int ROM2 = 0xE800;
   int hpos;
   int vpos;

   // get the offset to the correct bit for each bitplane
   sb1 = c * 64;

   // for each bit in character
	sc = (fx?0:7);
   for (y = 0; y < 8; y ++)
      {
      //-- adjust horizontal scrolling
      hpos = sc+col;

      // get a BYTE at a time
	   sr = (fy?7:0);

	   res1 = pRAM[(sb1>>3)+ROM1+(!fy?(fx?0:16):(fx?8:24))];
      res2 = pRAM[(sb1>>3)+ROM2+(!fy?(fx?0:16):(fx?8:24))];
      res3 = pRAM[(sb1>>3)+ROM1+(!fy?(fx?8:24):(fx?0:16))];
      res4 = pRAM[(sb1>>3)+ROM2+(!fy?(fx?8:24):(fx?0:16))];
	   res5 = pRAM[(sb1>>3)+ROM1+(!fy?(fx?16:0):(fx?24:8))];
      res6 = pRAM[(sb1>>3)+ROM2+(!fy?(fx?16:0):(fx?24:8))];
      res7 = pRAM[(sb1>>3)+ROM1+(!fy?(fx?24:8):(fx?16:0))];
      res8 = pRAM[(sb1>>3)+ROM2+(!fy?(fx?24:8):(fx?16:0))];

      // for each bit in BYTE
      for (x=0; x<8; x++)
         {
         //-- adjust vertical scrolling
         vpos = sr+row;

         // find the color by adding the bitplane values
         mask = 0x80 >> x; 
         result = 0;
         if ((res1 & mask) == mask) result = 2;
         if ((res2 & mask) == mask) result |= 1;
         if (result && hpos > 15 && hpos < 240)
            pDisplay->Picture()->line[vpos][hpos] = pColors[(color<<2)+result];
         result = 0;
         if ((res3 & mask) == mask) result = 2;
         if ((res4 & mask) == mask) result |= 1;
         if (result && hpos > 15 && hpos < 240)
            pDisplay->Picture()->line[vpos+8][hpos] = pColors[(color<<2)+result];
         result = 0;
         if ((res5 & mask) == mask) result = 2;
         if ((res6 & mask) == mask) result |= 1;
         if (result && hpos > 7 && hpos < 232)
            pDisplay->Picture()->line[vpos][hpos+8] = pColors[(color<<2)+result];
         result = 0;
         if ((res7 & mask) == mask) result = 2;
         if ((res8 & mask) == mask) result |= 1;
         if (result && hpos > 7 && hpos < 232)
            pDisplay->Picture()->line[vpos+8][hpos+8] = pColors[(color<<2)+result];
         if(fy) sr--; else sr++;
         sb1++;
         }

	   if(fx) sc++; else sc--;
      }
}
                         
void ScramBase::SetupStars(void)
{
	int generator = 0;

	// precalculate the star background 
   bStars = 0;
	StarCount = 0;
   StarsBlink=0;
	BlinkCount=0;

	for (int i=255; i>=0; i--)
   	{
		for (int j=511; j>=0; j--)
		   {
			int bit1,bit2;

			generator <<= 1;
			bit1 = (~generator >> 17) & 1;
			bit2 = (generator >> 5) & 1;

			if (bit1 ^ bit2) generator |= 1;

			if (i >= 16 && i < 240 && ((~generator >> 16) & 1) && 
			   (generator & 0xff) == 0xff)
			   {
				int color;

				color = (~(generator >> 8)) & 0x07;
				if (color && StarCount < SCRAMBLE_MAX_STARS)
				   {
					StarX[StarCount] = i;
					StarY[StarCount] = j;
               StarCode[StarCount] = color;
					StarColor[StarCount++] = scrambleStarColors[color];
   				}
			   }
		   }
	   }
}

BYTE ScrambleReadPlayer0(WORD A)
{
   return pScramble->Player0->Get();
}

BYTE ScrambleReadPlayer1(WORD A)
{
   return pScramble->Player1->Get();
}

BYTE ScrambleReadDipSwitch(WORD A)
{
   BYTE rtn = pScramble->Dipswitch->Get();
   if (pScramble->pProcZ80->PC() == 0x00E4)
      rtn &= 0x7f;
   return rtn;
}

BYTE ScrambleTest(WORD A)
{
   return pGame->RAM()[A];
}

BYTE ScrambleReadScores(WORD A)
{
   pScramble->LoadScore();
   return pGame->RAM()[A];
}

int ScrambleOut(WORD A,  BYTE B)
{
   if (A >= pScramble->VideoAddress && A < pScramble->VideoAddress+0x0400 && pGame->RAM()[A] != B)
      pScramble->BackBuffer[A-pScramble->VideoAddress] = 1;
	else if ((A & 1) && pGame->RAM()[A] != B)
		for (int i=(A-pScramble->SpriteAddress)>>1; i<0x400; i+=0x20)
         pScramble->BackBuffer[i] = 1;
   pGame->RAM()[A] = B;
   return 0;
}

BYTE ScrambleSecurity(WORD A)
{
   return 0x6F;
}

int ScrambleSoundCommand(WORD A,  BYTE B)
{
   return 0;   
}

int ScrambleStars(WORD A,  BYTE B)
{
	pScramble->bStars = B & 1;
   pScramble->RAM()[A] = B;
   return 0;
}

BYTE ScrambleInProc(WORD port)
{
   return 0;
}

void ScrambleOutProc(BYTE data, BYTE port)
{
}

BYTE ScrambleAudioInProc(WORD port)
{
   return 0;
}

void ScrambleAudioOutProc(BYTE data, BYTE port)
{
}

Scramble::Scramble() 
   : ScramBase (scrambleRoms, scrambleAudioRoms, "scramble")
{
   strcpy(Name, "scramble");
   strcpy(Descript, "Scramble");

   pProcZ80->SetMemoryRW(ScrambleWrite, ScrambleRead, pRWTable, ScrambleInProc,
      ScrambleOutProc, 30000, pRAM, 1);
   pProcZ80->SetMemoryRW(ScrambleAudioWrite, ScrambleAudioRead, pAudioRWTable, 
     ScrambleAudioInProc, ScrambleAudioOutProc, 20000, pAudioRAM, 2);

   // input ports
   Player1->SetMen(0, 0, "3 Jets", "4 Jets", "5 Jets", "256 Jets");
   Dipswitch->SetCoins(1, 0, "1 Coin 1 Play", "2 Coins 1 Play", "3 Coins 1 Play", 
      "1 Coin 2 Plays");
}

Atlantis::Atlantis() 
   : ScramBase (atlantisRoms, atlantisAudioRoms, "atlantis")
{
   strcpy(Name, "atlantis");
   strcpy(Descript, "Atlantis");

   pProcZ80->SetMemoryRW(ScrambleWrite, ScrambleRead, pRWTable, ScrambleInProc,
      ScrambleOutProc, 30000, pRAM, 1);
   pProcZ80->SetMemoryRW(ScrambleAudioWrite, ScrambleAudioRead, pAudioRWTable, 
     ScrambleAudioInProc, ScrambleAudioOutProc, 20000, pAudioRAM, 2);

   // input ports
   Player1->SetMen(1, 1, "5 Jets", "3 Jets");
   Dipswitch->SetCoins(1, 0, "1 Coin 1 Play", "2 Coins 1 Play", "3 Coins 1 Play", 
      "1 Coin 2 Plays");
}

TheEnd::TheEnd() 
   : ScramBase (theendRoms, theendAudioRoms, "theend")
{
   strcpy(Name, "theend");
   strcpy(Descript, "The End");

   pProcZ80->SetMemoryRW(ScrambleWrite, ScrambleRead, pRWTable, ScrambleInProc,
      ScrambleOutProc, 30000, pRAM, 1);
   pProcZ80->SetMemoryRW(ScrambleAudioWrite, ScrambleAudioRead, pAudioRWTable, 
     ScrambleAudioInProc, ScrambleAudioOutProc, 20000, pAudioRAM, 2);

   // input ports
   Player1->SetMen(0, 0, "3 Jets", "4 Jets", "5 Jets", "256 Jets");
   Dipswitch->SetCoins(1, 0, "1 Coin 1 Play", "2 Coins 1 Play", "3 Coins 1 Play", 
      "1 Coin 2 Plays");

   ShotDir = 1;
}

SCobra::SCobra() 
   : ScramBase (scobraRoms, scobraAudioRoms, "scobra")
{
   strcpy(Name, "scobra");
   strcpy(Descript, "Super Cobra");

   pProcZ80->SetMemoryRW(SCobraWrite, SCobraRead, pRWTable, ScrambleInProc,
      ScrambleOutProc, 30000, pRAM, 1);
   pProcZ80->SetMemoryRW(ScrambleAudioWrite, ScrambleAudioRead, pAudioRWTable, 
     ScrambleAudioInProc, ScrambleAudioOutProc, 20000, pAudioRAM, 2);

   // input ports
   Player1->SetExtra(0, 0, "No Continue", "Allow Continue");
   Player1->SetMen(1, 0, "3 Jets", "5 Jets");
   Dipswitch->SetCoins(1, 1, "1 Coin 99 Plays", "1 Coins 1 Play", "2 Coins 1 Play", 
      "4 Coin 3 Plays");
   pExtraDSW = Player1;

   SaveRAMStart = 0x8000;
   SaveRAMSize = 0x1100;
   VideoAddress = 0x8800;
   SpriteAddress = 0x9000;
   IntEnableAddress = 0xA801;
}




