/* (C)2013 Gabor Lenart LGB http://ep.lgb.hu/jsep/
 * Emulates WD177x/EXDOS card for the JavaScript based Enterprise 128 emulator.
 * This emulation is VERY far from being accurate, the only goal is
 * to have a bare usable with the emulator (currently read-only) and _nothing_ more.
 * Eg operations are always done without any waiting, etc.
 * WD part is written with reading this doc I had found with G:
 * http://info-coach.fr/atari/documents/mydoc/WD1772-JLG-V11.pdf
 */

"use strict";

var WDINT_ON = 0x3E;
var WDINT_OFF = 0x3C;
var WDDRQ = 2;
var SEEK_ERROR = 32 | 16 | 8; // bit mask to set in WD status in case of seek error [spin-up-completed, seek-error, CRC]
var SEEK_OK = 32; // bit mask to set WD status in case of seek OK  [spin-up-completed]
var DEBUG_DISK = false;
var DISK_FORMATS = [
	[40,  8],
	[40,  9],
	[80,  8],
	[80,  9],
	[80, 15],
	[80, 18],
	[80, 21],
	[82, 21],
	[80, 36]
];
var MAX_TRACKS = 0;
var MAX_SECTORS = 0;
var EXDOS_INI_CONTENT = "LOAD RUNME\r\n";


var readAddressBuffer = new Uint8Array(6);
var lastDiskStat, lastCmdDesc, wdStatus, wdTrack, wdData, wdSector;
var wdCommand, wdInterrupt, wdDRQ, driveSel, diskSide, diskInserted;
var wdBuffer, wdBufferSize, wdBufferPos;


/* Tricky way to load a single file program: we create a "fake" disk image on-the-fly
 * and we use EXDOS then to load it from disk within the emulator :) also_load parameter
 * can be true to signal to auto-run program, this is done by creating EXDOS.INI file too.
 * Note: the created disk image is not "correct" in every way, but the goal is only
 * to have an image which is readable by EXDOS and I don't care about problems a FAT checker
 * would report about this image :) This image is not realized as a "real" existing image
 * file anywhere, so who cares. */
function loadDiskConstructedFromFile(fileName, also_load) {
	var imageBuffer = new ArrayBuffer(327680); // faked disk image is always 40/8 double sided, so this is the size
	var image = new Uint8Array(imageBuffer);
	var prg = new Uint8Array(imageBuffer, 0x1800, 65536);
	var ini = new Uint8Array(imageBuffer, 0x1400,  1024);
	if (installBinary("FAKEDISKSKELETON.imghead", 0x1400, 0x1400, image, 0) == false) {
		window.alert("Cannot download the image skeleton used by emulator to construct on-the-fly disk image :(");
		return false;
	}
	var len = installBinary(fileName, 17, 0xBF10, prg, 0);
	if (len == false) {
		window.alert("Cannot download program file: " + fileName);
		return false;
	}
	if (prg[0] || (!prg[1]) || prg[15]) {
		window.alert("Downloaded program file (fileName) does not seem to be a valid EXOS file with EXOS header!");
		return false;
	}
	image[0x65C] = len & 255; // set the correct size of the file
	image[0x65D] = len >> 8;
	image[0x65E] = 0;
	if (also_load) {
		for (var i = 0; i < EXDOS_INI_CONTENT.length; i++)
			ini[i] = EXDOS_INI_CONTENT.charCodeAt(i);
	} else {
		image[0x628] = 79; // modify EXDOS.INI filename, since we don't need it!
		image[0x629] = 70;
		image[0x62A] = 70;
	}
	MAX_TRACKS = 40; // see comment at the declaration of imageBuffer at the beginning of this function
	MAX_SECTORS = 8;
	debug("DISK: faked disk image is creating");
	return image;
}


function loadDisk(diskName) {
	var i, j, max = 0, min = 99999999, all = "", df = {};
	for (i in DISK_FORMATS) {
		all += DISK_FORMATS[i][0] + "/" + DISK_FORMATS[i][1] + " ";
		j = (DISK_FORMATS[i][0] * DISK_FORMATS[i][1]) << 10;
		if (df[j] != null)
			window.alert("Internal Error: two or more disk formats result in the same image size! Please fix DISK_FORMATS in exdos.js!");
		df[j] = [DISK_FORMATS[i][0], DISK_FORMATS[i][1]];
		if (j < min) min = j;
		if (j > max) max = j;
	}
	if (DEBUG_DISK) {
		debug("Supported disk formats: " + all);
		debug("Disk image min/max size: " + min + " ... " + max);
	}
	debug("Installing DISK image '" + diskName + "' in JS runtime via HTTP.");
	i = installBinary(diskName, min, max, false, 0);
	if (i == false) {
		window.alert("Disk image cannot be downloaded, disk won't be accessible!");
		return i;
	}
	var len = i.length;
	if (df[len] == null) {
		window.alert("Unknown disk image file size (" + len + "), cannot guess geometry! Disk won't be accessible! Supported disk formats " +
		"(tracks/sectors, always 2 side): " + all);
		return false;
	}
	MAX_TRACKS  = df[len][0];
	MAX_SECTORS = df[len][1];
	debug("DISK image: heads=2[always], tracks=" + MAX_TRACKS + "[guessed], sectors=" + MAX_SECTORS + "[guessed], image_size=" + len + "[got]");
	return i;
}


function diskStat() {
	var i = '<font color="' + (driveSel ? "green" : "gray") + '">' + wdTrack + "/" + wdSector + "#" + diskSide + "=" + lastCmdDesc + "</font>";
	if (i != lastDiskStat) {
		lastDiskStat = i;
		document.getElementById("disk").innerHTML = i;
	}
}


function exdosReset() {
	lastDiskStat = "";
	lastCmdDesc = "?";
	wdStatus = 4; // track 0 flag is on at initialization
	wdTrack = 0;
	wdData = 0;
	wdSector = 1;
	wdCommand = 0xD0; // fake last command as force interrupt
	wdInterrupt = WDINT_OFF; // interrupt output is OFF
	wdDRQ = 0; // no DRQ (data request)
	driveSel = (disk != false); // drive is selected by default if there is disk image!
	diskSide = 0; // selected disk side
	diskInserted = (disk == false) ? 1 : 0; // 0 means inserted disk, 1 means = not inserted
	//diskStat();
	debug("EXDOS card got reset.");
}


function doCRC(buf, nBytes, n) { // CRC16 hmmm Just copied from ep128emu :) - thanks!
	var nBits = nBytes << 3;
	var bitCnt = 0;
	var bitBuf = 0;
	var bufP = 0;
	while (nBits--) {
		if (bitCnt == 0) {
			bitBuf = buf[bufP++];
			bitCnt = 8;
		}
		if ((bitBuf ^ (n >> 8)) & 0x80)
			n = (n << 1) ^ 0x1021;
		else
			n = (n << 1);
		n = n & 0xFFFF;
		bitBuf = (bitBuf << 1) & 0xFF;
		bitCnt--;
	}
	return n;
}


function wdGetData() {
	if (wdDRQ) {
		wdData = wdBuffer[wdBufferPos++];
		if ((--wdBufferSize) == 0)
			wdDRQ = 0; // end of data, switch of DRQ!
	}
	return wdData;
}


function wdSetData(val) {
	wdData = val;
	lastCmdDesc="set-data";
	//diskStat();
}


function logoSkippingHackOnDiskAccess() {
	if (LOGO_SKIPPING) {
		keyStates[8] |= 0x40;
		LOGO_SKIPPING = false;
		debug("KBD: logo skipping mode is deactivated now because of disk access! Space is released by software.");
	}
}



function wdSendCommand (cmd) {
	wdCommand = cmd;
	if (DEBUG_DISK)
		debug("WD: command 0b" + (cmd >> 4).toString(2) + " (low nibble=0b" + (cmd & 15).toString(2) + ")");
	wdDRQ = 0; // reset DRQ
	wdInterrupt = WDINT_OFF; // reset INTERRUPT
	switch (cmd >> 4) {
		case 0: // restore (type I), seeks to track zero
			if (driveSel) {
				wdStatus = 4 | SEEK_OK; // 4->set track0
				wdTrack = 0;
			} else {
				wdStatus = SEEK_ERROR | 4; // set track0 flag (4) is needed here not to be mapped A: as B: by EXDOS, or such?!
			}
			wdInterrupt = WDINT_ON;
			lastCmdDesc = "restore";
			break;
		case 1: // seek (type I)
			if (wdData < MAX_TRACKS && driveSel) {
				wdTrack = wdData;
				wdStatus = SEEK_OK;
			} else {
				wdStatus = SEEK_ERROR;
			}
			wdInterrupt = WDINT_ON;
			if (wdTrack == 0) wdStatus |= 4; // set track-0 if we're on track 0
			lastCmdDesc = "seek";
			break;
		case 8: // read sector, single (type II)
			if (DEBUG_DISK)
				debug("WD: reading " + wdTrack + "/" + wdSector + "S" + diskSide);
			//diskStat();
			if (wdSector > 0 && wdSector <= MAX_SECTORS && driveSel) {
				wdDRQ = WDDRQ; // generate DRQ!
				wdBufferSize = 512;
				wdBuffer = disk;
				//wdBufferPos = (wdTrack * MAX_SECTORS + wdSector - 1) << 9;
				//wdBufferPos = (wdTrack * MAX_SECTORS + wdSector - 1 + MAX_SECTORS * MAX_TRACKS * diskSide) << 9;
				wdBufferPos = (wdTrack * MAX_SECTORS * 2 + wdSector - 1 + MAX_SECTORS * diskSide) << 9;
				wdStatus = 0;
			} else {
				wdStatus = 16; // record not found
			}
			wdInterrupt = WDINT_ON;
			lastCmdDesc = "readsec";
			break;
		case 12: // read address (type III)
			if (driveSel) {
				readAddressBuffer[0] = wdTrack;
				wdSector = wdTrack; // why?! it seems WD puts track number into sector register. Sounds odd ...
				readAddressBuffer[1] = diskSide;
				readAddressBuffer[2] = 1; // first sector!
				readAddressBuffer[3] = 2; // sector size???
				var i = doCRC(readAddressBuffer, 4, 0xB230);
				readAddressBuffer[4] = i >> 8; // CRC1
				readAddressBuffer[5] = i & 0xFF; // CRC2
				wdDRQ = WDDRQ;
				wdBufferSize = 6;
				wdBuffer = readAddressBuffer;
				wdBufferPos = 0;
				wdStatus = 0;
				logoSkippingHackOnDiskAccess();
			} else {
				wdStatus = 16; // record not found
			}
			wdInterrupt = WDINT_ON;
			lastCmdDesc = "readaddr";
			break;
		case 13: // force interrupt (type IV)
			if (cmd & 15) wdInterrupt = WDINT_ON;
			wdStatus = (wdTrack == 0) ? 4 : 0;
			lastCmdDesc = "forceint";
			break;
		default:
			debug("WD: this command is not handled yet: 0b" + (cmd >> 4).toString(2));
			wdStatus = 4 | 8 | 16 | 64; // unimplemented command results in large number of problems reported :)
			wdInterrupt = WDINT_ON;
			lastCmdDesc = "unknown";
			break;
	}
	//diskStat();
}



function exdosPortRead (addr) {
	if (addr == 0x10 || addr == 0x14) {
		wdInterrupt = WDINT_OFF; // on reading WD status, interrupt is reset!
		return 128 | wdStatus | wdDRQ; // motor is always on, the given wdStatus bits, DRQ handled separately (as we need it for exdos status too!)
	} else if (addr == 0x11 || addr == 0x15) {
		return wdTrack;
	} else if (addr == 0x12 || addr == 0x16) {
		return wdSector;
	} else if (addr == 0x13 || addr == 0x17) {
		return wdGetData();
	} else { // get exdos status
		return wdInterrupt | (wdDRQ << 6) | diskInserted | 0x40; // 0x40 -> disk not changed
	}
}


function exdosPortWrite (addr, val) {
	if (addr == 0x10 || addr == 0x14) {
		wdSendCommand(val);
	} else if (addr == 0x11 || addr == 0x15) {
		wdTrack = val;
	} else if (addr == 0x12 || addr == 0x16) {
		wdSector = val;
	} else if (addr == 0x13 || addr == 0x17) {
		wdSetData(val);
	} else { // set exdos control
		driveSel = (disk != false) && ((val & 15) == 1);
		diskSide = (val >> 4) & 1;
		diskInserted = driveSel ? 0 : 1;
		lastCmdDesc = "exdos";
		//diskStat();
		//debug("EXDOS set: select=" + driveSel + " side=" + diskSide + " inserted[0=yes]=" + diskInserted);
	}
}
