/* (C)2013 Gábor Lénárt LGB http://ep.lgb.hu/jsep/
 * Extra networking stuff for the JavaScript based Enterprise 128 emulator.
 * It can be used via EXOS and needs netlinkfs.rom (source: netlinkfs.asm).
 * Note: EXDOS must be installed, barely because we use its error codes :)
 * It's a cleaner way as those error messages may be translated on on-English
 * machines, and also we can save memory/code to represent them here :)
 * Future plans: remove AJAX call from enterprise.js, ROM/disk/snapshot image
 * can be used via this subsytem, so unifying the place where HTTP requests
 * are delivered from.
 */

"use strict";

var DEBUG_NETLINK = true;

var NL_SEGMENT = 0x18;
var NL_ADDR;
var NL_BUFFER_SIZE = 256; // max of 1024! greater value: faster execution but harder to interrupt with soft IRQ stop key for example ...

var nlHTTP;
var nlPath;
var nlBusy;
var nlBuffer, nlBufferSize, nlBufferOffs;
var nlValidDirs = ["\\"];
var nlRegs = {};

// EXOS error codes:  http://ep.homeserver.hu/Dokumentacio/Konyvek/EXOS_2.1_technikal_information/exos/constants/Ch3.html
// EXDOS error codes: http://www.ep128.hu/Ep_Konyv/Exdos_Muszaki_Leiras.htm#15
// Summ: http://enterprise.iko.hu/errors.htm
var ERR = {
	"IPATH": 0xA8, // Invalid pathname string (EXDOS)
	"INP":   0x9F  // Wrong number of parameters / Invalid number of parameters (EXDOS)
};



function netlinkSendRawRequest(url, method, post_data, callback) {
	nlHTTP.abort();
	nlHTTP.callback = callback;
	nlBusy = true;
	if (DEBUG_NETLINK)
		debug("NetLink: sending async HTTP " + method + " request to " + url);
	nlHTTP.open(method, url, true); // async mode, state change callback is set in netlinkInitialization()
	nlHTTP.overrideMimeType('text\/plain; charset=x-user-defined');
	nlHTTP.startedAt = Date.now();
	nlHTTP.send(post_data);
}




function netlinkReset() {
	nlHTTP.abort();
	nlPath = "\\";
	nlBufferSize = 0;
	nlBufferOffs = 0;
	nlBuffer = null;
	nlBusy = false;
	nlRegs = {};
	debug("NetLink: got reset");
}

function netlinkInitialization() {
	nlHTTP = new XMLHttpRequest();
	nlHTTP.onreadystatechange = function () {
		if (nlHTTP.readyState == 4) {
			nlHTTP.timeSpent = Date.now() - nlHTTP.startedAt;
			if (DEBUG_NETLINK)
				debug("NetLink: got HTTP async reply (" + nlHTTP.status + ") within " + nlHTTP.timeSpent + "ms.");
			nlHTTP.callback();
			nlBusy = false;
		}
	};
	NL_ADDR = (NL_SEGMENT << 14) | 0x3BFE;
	for (var a = 1; a < nlCommandTab.length; a++) {
		nlCommandTab[0][1] += " " + nlCommandTab[a][0];
		nlCommandTab[a][1] = "Help on " + nlCommandTab[a][0] + ":\r\n  " + nlCommandTab[a][1];
	}
	nlCommandTab[0][1] += NL_ROM_HELP_TAIL;
	debug("NetLink: active in segment 0x" + NL_SEGMENT.toString(16));
}

function netlinkPrintOutToBuffer(s) {
	nlBuffer = new Uint8Array(s.length + 2);
	for (nlBufferSize = 0; nlBufferSize < s.length; nlBufferSize++)
		nlBuffer[nlBufferSize] = s.charCodeAt(nlBufferSize);
	nlBuffer[nlBufferSize++] = 13;
	nlBuffer[nlBufferSize++] = 10;
	nlBufferOffs = 0;
}

function netlinkGetString(lo, hi) {
	var addr = (hi << 8) | lo;
	var len = readbyte(addr);
	var out = "";
	while (len--) {
		addr = (addr + 1) & 0xFFFF;
		out += String.fromCharCode(readbyte(addr));
	}
	return out;
}


var NL_ROM_HELP_HEAD = "NL    version 1.0  (JSep NetLink)";
var NL_ROM_HELP_MORE = "\r\n      (C)2013 Gabor Lenart LGB\r\n      http://ep.lgb.hu/jsep\r\n\r\n"
var NL_ROM_HELP_TAIL = "\r\n\r\nUse help on the given command.";

var WEBDEMO_CHRW_MAP = {
	0xE1: 0x8A,	// á
	0xE9: 0x88,	// é
	0xED: 0x69,	// í (conv to i ...)
	0xF3: 0x6F,	// ó (conv to o ...)
	0xF6: 0x93,	// ö
	0xF5: 0x93,	// ő (conv to ö ...)
	0xFA: 0x75,	// ú (conv to u ...)
	0xFC: 0xFC,	// ü
	0xFB: 0xFC	// ű (conv to ü ...)
};


function nlPing(req) {
	if (req.length) return _exosError(ERR.INP);
	netlinkSendRawRequest(AJAX_BIN_URL + "null", "GET", null, function () {
		netlinkPrintOutToBuffer(nlHTTP.responseText.length + " bytes (HTTP=" + nlHTTP.status + "): time=" + nlHTTP.timeSpent + " ms");
	});
}


var nlChatBytePos = 0;
var nlChatNick = null;

function nlChat(req) {
	if (nlChatNick == null)
		return netlinkPrintOutToBuffer("You must use NCNICK command first, to set up a nick name.");
	req = req.join(" ");
	if (req) req = "[" + nlChatNick + "] " + req;
	netlinkSendRawRequest("http://ep.lgb.hu/jsep/chatapi/?bp=" + nlChatBytePos, "POST", req, function () {
		if (nlHTTP.status == 200) {
			var len = nlHTTP.responseText.length;
			if (len >= 4) {
				if (DEBUG_NETLINK) {
					debug("NetLink: CHAT: nlChatBytePos bytes = " +
						nlHTTP.responseText.charCodeAt(0) + "," +
						nlHTTP.responseText.charCodeAt(1) + "," +
						nlHTTP.responseText.charCodeAt(2) + "," +
						nlHTTP.responseText.charCodeAt(3)
					);
				}
				nlChatBytePos =
					(nlHTTP.responseText.charCodeAt(0) & 255) |
					((nlHTTP.responseText.charCodeAt(1) & 255) << 8) |
					((nlHTTP.responseText.charCodeAt(2) & 255) << 16) |
					((nlHTTP.responseText.charCodeAt(3) & 255) << 24)
				;
				if (DEBUG_NETLINK) {
					debug("NetLink: CHAT: nlChatBytePos bytes = " +
						nlHTTP.responseText.charCodeAt(0) + "," +
						nlHTTP.responseText.charCodeAt(1) + "," +
						nlHTTP.responseText.charCodeAt(2) + "," +
						nlHTTP.responseText.charCodeAt(3)
					);
					debug("NetLink: CHAT: nlChatBytePos=" + nlChatBytePos);
					debug("NetLink: READ: " + nlHTTP.responseText);
				}
				if (len > 4) {
					netlinkPrintOutToBuffer(nlHTTP.responseText.slice(4).trim());
					// nlChatBytePos += len - 4;
				}
			}
		} else
			netlinkPrintOutToBuffer("HTTP error: " + nlHTTP.status);
	});
}


function nlWebDemo(req) {
	if (req.length != 1)
		return _exosError(ERR.INP);
	req = req[0];
	if (!(/^[0-9a-zA-Z-.]+$/.test(req)))
		return netlinkPrintOutToBuffer("Bad param syntax, domain name wanted.");
	netlinkSendRawRequest(PROXY_BIN_URL + "http://" + req + "//EPcnV/00", "GET", null, function () {
		if (nlHTTP.status == 200) {
			for (var a = 0, o = ""; a < nlHTTP.responseText.length; a++) {
				var c = nlHTTP.responseText.charCodeAt(a);
				if (c == 10)
					o += "\r\n";
				else if (c >= 32 && c < 160)
					o += String.fromCharCode(c);
				else if (WEBDEMO_CHRW_MAP[c] != null) {
					o += String.fromCharCode(WEBDEMO_CHRW_MAP[c]);
				} else
					o += "?";
			}
			netlinkPrintOutToBuffer("HTML view: " + req + " (" + nlHTTP.timeSpent + "ms)\r\n" + o);
		} else
			netlinkPrintOutToBuffer("HTTP error: " + nlHTTP.status);
	});
}


function _exosError(code) {
	nlRegs.a = code;
	return false;
}


// Canonicalize path
function netlinkCanonicalizePath(path) {
	path = path.replace(/^NL:/i, "");
	if (path[0] != "\\") path = nlPath + "\\" + path;
	var o = [];
	path = path.split("\\");
	for (var a in path) {
		a = path[a];
		if (a == "..") {
			if (o.pop() == undefined) return false;
		} else if (a != "." && a != "") {
			if (!(/^[0-9a-zA-Z_.-]+$/.test(a))) return false;
			o.push(a.toUpperCase());
		}
	}
	return "\\" + o.join("\\");
}


function nlCd(req) {
	if (!req.length)
		return netlinkPrintOutToBuffer("NL:" + nlPath); // without parameter just return the current directory to be printed
	if (req.length > 1)
		return _exosError(ERR.INP);
	req = netlinkCanonicalizePath(req[0]);
	if (!req)
		return _exosError(ERR.IPATH);
	if (nlValidDirs.indexOf(req) < 0) { // TODO: this is the place where request should be sent to the server to check if directory exists/is accessable at all ... Root directory is assumed to be accessable always though, no check for that.
		if (DEBUG_NETLINK)
			debug("NetLink: validation of path: " + req);
		nlValidDirs.push(req);
	}
	nlPath = req; // OK, activate the new path as the current working directory property of JSep
}


function nlDir(req) {
	if (req.length > 1)
		return _exosError(ERR.INP);
	if (req.length) {
		req = netlinkCanonicalizePath(req[0]);
		if (!req)
			return _exosError(ERR.IPATH);
	} else
		req = nlPath;
	//Volume in drive A: is FASZOMABRA
	netlinkPrintOutToBuffer("Directory of NL:" + req);
	// TODO maybe: if dir is OK, validate it for nlCd ...
}


// uses callback table in ROM (check netlinkfs.asm out)
// you can/should call this from netlinkTrapWrite()
// if id=0, dat=0
function romJP(n) {
	if (n >= readbyte(0xC00D)) return;
	z80.l = readbyte(0xC00E + n * 2);
	z80.h = readbyte(0xC00F + n * 2);
	z80.a = 1;
}



var nlCommandTab = [
	["NL", NL_ROM_HELP_HEAD + NL_ROM_HELP_MORE + "Available commands:", function () {
		netlinkPrintOutToBuffer(nlCommandTab[0][1]);
	} ],
	["NL:", "Selects NLFS as default device", function () {
		romJP(0);
	} ],
	["LGB", "Test only, ignore it!", function () {
		romJP(1);
	} ],
	["SYSINFO", "Test only, ignore it!", function () {
		romJP(2);
	} ],
	["NCHAT", "NLFS iRC client gateway", nlChat],
	["NCNICK", "Set iRC (pseudo) nick name", function (req) {
		if (req.length != 1) return _exosError(ERR.INP);
		if (!(/^[a-zA-Z0-9]{2,16}$/.test(req[0])))
			return netlinkPrintOutToBuffer("Nick name can only consist of US letters and numbers and must be at least two bytes and max of 16.");
		nlChatNick = req[0];
		netlinkPrintOutToBuffer("Chat nick is set to: " + nlChatNick);
		//nlChat(["JSep user set nick to: " + nlChatNick]);
	} ],
	["NCD", "Print/change NLFS directory", nlCd],
	["NPING", "Test server connection", nlPing],
	["NDIR", "Display directory", nlDir],
	["WDEMO", "View head of web sites", nlWebDemo],
	["JSEP", "Print JSEP information", function () {
		netlinkPrintOutToBuffer(
			"JSep: the JavaScript EP128 emulator\r\n" +
			"Version: build #" + BUILDID + "\r\n" +
			"Home: http://ep.lgb.hu/jsep/\r\n" +
			"Server: " + NL_API_URL + "\r\n" +
			"Cwd: NL:" + nlPath + "\r\n" +
			"Location: " + window.location + "\r\n" +
			"Browser: " + window.navigator.userAgent
		);
	} ]
];


function netlinkSearchCommand(cmd) {
	for (var a = 0; a < nlCommandTab.length; a++)
		if (cmd == nlCommandTab[a][0])
			return nlCommandTab[a];
	return null;
}


function netlinkTrapRead(id) {
	return 0xFF;
}


function netlinkTrapWrite(id, dat) {
	if (DEBUG_NETLINK)
		debug("NetLink: TRAP: id=" + id + ", dat=" + dat + ", c=" + z80.c);
	if (id) return;
	switch (dat) {

		case 0: // **** this is the ROM entry transfer point

			nlBusy = false; // default status
			nlBuffer = null;
			nlBufferSize = 0;
			nlBufferOffs = 0;
			nlRegs = {};
			if (z80.c == 2 || z80.c == 3) {
				var req = netlinkGetString(z80.e, z80.d).trim().split(/ +/);
				if (DEBUG_NETLINK)
					debug("NetLink: rom entry request, c=" + z80.c + ", b=" + z80.b + ", cmdstr=\"" + req.join(" ") + "\"");
				if (req.length == 0) return;
				var cmd = netlinkSearchCommand(req[0]);
				req = req.slice(1);
			}
			switch (z80.c) {
				case 2:
					if (cmd != null) {
						nlRegs.c = 0;
						nlRegs.a = 0;
						if (cmd[2] != null)
							cmd[2](req); // commands should call netlinkPrintOutToBuffer() by their own if they need it!
						else
							netlinkPrintOutToBuffer("Direct call is not supported on " + cmd[0]);
					}
					break;
				case 3:
					if (z80.b) {
						if (cmd != null) {
							nlRegs.c = 0;
							nlRegs.a = 0;
							netlinkPrintOutToBuffer(cmd[1]);
						}
					} else {
						if (DEBUG_NETLINK)
							debug("NetLink: general help request, take it easy ...");
						netlinkPrintOutToBuffer(NL_ROM_HELP_HEAD);
					}
					break;
			}
			break;

		case 1: // **** this is the query busy status of last OP + get buffer, modifying Z80 register A

			if (nlBusy)
				z80.a = 0; // still busy!
			else if (nlBufferOffs < nlBufferSize) {
				for (var a = 0; a < NL_BUFFER_SIZE && nlBufferOffs < nlBufferSize; a++)
					memory[NL_ADDR + 2 + a] = nlBuffer[nlBufferOffs++];
				if (DEBUG_NETLINK)
					debug("NetLink: buffer transferred, " + a + " byte(s).");
				memory[NL_ADDR] = a & 0xFF;
				memory[NL_ADDR + 1] = a >> 8;
				z80.a = 129; // ready + there is buffer content
				if (nlBufferOffs >= nlBufferSize) {
					nlBuffer = null;
					nlBufferSize = 0;
					nlBufferOffs = 0;
				}
			} else
				z80.a = 1; // ready, but no buffer content
			break;

		case 2: // **** this modifies Z80 registers with the needed response, should only be called if query busy status != 0

			for (var a in nlRegs) {
				if (DEBUG_NETLINK)
					debug("NetLink: Z80 reg " + a + " was modified to 0x" + z80[a].toString(16));
				z80[a] = nlRegs[a];
			}
			nlRegs = {};
			break;

		case 3: // **** sysinfo backreport

			break;
	}
}
