/* (C)2013 Gábor Lénárt LGB http://ep.lgb.hu/jsep/
 * JavaScript based Enterprise-128 emulator.
 * Z80 core (z80*.js) is from JSspeccy (v1). */

"use strict";



var browserIsGood = true;

/* test for typed arraies */
if(typeof Uint8Array == "undefined") {
	window.alert("Uint8Array() cannot be used. You should consider to use a better browser (or newer version at least). In case of IE, the minimum is said to be IE 10.");
	browserIsGood = false;
}
/* test for AJAX via XMLHttpRequest */
if (!window.XMLHttpRequest) {
	window.alert("XMLHttpRequest object does not exist. You should consider to use a better browser.");
	browserIsGood = false;
}
if (!window.DataView) {
	window.alert("DataView is not supported by your browser. You should consider to use a better browser.");
	browserIsGood = false;
}


var DEBUG_PORT = true;

var running = false;

//var EXOS_ROM = "exos21.rom";
var EXOS_ROM = "EXOS24UK-NOILLOPS.ROM";
//var BASIC_ROM = "basic21.rom";
var BASIC_ROM = "no";
//var EXDOS_ROM = "exdos13.rom";
var EXDOS_ROM = "EXDOS20.ROM";
//var ZT_ROM = "zt18uk.rom";
var ZT_ROM = "ZT18UK-emuhoz.ROM";
var NL_ROM = "netlinkfs.rom";

var AJAX_BIN_URL = "http://ep.lgb.hu/jsep/data/?GetBinary=";
var PROXY_BIN_URL = "http://ep.lgb.hu/jsep/data/?ProxyBinary=";
var NL_API_URL = "not-yet";
//var PROXY_BIN_URL = "";

var RAM_SIZE = 128; /* in Kbytes, can be overridden via netconfig */
var DISK_IMAGE = "lgb.img"; /* disk image name, can be overridden via netconfig */
var DISK_HACK = "";
var LOAD_SNAPSHOT = "";

var SOUND = false;

var CPU_CLOCK = 4000000;

var disk, snapshot, firstRamByte, keySel, canvas, ctx, imageData, imageDataData, refresh;
var memoryBuffer = new ArrayBuffer(0x400000);
var memory = new Uint8Array(memoryBuffer); // full EP memory space (including "free holes") as bytes (4M)
var vram = new Uint8Array(memoryBuffer, 0x3F0000); // export video ram as bytes (64K)
//var pageBases = new Array(4);
var pageBases = new Uint32Array(4);
var port = new Uint8Array(256);
var cmosRam = new Uint8Array(256);
var keyStates = new Uint8Array(10);
var palR = new Uint8Array(256);
var palG = new Uint8Array(256);
var palB = new Uint8Array(256);
var col4Trans = new Uint8Array(256 * 4);
var col16Trans = new Uint8Array(256 * 2);
var lpb = new Uint8Array(8); // this is Nick's LPB, but only first 8 bytes (pal info is in lpbR,G,B)
/* these are RGB decoded values for the current pal index used from current LPB and via BIAS setting as well */
var lpbR = new Uint8Array(16);
var lpbG = new Uint8Array(16);
var lpbB = new Uint8Array(16);
var CPU_STATES_PER_NICK_SLOT;
var CPU_CLOCK_1000;
var NICK_SLOTS_PER_DAVE_TICK;

var audioCtx, audioNode;

var DEBUG_CONTENT = null;
var DEBUG_IS_NEW = false;
var DEBUG_LAST_LINE = "";
var DEBUG_SAME_COUNTER = 1;

function debug_nonew() {
	document.getElementById("debug").style.backgroundColor = "#FFFFFF";
	DEBUG_IS_NEW = false;
}

function debug_show() {
	document.getElementById("debug").innerHTML = DEBUG_CONTENT;
	debug_nonew();
}

function debug_old(msg) {
	if (DEBUG_CONTENT == null)
		DEBUG_CONTENT = document.getElementById("debug").innerHTML;
	if (msg == DEBUG_LAST_LINE)
		DEBUG_SAME_COUNTER++;
	else {
		if (DEBUG_SAME_COUNTER > 1) {
			DEBUG_CONTENT += "REPEAT: last line was repeated " + DEBUG_SAME_COUNTER + " times.\n";
			DEBUG_SAME_COUNTER = 1;
		}
		DEBUG_CONTENT += msg + "\n";
		DEBUG_LAST_LINE = msg;
	}
	if (!DEBUG_IS_NEW) {
		DEBUG_IS_NEW = true;
		document.getElementById("debug").style.backgroundColor = "#FF8080";
		//document.getElementById("debug").innerHTML = "Stop the emulation to be able to see messages!";
	}
}

var debug = console.log;
var debug = debug_old;

function enterpriseReset() {
	debug("EP+Z80 reset is starting!");
	for (var i = 0; i < 0x100; i++) {
		port[i] = 0xFF;
	}
	//z80_reset();
	daveReset();
	keyboardReset();
	keySel = -1;
	exdosReset();
	netlinkReset();
	rtcReset();
	z80_reset();
	debug("EP+Z80 reset is done.");
	if (disk != false && DISK_HACK == "load") {
		debug("Logo skipping mode is requested!");
		LOGO_SKIPPING = true;
	}
}


function enterpriseRamClear() {
	for (var i = firstRamByte; i < 0x400000; i++)
		memory[i] = 0xFF;
}


function enterpriseHardReset() {
	debug("HARD RESET was requested (clearing RAM first).");
	enterpriseRamClear();
	enterpriseReset();
}


function enterpriseEmergencyReset(msg) {
	msg = "FATAL event, doing hard reset: " + msg;
	debug(msg);
	window.alert(msg);
	showz80();
	enterpriseHardReset();
}


function urlDecode(url) {
	return decodeURIComponent(url.replace(/\+/g, ' '));
}


function installBinary(objID, minSize, maxSize, installTo, installOffset) {
	var url, ud = urlDecode(objID);
	if (ud.substr(0, 7).toLowerCase() != "http://" && ud.substr(0, 8).toLowerCase() != "https://" && ud.substr(0, 6).toLowerCase() != "ftp://")
		url = AJAX_BIN_URL + objID + "&b=" + BUILDID;
	else {
		if (PROXY_BIN_URL != "") {
			debug("Using Proxy-URL to get " + ud);
			url = PROXY_BIN_URL + objID;
		} else
			url = ud;
	}
	document.getElementById("h1name").innerHTML = "Downloading: " + url;
	debug("AJAX data transfer from " + url);
	var http = new XMLHttpRequest();
	http.open("GET", url, false);
	http.overrideMimeType('text\/plain; charset=x-user-defined'); // this is a nice trick I found on the net: it tricks the browser not parse data, so you can transfer binary files!
	// TODO error handler to catch errors ...
	var err;
	try {
		http.send(null);
	} catch (err) {
		debug("... got exception " + err);
		window.alert("Cannot download data (" + url + ") file (" + objID + ") because of exception: " +
		err.message + " (" + err + ")\n\n" +
		"It is recommended to check the JavaScript console out of your browser!"
		);
		document.getElementById("h1name").innerHTML = document.title;
		return false;
	}
	if (http.status != 200) {
		debug("... got status " + http.status + " ERROR");
		window.alert("Cannot download data (" + url + ") file (" + objID + ") because of HTTP error " + http.status);
		document.getElementById("h1name").innerHTML = document.title;
		return false;
	}
	var len = http.responseText.length;
	debug("... got status " + http.status + ", response data length is " + len);
	if (len < minSize || len > maxSize) {
		window.alert("Bad result for data file (" + objID + ") it must be sized between " + minSize + " and " + maxSize + " but we got " + len + " bytes.");
		document.getElementById("h1name").innerHTML = document.title;
		return false;
	}
	var retval;
	if (installTo == false) {
		installOffset = 0;
		installTo = new Uint8Array(len);
		retval = installTo;
	} else
		retval = len;
	for (var i = 0; i < len; i++)
		installTo[i + installOffset] = http.responseText.charCodeAt(i);
	document.getElementById("h1name").innerHTML = document.title;
	return retval;
}


function installRom(name, segment) {
	debug("Installing ROM image '" + name + "' at segment " + segment + " via HTTP.");
	return installBinary(name, 16384, 65536, memory, segment << 14);
}


function epNetConfig() {
	var p = window.location.search.replace("?", "");
	if (p == "") {
		debug("NetConfig: no parameters in the URL field, using defaults.");
		return;
	}
	var pp = p.split("&");
	for (var i = 0; i < pp.length; i++) {
		var ppp = pp[i].split("=");
		if (ppp.length != 2) {
			debug("NetConfig: warning: bad option syntax: " + pp[i]);
			continue;
		}
		var nc = "";
		if (ppp[0] == "clock") {
			nc = "CPU clock";
			ppp[1] = parseFloat(ppp[1]);
			if (isNaN(ppp[1]) || ppp[1] < 4.0 || ppp[1] > 12.0) {
				debug("NetConfig: invalid CPU clock");
				continue;
			}
			CPU_CLOCK = ppp[1] * 1000000;
		} else if (ppp[0] == "mem") {
			nc = "RAM size";
			ppp[1] = parseInt(ppp[1]);
			if (isNaN(ppp[1])) {
				debug("NetConfig: invalid RAM size");
				continue;
			}
			RAM_SIZE = ppp[1];
		} else if (ppp[0] == "disk") {
			nc = "DISK image";
			DISK_IMAGE = ppp[1];
		} else if (ppp[0] == "exos") {
			nc = "EXOS image";
			EXOS_ROM = ppp[1];
		} else if (ppp[0] == "basic") {
			nc = "BASIC image";
			BASIC_ROM = ppp[1];
		} else if (ppp[0] == "exdos") {
			nc = "EXDOS image";
			EXDOS_ROM = ppp[1];
		} else if (ppp[0] == "diskhack") {
			nc = "file-to-disk conversion";
			DISK_HACK = ppp[1];
		} else if (ppp[0] == "autostart") {
			nc = "autostart";
			AUTO_START = ppp[1];
		} else if (ppp[0] == "zt") {
			nc = "ZT image";
			ZT_ROM = ppp[1];
		} else if (ppp[0] == "kbdreset") {
			nc = "KBD reset on unknown key";
			KBD_RESET_ON_UNKNOWN = ppp[1];
		} else if (ppp[0] == "snapshot") {
			nc = "SNAPSHOT file to load";
			LOAD_SNAPSHOT = ppp[1];
		} else if (ppp[0] == "skiplogo") {
			LOGO_SKIPPING = ((ppp[1] == "yes") ? true : false);
			nc = "Logo skipping mode";
		} else if (ppp[0] == "sound") {
			nc = "Sound support";
			SOUND = (ppp[1] == "yes");
		}
		if (nc != "") {
			debug("NetConfig: setting " + nc + " '" + ppp[0] + "' to " + ppp[1]);
		} else {
			debug("NetConfig: warning: unknown option: " + ppp[0] + "=" + ppp[1]);
		}
	}
}



function enterpriseInitEmulation() {
	var i;
	debug("Initializing emulator ...");
	debug("Browser: " + navigator.userAgent);
	debug("JavaScript strict mode: " + (!this));
	/* Configuration can be done via web! */
	epNetConfig();
	/* CPU clock related stuffs */
	CPU_STATES_PER_NICK_SLOT = CPU_CLOCK / NICK_SLOTS_PER_SEC;
	CPU_CLOCK_1000 = CPU_CLOCK / 1000.0;
	NICK_SLOTS_PER_DAVE_TICK = NICK_SLOTS_PER_DAVE_TICK_HI;
	debug("Z80 clock: " + (CPU_CLOCK / 1000000) + " MHz  [t-states/slot = " + CPU_STATES_PER_NICK_SLOT + "]");
	/* disk image */
	if (DISK_IMAGE != "no") {
		if (DISK_HACK == "use" || DISK_HACK == "load") {
			disk = loadDiskConstructedFromFile(DISK_IMAGE, DISK_HACK == "load");
			if (disk == false)
				window.alert("Problem during file/disk image on-the file conversion, no disk image is used, EXDOS access won't work!");
			else {
				if (DISK_HACK == "load") {
					ZT_ROM = "no";
					debug("Disabling ZT ROM on diskhack=load");
				}
			}
		} else {
			disk = loadDisk(DISK_IMAGE);
			if (disk == false)
				window.alert("DISK image cannot be found/downloaded, EXDOS access won't work!");
		}
	} else {
		debug("No disk image was requested, disk won't be accessible!");
		disk = false;
	}
	/* Memory configuration */
	/*if (RAM_SIZE < 64)
		RAM_SIZE = 64;
	else if (RAM_SIZE > 2048)
		RAM_SIZE = 2048;
	else
		RAM_SIZE &= 0xFF0;*/
	//debug("Total RAM size: " + RAM_SIZE + "K.");
	/*i = "Enterprise-128 (" + RAM_SIZE + "K) JavaScript Emulator";
	document.title = i;
	document.getElementById('h1name').innerHTML = i;*/
	document.title = document.getElementById("h1name").innerHTML = "Enterprise-128 JavaScript Emulator";
	for (i = 0; i < 0x400000; i++)
		memory[i] = 0xFF;
	if (LOAD_SNAPSHOT != "")
		snapshot = loadSnapshot(LOAD_SNAPSHOT);
	else
		snapshot = false;
	//firstRamByte = 0x400000 - (RAM_SIZE << 10); // export first memory byte
	if (snapshot == false) {
		if (RAM_SIZE < 64)
			RAM_SIZE = 64;
		else if (RAM_SIZE > 2048)
			RAM_SIZE = 2048;
		else
			RAM_SIZE &= 0xFF0;
		debug("Total RAM size: " + RAM_SIZE + "K.");
		installRom(EXOS_ROM, 0);
		if (BASIC_ROM != "no")
			installRom(BASIC_ROM, 4);
		else
			debug("No BASIC ROM image is configured, skipping.");
		if (EXDOS_ROM != "no")
			installRom(EXDOS_ROM, 0x20);
		else
			debug("No EXDOS ROM image is configured, skipping.");
		if (ZT_ROM != "no")
			installRom(ZT_ROM, 0x10);
		else
			debug("No ZT ROM image is configured, skipping.");
		if (NL_ROM != "no")
			installRom(NL_ROM, NL_SEGMENT);
		else
			debug("No NL ROM image is configured, skipping.");
	} else {
		RAM_SIZE = snapshot["RAM_SIZE"];
		debug("SNAPSHOT: RAM size determined by snapshot: " + RAM_SIZE + "K.");
		debug("SNAPSHOT: snapshot loading mode, skipping ROM install!");
		LOGO_SKIPPING = false;
	}
	//debug("Total RAM size: " + RAM_SIZE + "K.");
	firstRamByte = 0x400000 - (RAM_SIZE << 10); // export first memory byte
	/* generate EP palette in RGB colour space and other tables */
	for (i = 0; i < 256; i++) {
		cmosRam[i] = 0;
		// this is the RGB palette
		palR[i] = (((i << 2) & 4) | ((i >> 2) & 2) | ((i >> 6) & 1)) * 255 / 7;
		palG[i] = (((i << 1) & 4) | ((i >> 3) & 2) | ((i >> 7) & 1)) * 255 / 7;
		palB[i] = (                 ((i >> 1) & 2) | ((i >> 5) & 1)) * 255 / 3;
		// this is translation table for  4 colour modes
		col4Trans[i * 4 + 0] = ((i >> 2) & 2) | ((i >> 7) & 1);
		col4Trans[i * 4 + 1] = ((i >> 1) & 2) | ((i >> 6) & 1);
		col4Trans[i * 4 + 2] = ((i     ) & 2) | ((i >> 5) & 1);
		col4Trans[i * 4 + 3] = ((i << 1) & 2) | ((i >> 4) & 1);
		// this is translation table for 16 colour modes
		col16Trans[i * 2 + 0] = ((i << 2) & 8) | ((i >> 3) & 4) | ((i >> 2) & 2) | ((i >> 7) & 1);
		col16Trans[i * 2 + 1] = ((i << 3) & 8) | ((i >> 2) & 4) | ((i >> 1) & 2) | ((i >> 6) & 1);
	}
	/* browser specific stuffs */
	canvas = document.getElementById("screen");
	canvas.width = WIDTH.toString();
	canvas.height = HEIGHT.toString();
	canvas.style.width = (WIDTH * WIDTH_SCALING).toString() + "px";
	canvas.style.height = (HEIGHT * HEIGHT_SCALING).toString() + "px";
	debug(
		"Canvas native size is " + canvas.width + "x" + canvas.height +
		", display size is " + (WIDTH * WIDTH_SCALING) + "x" + (HEIGHT * HEIGHT_SCALING)
	);
	ctx = canvas.getContext("2d");
	ctx.fillStyle = "black";
	ctx.fillRect(0, 0, WIDTH, HEIGHT); /* set alpha to opaque */
	if (ctx.getImageData) {
		imageData = ctx.getImageData(0, 0, WIDTH, HEIGHT);
		imageDataData = imageData.data;
		debug("Canvas data type is " + imageDataData.toString());
	} else {
		window.alert("This browser does not support getImageData/putImageData. Emulator won't work. Try a decent, standard compliant (let's say latest versions of firefox, chrome or such) browser!");
		browserIsGood = false;
	}
	netlinkInitialization();
	/* finally EP reset */
	enterpriseReset();
	nickPowerOn(); // Nick won't reset for real on reset, it just have "some" values even at power-up.
	/* snapshot related!
	 * If snapshot was read, then we override some settings according to the snapshot */
	if (snapshot) {
		debug("SNAPSHOT: Overriding HW state from the snapshot now!");
		initFromSnapshot(snapshot);
		snapshot = true; // let's free the used memory by the snapshot! [well, garbage collector will do it]
	}
	/* audio setup */
	if (SOUND) {
		window.AudioContext = window.AudioContext || window.webkitAudioContext;
		if (typeof AudioContext == "undefined") {
			debug("AUDIO: Your browser does not support Audio API!");
			window.alert("Audio was requested but your browser does not support Audio API! Switching audio off.");
			SOUND = false;
		} else {
			debug("AUDIO: API is OK (" + (
				(typeof webkitAudioContext == "undefined") ? "W3C" : "webkit"
			) + "): " + AudioContext.toString());
		}
	} else {
		debug("AUDIO: was not requested");
	}
	if (SOUND) {
		audioCtx = new AudioContext();
		debug("AUDIO: sound buffer size = " + SOUND_BUFFER_SIZE);
		debug("AUDIO: sampling rate: audio = " +
			audioCtx.sampleRate + "Hz");
		NICK_SLOTS_PER_AUDIO_SAMPLE = NICK_SLOTS_PER_SEC / audioCtx.sampleRate;
		audioRenderingCounter = NICK_SLOTS_PER_AUDIO_SAMPLE;
		debug("AUDIO: nick slots per audio sampling: " + NICK_SLOTS_PER_AUDIO_SAMPLE);
		audioNode = audioCtx.createJavaScriptNode(SOUND_BUFFER_SIZE, 2, 2);
		audioNode.onaudioprocess = processAudioJavaScriptCallBack;
		//window.alert("Your browser supports Web Audio API :) It is good, but the support in JSep is not ready yet :-(");
	}
	/* and set JS kbd events */
	document.onkeydown = keyDown;
	document.onkeyup = keyUp;
	document.onkeypress = keyDown;
}

function audioPlay() {
	if (SOUND) audioNode.connect(audioCtx.destination);
}

function audioStop() {
	if (SOUND) audioNode.disconnect();
}


function contend_memory(addr) {
	return 0; /* TODO: implement. Comment from JSspeccy */
}
function contend_port(addr) {
	return 0; /* TODO: implement. Comment from JSspeccy */
}


function readbyte_internal(addr) {
	return memory[pageBases[addr >> 14] | (addr & 0x3FFF)];
}
var readbyte = readbyte_internal;


function writebyte_internal(addr, val) {
	addr = pageBases[addr >> 14] | (addr & 0x3FFF);
	if (addr >= firstRamByte)
		memory[addr] = val;
}
var writebyte = writebyte_internal;


function readport(addr) {
	addr &= 0xFF;
	if (addr >= 0xB0 && addr <= 0xB3) {
		return port[addr];
	} else if (addr == 0xB4) {
		return daveInterruptR;
	} else if (addr == 0xB5) {
		return (keySel == -1) ? 0xFF : keyStates[keySel];
	} else if ((addr & 0xF0) == 0x10) {
		return exdosPortRead(addr);
	} else if (addr == 0x7F) {
		return rtcGet();
	} else if ((addr & 0xF0) == 0x90 && port[(z80.pc >> 14) | 0xB0] == NL_SEGMENT) {
		return netlinkTrapRead(addr & 0xF);
	} else {
		if (DEBUG_PORT)
			debug("Reading undecoded port 0x" + addr.toString(16) + " [PC=0x" + z80.pc.toString() + "]");
		return 0xFF; // unused port, or not defined for reading
	}
}


function writeport(addr, val) {
	addr &= ((addr & 0xF0) == 0x80) ? 0xF3 : 0xFF;
	port[addr] = val;
	if (addr == 0xB0)
		pageBases[0] = val << 14;
	else if (addr == 0xB1)
		pageBases[1] = val << 14;
	else if (addr == 0xB2)
		pageBases[2] = val << 14;
	else if (addr == 0xB3)
		pageBases[3] = val << 14;
	else if (addr == 0xB4)
		daveConfigureInterrupts(val);
	else if (addr == 0xB5) {
		keySel = ((val & 15) < 10) ? (val & 15) : -1;
	} else if (addr == 0x80) {
		setBIAS(val);
	} else if (addr == 0x81) {
		setBORDER(val);
	} else if (addr == 0x82) {
		//setLPT(false);
		setLPT(!(port[0x83] & 128));
	} else if (addr == 0x83) {
		setLPT(!(val & 128));
	} else if ((addr & 0xF0) == 0x10) {
		exdosPortWrite(addr, val);
	} else if (addr == 0x7E) {
		rtcSetReg(val);
	} else if (addr == 0x7F) {
		rtcSet(val);
	} else if (addr == 0xBF) {
		NICK_SLOTS_PER_DAVE_TICK = (val & 2) ? NICK_SLOTS_PER_DAVE_TICK_LO : NICK_SLOTS_PER_DAVE_TICK_HI;
		debug("IO: Dave: 0xBF timing nick slots = " + NICK_SLOTS_PER_DAVE_TICK + " (" + (NICK_SLOTS_PER_SEC / NICK_SLOTS_PER_DAVE_TICK) + "Hz)");
	} else if ((addr & 0xF0) == 0x90 && port[(z80.pc >> 14) | 0xB0] == NL_SEGMENT) {
		netlinkTrapWrite(addr & 0xF, val);
	} else {
		if (DEBUG_PORT && (addr < 0xA0 || addr > 0xBF))
			debug("Writing undecoded port 0x" + addr.toString(16) + " with value 0x" + val.toString(16) + " [PC=0x" + z80.pc.toString(16) + "]");
	}
}

var daveTickCounter = 0;
var audioRenderingCounter = 0;


function emulatorIteration() {
	if (!running) return;
	tstates = 0;
	refresh = false;
	event_next_event = 0;
	while (!refresh) {
		nickDoSlot();
		if ((--daveTickCounter) <= 0) {
			daveTickCounter += NICK_SLOTS_PER_DAVE_TICK;
			daveTick();
		}
		event_next_event += CPU_STATES_PER_NICK_SLOT;
		z80_do_opcodes();
		if (daveInterruptR & 0xAA) // if any of the dave interrupt latches is set, activate Z80 interrupt
			z80_interrupt();
		if (SOUND) {
			if ((--audioRenderingCounter) <= 0) {
				audioRenderingCounter += NICK_SLOTS_PER_AUDIO_SAMPLE;
				daveRenderAudioSample();
			}
		}
	}
	return tstates;
}


var statCounter = 0;


function emulatorStat(fps, timeout_set, timeout_all) {
	if (!statCounter) {
		document.getElementById("timeout").innerHTML = timeout_set + "/" + timeout_all;
		document.getElementById("fps").innerHTML = fps;
		diskStat();
		statCounter = 5;
		//daveAudioBufferRPos = daveAudioBufferWPos;
	} else
		statCounter--;
}


function emulator() {
	if (!running) return;
	var t = Date.now();
	frameSkip = true;
	var cycs = emulatorIteration();
	frameSkip = false;
	cycs += emulatorIteration();
	//if (pixAddr >= PIXADDR_HARD_LIMIT) pixAddr = PIXADDR_INITIAL;
	var td = Date.now() - t;
	t = ((cycs / CPU_CLOCK_1000) - td);
	if (t < 1 || TURBO_SPEED) t = 1;
	emulatorStat((1000 / td) | 0, t | 0, (cycs / CPU_CLOCK_1000) | 0);
	//document.getElementById("timeout").innerHTML = t + "/" + ((cycs / 4000) | 0);
	//document.getElementById("fps").innerHTML = (1000 / td) | 0;
	if (running) setTimeout(emulator, t);
}
