/*
  AdViEmulator - AdventureVision emulator
  Copyright (C) 2012-2013  JustBurn

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <string.h>
#include "core8048.h"

const char *Core8048::InstructionName[256] = {
	//  \x02 = Immediate
	//  \x04 = Register 0~1
	//  \x05 = Register 0~7
	//  \x06 = PC[7..0]
	//  \x07 = PC[10..0]
	// 00 ~ 0F
	"NOP", "Invalid", "OUTL BUS, A", "ADD A, \x02", "JMP \x07", "EN I", "Invalid", "DEC A",
	"INS A, BUS", "IN A, P1", "IN A, P2", "Invalid", "MOVD A, P4", "MOVD A, P5", "MOVD A, P6", "MOVD A, P7",
	// 10 ~ 1F
	"INC @R0", "INC @R1", "JB0 \x06", "ADDC A, \x02", "CALL \x07", "DIS I", "JTF \x06", "INC A",
	"INC R0", "INC R1", "INC R2", "INC R3", "INC R4", "INC R5", "INC R6", "INC R7",
	// 20 ~ 2F
	"XCH A, @R0", "XCH A, @R1", "Invalid", "MOV A, \x02", "JMP \x07", "EN TCNTI", "JNT0 \x06", "CLR A",
	"XCH A, R0", "XCH A, R1", "XCH A, R2", "XCH A, R3", "XCH A, R4", "XCH A, R5", "XCH A, R6", "XCH A, R7",
	// 30 ~ 3F
	"XCHD A, @R0", "XCHD A, @R1", "JB1 \x06", "Invalid", "CALL \x07", "DIS TCNTI", "JT0 \x06", "CPL A",
	"Invalid", "OUTL P1, A", "OUTL P2, A", "Invalid", "MOVD P4, A", "MOVD P5, A", "MOVD P6, A", "MOVD P7, A",
	// 40 ~ 4F
	"ORL A, @R0", "ORL A, @R1", "MOV A, T", "ORL A, \x02", "JMP \x07", "STRT CNT", "JNT1 \x06", "SWAP A",
	"ORL A, R0", "ORL A, R1", "ORL A, R2", "ORL A, R3", "ORL A, R4", "ORL A, R5", "ORL A, R6", "ORL A, R7",
	// 50 ~ 5F
	"ANL A, @R0", "ANL A, @R1", "JB2 \x06", "ANL A, \x02", "CALL \x07", "STRT T", "JT1 \x06", "DA A",
	"ANL A, R0", "ANL A, R1", "ANL A, R2", "ANL A, R3", "ANL A, R4", "ANL A, R5", "ANL A, R6", "ANL A, R7",
	// 60 ~ 6F
	"ADD A, @R0", "ADD A, @R1", "MOV T, A", "Invalid", "JMP \x07", "STOP CNT", "Invalid", "RRC A",
	"ADD A, R0", "ADD A, R1", "ADD A, R2", "ADD A, R3", "ADD A, R4", "ADD A, R5", "ADD A, R6", "ADD A, R7",
	// 70 ~ 7F
	"ADDC A, @R0", "ADDC A, @R1", "JB3 \x06", "Invalid", "CALL \x07", "ENT0 CLK", "JF1 \x06", "RR A",
	"ADDC A, R0", "ADDC A, R1", "ADDC A, R2", "ADDC A, R3", "ADDC A, R4", "ADDC A, R5", "ADDC A, R6", "ADDC A, R7",
	// 80 ~ 8F
	"MOVX A, @R0", "MOVX A, @R1", "Invalid", "RET", "JMP \x07", "CLR F0", "JNI \x06", "Invalid",
	"ORL BUS, \x02", "ORL P1, \x02", "ORL P2, \x02", "Invalid", "ORLD P4, A", "ORLD P5, A", "ORLD P6, A", "ORLD P7, A",
	// 90 ~ 9F
	"MOVX @R0, A", "MOVX @R1, A", "JB4 \x06", "RETR", "CALL \x07", "CPL F0", "JNZ \x06", "CLR C",
	"ANL BUS, \x02", "ANL P1, \x02", "ANL P2, \x02", "Invalid", "ANLD P4, A", "ANLD P5, A", "ANLD P6, A", "ANLD P7, A",
	// A0 ~ AF
	"MOV @R0, A", "MOV @R1, A", "Invalid", "MOVP A, @A", "JMP \x07", "CLR F1", "Invalid", "CPL C",
	"MOV R0, A", "MOV R1, A", "MOV R2, A", "MOV R3, A", "MOV R4, A", "MOV R5, A", "MOV R6, A", "MOV R7, A",
	// B0 ~ BF
	"MOV @R0, \x02", "MOV @R1, \x02", "JB5 \x06", "JMPP @A", "CALL \x07", "CPL F1", "JF0 \x06", "Invalid",
	"MOV R0, \x02", "MOV R1, \x02", "MOV R2, \x02", "MOV R3, \x02", "MOV R4, \x02", "MOV R5, \x02", "MOV R6, \x02", "MOV R7, \x02",
	// C0 ~ CF
	"Invalid", "Invalid", "Invalid", "Invalid", "JMP \x07", "SEL RB0", "JZ \x06", "MOV A, PSW",
	"DEC R0", "DEC R1", "DEC R2", "DEC R3", "DEC R4", "DEC R5", "DEC R6", "DEC R7",
	// D0 ~ DF
	"XRL A, @R0", "XRL A, @R1", "JB6 \x02", "XRL A, \x02", "CALL \x07", "SEL RB1", "Invalid", "MOV PSW, A",
	"XRL A, R0", "XRL A, R1", "XRL A, R2", "XRL A, R3", "XRL A, R4", "XRL A, R5", "XRL A, R6", "XRL A, R7",
	// E0 ~ EF
	"Invalid", "Invalid", "Invalid", "MOVP3 A, @A", "JMP \x07", "SEL MB0", "JNC \x06", "RL A",
	"DJNZ R0, \x06", "DJNZ R1, \x06", "DJNZ R2, \x06", "DJNZ R3, \x06", "DJNZ R4, \x06", "DJNZ R5, \x06", "DJNZ R6, \x06", "DJNZ R7, \x06",
	// F0 ~ FF
	"MOV A, @R0", "MOV A, @R1", "JB7 \x02", "Invalid", "CALL \x07", "SEL MB1", "JC \x06", "RLC A",
	"MOV A, R0", "MOV A, R1", "MOV A, R2", "MOV A, R3", "MOV A, R4", "MOV A, R5", "MOV A, R6", "MOV A, R7"
};

const uint8_t Core8048::InstructionSize[256] = {
  //0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F
	1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 00 ~ 0F
	1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 10 ~ 1F
	1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 20 ~ 2F
	1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 30 ~ 3F
	1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 40 ~ 4F
	1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 50 ~ 5F
	1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 60 ~ 6F
	1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 70 ~ 7F
	1, 1, 1, 1, 2, 1, 2, 1, 2, 2, 2, 2, 1, 1, 1, 1,	// 80 ~ 8F
	1, 1, 2, 1, 2, 1, 2, 1, 2, 2, 2, 2, 1, 1, 1, 1,	// 90 ~ 9F
	1, 1, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// A0 ~ AF
	2, 2, 2, 1, 2, 1, 2, 1, 2, 2, 2, 2, 2, 2, 2, 2,	// B0 ~ BF
	1, 1, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// C0 ~ CF
	1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// D0 ~ DF
	2, 2, 1, 1, 2, 1, 2, 1, 2, 2, 2, 2, 2, 2, 2, 2,	// E0 ~ EF
	1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1	// F0 ~ FF
};

const uint8_t Core8048::InstructionCycle[256] = {
  //0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F
	1, 1, 2, 2, 2, 1, 2, 1, 2, 2, 2, 2, 2, 2, 2, 2,	// 00 ~ 0F
	1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 10 ~ 1F
	1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 20 ~ 2F
	1, 1, 2, 2, 2, 1, 2, 1, 2, 2, 2, 2, 2, 2, 2, 2,	// 30 ~ 3F
	1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 40 ~ 4F
	1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 50 ~ 5F
	1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 60 ~ 6F
	1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 70 ~ 7F
	2, 2, 1, 2, 2, 1, 2, 1, 2, 2, 2, 2, 2, 2, 2, 2,	// 80 ~ 8F
	2, 2, 2, 2, 2, 1, 2, 1, 2, 2, 2, 2, 2, 2, 2, 2,	// 90 ~ 9F
	1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// A0 ~ AF
	2, 2, 2, 2, 2, 1, 2, 1, 2, 2, 2, 2, 2, 2, 2, 2,	// B0 ~ BF
	1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// C0 ~ CF
	1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// D0 ~ DF
	1, 1, 1, 2, 2, 1, 2, 1, 2, 2, 2, 2, 2, 2, 2, 2,	// E0 ~ EF
	1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1	// F0 ~ FF
};

void Core8048::GetInstructionName(const uint8_t *dopc, uint16_t addr, char *sout)
{
	uint8_t IR = dopc[0];
	const char *strtemp = InstructionName[IR];
	int strtemp_len = strlen(strtemp);
	for (int i=0; i<strtemp_len; i++) {
		switch (strtemp[i]) {
		case 2:	// Immediate
			{
				uint8_t IR2 = dopc[1];
				sout += sprintf(sout, "$%02X", IR2);
			} break;
		case 4:	// Register 0~1
			{
				sout += sprintf(sout, "R%i", IR & 1);
			} break;
		case 5:	// Register 0~7
			{
				sout += sprintf(sout, "R%i", IR & 7);
			} break;
		case 6:	// PC[7..0]
			{
				uint8_t IR2 = dopc[1];
				sout += sprintf(sout, "$%03X", (addr & 0xF00) | IR2);
			} break;
		case 7:	// PC[10..0]
			{
				uint8_t IR2 = dopc[1];
				sout += sprintf(sout, "$%03X", (addr & 0x800) | ((IR & 0xE0) << 3) | IR2);
			} break;
		default: // Regular character
			*sout++ = strtemp[i];
		}
	}
	*sout = 0;
}

void Core8048::GetInstructionName(uint16_t addr, char *sout)
{
	if (addr >= 4096) addr = 4095;
	int IR = OnDecodeROM(addr);
	const char *strtemp = InstructionName[IR];
	int strtemp_len = strlen(strtemp);
	for (int i=0; i<strtemp_len; i++) {
		switch (strtemp[i]) {
		case 2:	// Immediate
			{
				uint8_t IR2 = OnDecodeROM(addr + 1);
				sout += sprintf(sout, "$%02X", IR2);
			} break;
		case 4:	// Register 0~1
			{
				sout += sprintf(sout, "R%i", IR & 1);
			} break;
		case 5:	// Register 0~7
			{
				sout += sprintf(sout, "R%i", IR & 7);
			} break;
		case 6:	// PC[7..0]
			{
				uint8_t IR2 = OnDecodeROM(addr + 1);
				sout += sprintf(sout, "$%03X", (addr & 0xF00) | IR2);
			} break;
		case 7:	// PC[10..0]
			{
				uint8_t IR2 = OnDecodeROM(addr + 1);
				sout += sprintf(sout, "$%03X", (addr & 0x800) | ((IR & 0xE0) << 3) | IR2);
			} break;
		default: // Regular character
			*sout++ = strtemp[i];
		}
	}
	*sout = 0;
}

int Core8048::GetInstructionSize(uint16_t addr)
{
	if (addr >= 4096) addr = 4095;
	return InstructionSize[OnDecodeROM(addr)];
}

int Core8048::GetInstructionCycles(uint16_t addr)
{
	if (addr >= 4096) addr = 4095;
	return InstructionCycle[OnDecodeROM(addr)];
}

// PSW:
// Bit 0~2 = Stack Pointer
// Bit 3   = Always 1
// Bit 4   = Register Bank Select
// Bit 5   = Flag 0
// Bit 6   = Auxiliar Carry
// Bit 7   = Carry

Core8048::Core8048()
{
	Reset();
}

void Core8048::Reset(void)
{
	accu = 0;
	psw.w = 0x008;
	pc.w = 0;
	ff.w = 0;
	tmrsel = 0;
	gio_t1 = 0;
}

inline int Core8048::ExecSingle_Internal(void)
{
	uint8_t IR = ReadPC();
	switch (IR) {
	case 0x23:
		{	// MOV A, =data
			accu = ReadPC();
		} return 2;
	case 0xF8: case 0xF9: case 0xFA: case 0xFB:
	case 0xFC: case 0xFD: case 0xFE: case 0xFF:
		{	// MOV A, Rr
			accu = ReadReg(IR);
		} return 1;
	case 0xF0: case 0xF1:
		{	// MOV A, @Rr
			accu = ReadRegMem(IR);
		} return 1;
	case 0xC7:
		{	// MOV A, PSW
			accu = psw.b;
		} return 1;
	case 0xB8: case 0xB9: case 0xBA: case 0xBB:
	case 0xBC: case 0xBD: case 0xBE: case 0xBF:
		{	// MOV Rr, =data
			WriteReg(IR, ReadPC());
		} return 2;
	case 0xA8: case 0xA9: case 0xAA: case 0xAB:
	case 0xAC: case 0xAD: case 0xAE: case 0xAF:
		{	// MOV Rr, A
			WriteReg(IR, accu);
		} return 1;
	case 0xA0: case 0xA1:
		{	// MOV @Rr, A
			WriteRegMem(IR, accu);
		} return 1;
	case 0xB0: case 0xB1:
		{	// MOV @Rr, =data
			WriteRegMem(IR, ReadPC());
		} return 2;
	case 0xD7:
		{	// MOV PSW, A
			psw.b = accu;
		} return 1;
	case 0xA3:
		{	// MOVP A, @A
			accu = OnReadROM((pc.w & 0xF00) | accu);
		} return 2;
	case 0xE3:
		{	// MOVP3 A, @A
			accu = OnReadROM(0x300 | accu);
		} return 2;
	case 0x80: case 0x81:
		{	// MOVX A, @Rr
			accu = OnReadEx(ReadReg(IR & 1));
		} return 2;
	case 0x90: case 0x91:
		{	// MOVX @Rr, A
			OnWriteEx(ReadReg(IR & 1), accu);
		} return 2;
	case 0x28: case 0x29: case 0x2A: case 0x2B:
	case 0x2C: case 0x2D: case 0x2E: case 0x2F:
		{	// XCH A, Rr
			uint8_t tmp = accu;
			accu = ReadReg(IR);
			WriteReg(IR, tmp);
		} return 1;
	case 0x20: case 0x21:
		{	// XCH A, @Rr
			uint8_t tmp = accu;
			accu = ReadRegMem(IR);
			WriteRegMem(IR, tmp);
		} return 1;
	case 0xA7:
		{	// CPL C
			psw.c = !psw.c;
		} return 1;
	case 0x95:
		{	// CPL F0
			psw.f0 = !psw.f0;
		} return 1;
	case 0xB5:
		{	// CPL F1
			psw.f1 = !psw.f1;
		} return 1;
	case 0x97:
		{	// CLR C
			psw.c = 0;
		} return 1;
	case 0x85:
		{	// CLR F0
			psw.f0 = 0;
		} return 1;
	case 0xA5:
		{	// CLR F1
			psw.f1 = 0;
		} return 1;
	case 0x98: case 0x99: case 0x9A: case 0x9B:
		{	// ANL BUS, =data / ANL Pp, =data
			uint8_t irr = IR & 3;
			OnPortWr(irr, OnPortRd(irr) & ReadPC());
		} return 2;
	case 0x9C: case 0x9D: case 0x9E: case 0x9F:
		{	// ANLD Pp, A
			uint8_t irr = (IR & 3) + 4;
			OnPortWr(irr, OnPortRd(irr) & (accu & 0x0F));
		} return 2;
	case 0x08: case 0x09: case 0x0A: case 0x0B:
		{	// IN A, Pp / INS A, BUS
			accu = OnPortRd(IR & 3);
		} return 2;
	case 0x0C: case 0x0D: case 0x0E: case 0x0F:
		{	// MOVD A, Pp
			accu = OnPortRd((IR & 3) + 4) & 0x0F;
		} return 2;
	case 0x3C: case 0x3D: case 0x3E: case 0x3F:
		{	// MOVD Pp, A
			OnPortWr((IR & 3) + 4, accu & 0x0F);
		} return 2;
	case 0x88: case 0x89: case 0x8A: case 0x8B:
		{	// ORL BUS, =data / ORL Pp, =data
			uint8_t irr = IR & 3;
			OnPortWr(irr, OnPortRd(irr) | ReadPC());
		} return 2;
	case 0x8C: case 0x8D: case 0x8E: case 0x8F:
		{	// ORLD Pp, A
			uint8_t irr = (IR & 3) + 4;
			OnPortWr(irr, OnPortRd(irr) | (accu & 0x0F));
		} return 2;
	case 0x02:
		{	// OUTL BUS, A (warn=5)
			OnPortWr(IR & 3, accu);
		} return 2;
	case 0x38: case 0x39: case 0x3A: case 0x3B:
		{	// OUTL Pp, A
			OnPortWr(IR & 3, accu);
		} return 2;
	case 0xC8: case 0xC9: case 0xCA: case 0xCB:
	case 0xCC: case 0xCD: case 0xCE: case 0xCF:
		{	// DEC Rr
			WriteReg(IR, ReadReg(IR) - 1);
		} return 1;
	case 0x18: case 0x19: case 0x1A: case 0x1B:
	case 0x1C: case 0x1D: case 0x1E: case 0x1F:
		{	// INC Rr
			WriteReg(IR, ReadReg(IR) + 1);
		} return 1;
	case 0x10: case 0x11:
		{	// INC @Rr
			WriteRegMem(IR, ReadRegMem(IR) + 1);
		} return 1;
	case 0x14: case 0x34: case 0x54: case 0x74:
	case 0x94: case 0xB4: case 0xD4: case 0xF4:
		{	// CALL addr
			Call(((IR & 0xE0) << 3) | ReadPC());
		} return 2;
	case 0x83:
		{	// RET
			Pop(false);
		} return 2;
	case 0x93:
		{	// RETR
			Pop(true);
			ff.intf = 0;
		} return 2;
	case 0x25:
		{	// EN TCNTI
			ff.tcnti = 1;
		} return 1;
	case 0x35:
		{	// DIS TCNTI
			ff.tcnti = 0;
		} return 1;
	case 0x42:
		{	// MOV A, T
			accu = tcnt;
		} return 1;
	case 0x62:
		{	// MOV T, A
			tcnt = accu;
		} return 1;
	case 0x65:
		{	// STOP TCNT
			tmrsel = TMRSEL_OFF;
		} return 1;
	case 0x45:
		{	// STRT CNT
			tmrsel = TMRSEL_OUT;
		} return 1;
	case 0x55:
		{	// STRT T
			tmrsel = TMRSEL_IN;
			tcnt_ps = 0;
		} return 1;
	case 0x00:
		{	// NOP
		} return 1;
	case 0x03:
		{	// ADD A, =data
			ADD(ReadPC());
		} return 2;
	case 0x68: case 0x69: case 0x6A: case 0x6B:
	case 0x6C: case 0x6D: case 0x6E: case 0x6F:
		{	// ADD A, Rr
			ADD(ReadReg(IR));
		} return 1;
	case 0x60: case 0x61:
		{	// ADD A, @Rr
			ADD(ReadRegMem(IR));
		} return 1;
	case 0x13:
		{	// ADDC A, =data
			ADDC(ReadPC());
		} return 2;
	case 0x78: case 0x79: case 0x7A: case 0x7B:
	case 0x7C: case 0x7D: case 0x7E: case 0x7F:
		{	// ADDC A, Rr
			ADDC(ReadReg(IR));
		} return 1;
	case 0x70: case 0x71:
		{	// ADDC A, @Rr
			ADDC(ReadRegMem(IR));
		} return 1;
	case 0x53:
		{	// ANL A, =data
			accu = accu & ReadPC();
		} return 2;
	case 0x58: case 0x59: case 0x5A: case 0x5B:
	case 0x5C: case 0x5D: case 0x5E: case 0x5F:
		{	// ANL A, Rr
			accu = accu & ReadReg(IR);
		} return 1;
	case 0x50: case 0x51:
		{	// ANL A, @Rr
			accu = accu & ReadRegMem(IR);
		} return 1;
	case 0x37:
		{	// CPL A
			accu = ~accu;
		} return 1;
	case 0x27:
		{	// CLR A
			accu = 0;
		} return 1;
	case 0x57:
		{	// DA A
			uint16_t res = accu;
			if (((res & 0x0F) > 0x09) || psw.ac) res += 0x06;
			if (((res & 0xF0) > 0x90) || psw.c) res += 0x60;
			accu = res & 0xFF;
			psw.c = (res >= 256);
		} return 1;
	case 0x07:
		{	// DEC A
			accu--;
		} return 1;
	case 0x17:
		{	// INC A
			accu++;
		} return 1;
	case 0x43:
		{	// ORL A, =data
			accu = accu | ReadPC();
		} return 2;
	case 0x48: case 0x49: case 0x4A: case 0x4B:
	case 0x4C: case 0x4D: case 0x4E: case 0x4F:
		{	// ORL A, Rr
			accu = accu | ReadReg(IR);
		} return 1;
	case 0x40: case 0x41:
		{	// ORL A, @Rr
			accu = accu | ReadRegMem(IR);
		} return 1;
	case 0xE7:
		{	// RL A
			accu = (accu << 1) | (accu & 0x80 ? 0x01 : 0);
		} return 1;
	case 0xF7:
		{	// RLC A
			uint8_t c = psw.c ? 0x01 : 0;
			psw.c = (accu & 0x80) ? 1 : 0;
			accu = (accu << 1) | c;
		} return 1;
	case 0x77:
		{	// RR A
			accu = (accu >> 1) | (accu & 0x01 ? 0x80 : 0);
		} return 1;
	case 0x67:
		{	// RCC A
			uint8_t c = psw.c ? 0x80 : 0;
			psw.c = (accu & 0x01) ? 1 : 0;
			accu = (accu >> 1) | c;
		} return 1;
	case 0x47:
		{	// SWAP A
			accu = (accu >> 4) | (accu << 4);
		} return 1;
	case 0x30: case 0x31:
		{	// XCHD A, @Rr
			uint8_t tmp = accu;
			accu = ReadRegMem(IR);
			WriteRegMem(IR, (tmp >> 4) | (tmp << 4));
		} return 1;
	case 0xD3:
		{	// XRL A, =data
			accu = accu ^ ReadPC();
		} return 2;
	case 0xD8: case 0xD9: case 0xDA: case 0xDB:
	case 0xDC: case 0xDD: case 0xDE: case 0xDF:
		{	// XRL A, Rr
			accu = accu ^ ReadReg(IR);
		} return 1;
	case 0xD0: case 0xD1:
		{	// XRL A, @Rr
			accu = accu ^ ReadRegMem(IR);
		} return 1;
	case 0xE8: case 0xE9: case 0xEA: case 0xEB:
	case 0xEC: case 0xED: case 0xEE: case 0xEF:
		{	// DJNZ Rr, addr
			uint8_t jp = ReadPC();
			uint8_t rr = ReadReg(IR) - 1;
			WriteReg(IR, rr);
			if (rr != 0) {
				Jump8(jp);
			}
		} return 2;
	case 0x12: case 0x32: case 0x52: case 0x72:
	case 0x92: case 0xB2: case 0xD2: case 0xF2:
		{	// JBb addr
			uint8_t jp = ReadPC();
			int bit = IR >> 5;
			if (accu & (1 << bit)) {
				Jump8(jp);
			}
		} return 2;
	case 0xF6:
		{	// JC addr
			uint8_t jp = ReadPC();
			if (psw.c) {
				Jump8(jp);
			}
		} return 2;
	case 0xB6:
		{	// JF0 addr
			uint8_t jp = ReadPC();
			if (psw.f0) {
				Jump8(jp);
			}
		} return 2;
	case 0x76:
		{	// JF1 addr
			uint8_t jp = ReadPC();
			if (psw.f1) {
				Jump8(jp);
			}
		} return 2;
	case 0x04: case 0x24: case 0x44: case 0x64:
	case 0x84: case 0xA4: case 0xC4: case 0xE4:
		{	// JMP addr
			Jump(((IR & 0xE0) << 3) | ReadPC());
		} return 2;
	case 0xB3:
		{	// JMPP @A
			pc.w &= 0xF00;
			pc.low = OnReadROM(pc.w | accu);
		} return 2;
	case 0xE6:
		{	// JNC addr
			uint8_t jp = ReadPC();
			if (!psw.c) {
				Jump8(jp);
			}
		} return 2;
	case 0x86:
		{	// JNI addr
			ReadPC();
		} return 2;
	case 0x26:
		{	// JNT0 addr
			ReadPC();
		} return 2;
	case 0x46:
		{	// JNT1 addr
			uint8_t jp = ReadPC();
			if (!gio_t1) {
				Jump8(jp);
			}
		} return 2;
	case 0x96:
		{	// JNZ addr
			uint8_t jp = ReadPC();
			if (accu != 0) {
				Jump8(jp);
			}
		} return 2;
	case 0x16:
		{	// JTF addr
			uint8_t jp = ReadPC();
			if (ff.tf) {
				ff.tf = 0;
				Jump8(jp);
			}
		} return 2;
	case 0x36:
		{	// JT0 addr
			uint8_t jp = ReadPC();
			Jump8(jp);
		} return 2;
	case 0x56:
		{	// JT1 addr
			uint8_t jp = ReadPC();
			if (gio_t1) {
				Jump8(jp);
			} else framecnt++;
		} return 2;
	case 0xC6:
		{	// JZ addr
			uint8_t jp = ReadPC();
			if (accu == 0) {
				Jump8(jp);
			}
		} return 2;
	case 0x05:
		{	// EN I
		} return 1;
	case 0x15:
		{	// DIS I
		} return 1;
	case 0x75:
		{	// ENT0 CLK
		} return 1;
	case 0xE5:
		{	// SEL MB0
			psw.mb = 0;
		} return 1;
	case 0xF5:
		{	// SEL MB1
			psw.mb = 1;
		} return 1;
	case 0xC5:
		{	// SEL RB0
			psw.rb = 0;
		} return 1;
	case 0xD5:
		{	// SEL RB1
			psw.rb = 1;
		} return 1;
	default:
		OnException(INVALID_INSTRUCTION, IR);
		return 1;
	}
}

uint8_t Core8048::ReadPC(void)
{
	uint8_t res = OnReadROM(pc.w);
	if (ff.intf) {
		pc.w = ((pc.w + 1) & 0x7FF);
	} else {
		pc.w = (pc.w & 0x800) | ((pc.w + 1) & 0x7FF);
	}
	return res;
}

uint8_t Core8048::ReadReg(int reg)
{
	if (psw.rb) {
		return OnReadRAM((reg & 7) + 24);
	} else {
		return OnReadRAM(reg & 7);
	}
}

void Core8048::WriteReg(int reg, uint8_t value)
{
	if (psw.rb) {
		OnWriteRAM((reg & 7) + 24, value);
	} else {
		OnWriteRAM(reg & 7, value);
	}
}

uint8_t Core8048::ReadRegMem(int reg)
{
	if (psw.rb) {
		return OnReadRAM(OnReadRAM((reg & 1) + 24));
	} else {
		return OnReadRAM(OnReadRAM(reg & 1));
	}
}

void Core8048::WriteRegMem(int reg, uint8_t value)
{
	if (psw.rb) {
		OnWriteRAM(OnReadRAM((reg & 1) + 24), value);
	} else {
		OnWriteRAM(OnReadRAM(reg & 1), value);
	}
}

void Core8048::Push(void)
{
	OnWriteRAM(psw.stack * 2 + 8, pc.low);
	OnWriteRAM(psw.stack * 2 + 9, (pc.high & 0x0F) | (psw.b & 0xF0));
	psw.stack++;
}

void Core8048::Pop(bool restorepsw)
{
	psw.stack--;
	uint8_t stk2 = OnReadRAM(psw.stack * 2 + 9);
	pc.low = OnReadRAM(psw.stack * 2 + 8);
	pc.high = (stk2 & 0x0F);
	if (restorepsw) psw.b = (stk2 & 0xF0) | (psw.b & 0x0F);
}

void Core8048::Jump(int addr)
{
	pc.w = (psw.mb ? 0x800 : 0x000) | (addr & 0x7FF);
}

void Core8048::Jump8(uint8_t addr)
{
	pc.w = ((pc.w - 1) & 0xF00) | addr;
}

void Core8048::Call(int addr)
{
	Push();
	pc.w = (addr & 0x7FF) | (psw.mb ? 0x800 : 0x000);
}

void Core8048::ADD(uint8_t value)
{
	uint16_t res = (uint16_t)accu + value;
	uint8_t res4 = (accu & 15) + (value & 15);
	psw.c = (res >= 256);
	psw.ac = (res4 >= 16);
	accu = res & 0xFF;
}

void Core8048::ADDC(uint8_t value)
{
	uint16_t res = (uint16_t)accu + value + (psw.c ? 1 : 0);
	uint8_t res4 = (accu & 15) + (value & 15) + (psw.c ? 1 : 0);
	psw.c = (res >= 256);
	psw.ac = (res4 >= 16);
	accu = res & 0xFF;
}

int Core8048::ExecSingle()
{
	switch (tmrsel) {
	case TMRSEL_OUT:
		if (gio_t1Last & !gio_t1) {
			tcnt++;
			if ((tcnt == 0) && (ff.tcnti) && (!ff.intf)) {
				ff.intf = 1;
				Call(0x007);
				return 2;
			}
		}
		gio_t1Last = gio_t1;
		break;
	case TMRSEL_IN:
		if (tcnt_ps >= 32) {
			tcnt_ps -= 32;
			tcnt++;
			if ((tcnt == 0) && (ff.tcnti) && (!ff.intf)) {
				ff.intf = 1;
				Call(0x007);
				return 2;
			}
		}
		break;
	}
	int cyc = ExecSingle_Internal();
	tcnt_ps += cyc;
	return cyc;
}
