/*
  AdViEmulator - AdventureVision emulator
  Copyright (C) 2012-2013  JustBurn

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <string.h>
#include "corecop411.h"

const char *CoreCOP411::InstructionName[256] = {
	//  \x03 = Immediate hexadecimal
	//  \x08 = 10nnnnnn
	//  \x09 = 11nnnnnn
	//  \x0A = $23 Expanded
	//  \x0B = $33 Expanded
	// 00 ~ 0F
	"CLRA", "SKMBZ 0", "XOR", "SKMBZ 2", "XIS 0", "LD 0", "X 0", "XDS 0",
	"LBI 0, 9", "LBI 0, 10", "LBI 0, 11", "LBI 0, 12", "LBI 0, 13", "LBI 0, 14", "LBI 0, 15", "LBI 0, 0",
	// 10 ~ 1F
	"Invalid", "SKMBZ 1", "Invalid", "SKMBZ 3", "XIS 1", "LD 1", "X 1", "XDS 1",
	"LBI 1, 9", "LBI 1, 10", "LBI 1, 11", "LBI 1, 12", "LBI 1, 13", "LBI 1, 14", "LBI 1, 15", "LBI 1, 0",
	// 20 ~ 2F
	"SKC", "SKE", "SC", "\x0A", "XIS 2", "LD 2", "X 2", "XDS 2",
	"LBI 2, 9", "LBI 2, 10", "LBI 2, 11", "LBI 2, 12", "LBI 2, 13", "LBI 2, 14", "LBI 2, 15", "LBI 2, 0",
	// 30 ~ 3F
	"ASC", "ADD", "RC", "\x0B", "XIS 3", "LD 3", "X 3", "XDS 3",
	"LBI 3, 9", "LBI 3, 10", "LBI 3, 11", "LBI 3, 12", "LBI 3, 13", "LBI 3, 14", "LBI 3, 15", "LBI 3, 0",
	// 40 ~ 4F
	"COMP", "Invalid", "RMB 2", "RMB 3", "NOP", "RMB 1", "SMB 2", "SMB 1",
	"RET", "RETSK", "Invalid", "SMB 3", "RMB 0", "SMB 0", "CBA", "XAS",
	// 50 ~ 5F
	"CAB", "AISC #$1", "AISC #$2", "AISC #$3", "AISC #$4", "AISC #$5", "AISC #$6", "AISC #$7",
	"AISC #$8", "AISC #$9", "AISC #$A", "AISC #$B", "AISC #$C", "AISC #$D", "AISC #$E", "AISC #$F",
	// 60 ~ 6F
	"JMP $0\x03", "JMP $1\x03", "Invalid", "Invalid", "Invalid", "Invalid", "Invalid", "Invalid",
	"JSR $0\x03", "JSR $1\x03", "Invalid", "Invalid", "Invalid", "Invalid", "Invalid", "Invalid",
	// 70 ~ 7F
	"STII #$0", "STII #$1", "STII #$2", "STII #$3", "STII #$4", "STII #$5", "STII #$6", "STII #$7",
	"STII #$8", "STII #$9", "STII #$A", "STII #$B", "STII #$C", "STII #$D", "STII #$E", "STII #$F",
	// 80 ~ 8F
	"\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08",
	"\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08",
	// 90 ~ 9F
	"\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08",
	"\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08",
	// A0 ~ AF
	"\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08",
	"\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08",
	// B0 ~ BF
	"\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08",
	"\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "\x08", "LQDI",
	// C0 ~ CF
	"\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09",
	"\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09",
	// D0 ~ DF
	"\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09",
	"\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09",
	// E0 ~ EF
	"\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09",
	"\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09",
	// F0 ~ FF
	"\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09",
	"\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "\x09", "JID"
};

const uint8_t CoreCOP411::InstructionSize[256] = {
  //0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 00 ~ 0F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 10 ~ 1F
	1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 20 ~ 2F
	1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 30 ~ 3F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 40 ~ 4F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 50 ~ 5F
	2, 2, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1,	// 60 ~ 6F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 70 ~ 7F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 80 ~ 8F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 90 ~ 9F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// A0 ~ AF
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// B0 ~ BF
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// C0 ~ CF
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// D0 ~ DF
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// E0 ~ EF
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1	// F0 ~ FF
};

const uint8_t CoreCOP411::InstructionCycle[256] = {
  //0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 00 ~ 0F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 10 ~ 1F
	1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 20 ~ 2F
	1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 30 ~ 3F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 40 ~ 4F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 50 ~ 5F
	2, 2, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1,	// 60 ~ 6F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 70 ~ 7F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 80 ~ 8F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// 90 ~ 9F
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// A0 ~ AF
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2,	// B0 ~ BF
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// C0 ~ CF
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// D0 ~ DF
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	// E0 ~ EF
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2	// F0 ~ FF
};

void CoreCOP411::GetInstructionName(const uint8_t *dopc, uint16_t addr, char *sout)
{
	uint8_t IR = dopc[0];
	const char *strtemp = InstructionName[IR];
	int strtemp_len = strlen(strtemp);
	for (int i=0; i<strtemp_len; i++) {
		switch (strtemp[i]) {
		case 3:	// Immediate hexadecimal
			{
				uint8_t IR2 = dopc[1];
				sout += sprintf(sout, "%02X", IR2);
			} break;
		case 8:	// 10nnnnnn
			{
				addr++;
				if ((addr >= 0x080) && (addr < 0x100)) {
					sout += sprintf(sout, "JP $%03X", (addr & 0x380) | (IR & 0x7F));
				} else {
					sout += sprintf(sout, "JSRP $%03X", 0x80 | (IR & 0x3F));
				}
			} break;
		case 9:	// 11nnnnnn
			{
				addr++;
				if ((addr >= 0x080) && (addr < 0x100)) {
					sout += sprintf(sout, "JP $%03X", (addr & 0x380) | (IR & 0x7F));
				} else {
					sout += sprintf(sout, "JP $%03X", (addr & 0x3C0) | (IR & 0x3F));
				}
			} break;
		case 10: // $23 Expanded
			{
				uint8_t IR2 = dopc[1];
				if ((IR2 >= 0x80) && (IR2 < 0xC0)) {
					sout += sprintf(sout, "XAD %x, %x", ((IR2 & 0x30) >> 4), IR2 & 0x0F);
				} else {
					sout += sprintf(sout, "Invalid");
				}
			} break;
		case 11: // $33 Expanded
			{
				uint8_t IR2 = dopc[1];
				switch (IR2) {
				case 0x60: case 0x61: case 0x62: case 0x63:
				case 0x64: case 0x65: case 0x66: case 0x67:
				case 0x68: case 0x69: case 0x6A: case 0x6B:
				case 0x6C: case 0x6D: case 0x6E: case 0x6F:
					sout += sprintf(sout, "LEI %x", IR2 & 0x0F);
					break;
				case 0x01:
					sout += sprintf(sout, "SKGBZ 0");
					break;
				case 0x03:
					sout += sprintf(sout, "SKGBZ 2");
					break;
				case 0x11:
					sout += sprintf(sout, "SKGBZ 1");
					break;
				case 0x13:
					sout += sprintf(sout, "SKGBZ 3");
					break;
				case 0x21:
					sout += sprintf(sout, "SKGZ");
					break;
				case 0x2A:
					sout += sprintf(sout, "ING");
					break;
				case 0x2E:
					sout += sprintf(sout, "INL");
					break;
				case 0x38:
					sout += sprintf(sout, "HALT");
					break;
				case 0x3A:
					sout += sprintf(sout, "OMG");
					break;
				case 0x3C:
					sout += sprintf(sout, "CAMQ");
					break;
				case 0x3E:
					sout += sprintf(sout, "OBD");
					break;
				default:
					sout += sprintf(sout, "Invalid");
					break;
				}
			} break;
		default: // Regular character
			*sout++ = strtemp[i];
			break;
		}
	}
	*sout = 0;
}

void CoreCOP411::GetInstructionName(uint16_t addr, char *sout)
{
	if (addr >= 4096) addr = 4095;
	int IR = OnDecodeROM(addr);
	const char *strtemp = InstructionName[IR];
	int strtemp_len = strlen(strtemp);
	for (int i=0; i<strtemp_len; i++) {
		switch (strtemp[i]) {
		case 3:	// Immediate hexadecimal
			{
				uint8_t IR2 = OnDecodeROM(addr + 1);
				sout += sprintf(sout, "%02X", IR2);
			} break;
		case 8:	// 10nnnnnn
			{
				addr++;
				if ((addr >= 0x080) && (addr < 0x100)) {
					sout += sprintf(sout, "JP $%03X", (addr & 0x380) | (IR & 0x7F));
				} else {
					sout += sprintf(sout, "JSRP $%03X", 0x80 | (IR & 0x3F));
				}
			} break;
		case 9:	// 11nnnnnn
			{
				addr++;
				if ((addr >= 0x080) && (addr < 0x100)) {
					sout += sprintf(sout, "JP $%03X", (addr & 0x380) | (IR & 0x7F));
				} else {
					sout += sprintf(sout, "JP $%03X", (addr & 0x3C0) | (IR & 0x3F));
				}
			} break;
		case 10: // $23 Expanded
			{
				uint8_t IR2 = OnDecodeROM(addr + 1);
				if ((IR2 >= 0x80) && (IR2 < 0xC0)) {

					sout += sprintf(sout, "XAD %x, %x", ((IR2 & 0x30) >> 4), IR2 & 0x0F);
				} else {
					sout += sprintf(sout, "Invalid");
				}
			} break;
		case 11: // $33 Expanded
			{
				uint8_t IR2 = OnDecodeROM(addr + 1);
				switch (IR2) {
				case 0x60: case 0x61: case 0x62: case 0x63:
				case 0x64: case 0x65: case 0x66: case 0x67:
				case 0x68: case 0x69: case 0x6A: case 0x6B:
				case 0x6C: case 0x6D: case 0x6E: case 0x6F:
					sout += sprintf(sout, "LEI %x", IR2 & 0x0F);
					break;
				case 0x01:
					sout += sprintf(sout, "SKGBZ 0");
					break;
				case 0x03:
					sout += sprintf(sout, "SKGBZ 2");
					break;
				case 0x11:
					sout += sprintf(sout, "SKGBZ 1");
					break;
				case 0x13:
					sout += sprintf(sout, "SKGBZ 3");
					break;
				case 0x21:
					sout += sprintf(sout, "SKGZ");
					break;
				case 0x2A:
					sout += sprintf(sout, "ING");
					break;
				case 0x2E:
					sout += sprintf(sout, "INL");
					break;
				case 0x38:
					sout += sprintf(sout, "HALT");
					break;
				case 0x3A:
					sout += sprintf(sout, "OMG");
					break;
				case 0x3C:
					sout += sprintf(sout, "CAMQ");
					break;
				case 0x3E:
					sout += sprintf(sout, "OBD");
					break;
				default:
					sout += sprintf(sout, "Invalid");
					break;
				}
			} break;
		default: // Regular character
			*sout++ = strtemp[i];
			break;
		}
	}
	*sout = 0;
}

int CoreCOP411::GetInstructionSize(uint16_t addr)
{
	if (addr >= 511) addr = 511;
	return InstructionSize[OnDecodeROM(addr)];
}

int CoreCOP411::GetInstructionCycles(uint16_t addr)
{
	if (addr >= 511) addr = 511;
	return InstructionCycle[OnDecodeROM(addr)];
}

CoreCOP411::CoreCOP411()
{
	for (int i=0; i<32; i++) ram[i] = 0;
	Reset();
}

void CoreCOP411::Reset(void)
{
	pc.w = 0;
	accu = 0;
	carry = 0;
	ramb = 0;
	sa = 0;
	sb = 0;
	skipnext = false;
	skiplbi = false;
}

inline int CoreCOP411::ExecSingle_33Ex(void)
{
	uint8_t IR = ReadPC();
	switch (IR) {
	case 0x3C:
		{	// CAMQ
			qreg = (accu << 4) | RAMRead(ramb);
		} return 2;
	case 0x60: case 0x61: case 0x62: case 0x63: case 0x64: case 0x65: case 0x66: case 0x67:
	case 0x68: case 0x69: case 0x6A: case 0x6B: case 0x6C: case 0x6D: case 0x6E: case 0x6F:
		{	// LEI y
			enreg = IR & 15;
		} return 2;
	case 0x21:
		{	// SKGZ
			if (port_g == 0) skipnext = true;
		} return 2;
	case 0x01:
		{	// SKGBZ 0
			if (!(port_g & 1)) skipnext = true;
		} return 2;
	case 0x11:
		{	// SKGBZ 1
			if (!(port_g & 2)) skipnext = true;
		} return 2;
	case 0x03:
		{	// SKGBZ 2
			if (!(port_g & 4)) skipnext = true;
		} return 2;
	case 0x13:
		{	// SKGBZ 3
			if (!(port_g & 8)) skipnext = true;
		} return 2;
	case 0x2A:
		{	// ING
			accu = port_g & 15;
		} return 2;
	case 0x2E:
		{	// INL
			RAMWrite(ramb, port_l >> 4);
			accu = port_l & 15;
		} return 2;
	case 0x3E:
		{	// OBD
			port_d = ramb & 15;
		} return 2;
	case 0x3A:
		{	// OMG
			port_g = RAMRead(ramb);
		} return 2;
	default:
		OnException(INVALID_INSTRUCTION, IR);
		return 2;
	}
}

inline int CoreCOP411::ExecSingle_Internal(void)
{
	uint8_t IR = ReadPC();
	switch (IR) {
	case 0x30:
		{	// ASC
			accu = accu + carry + RAMRead(ramb);
			if (accu > 15) {
				accu &= 15;
				carry = 1;
				skipnext = true;
			} else carry = 0;
		} return 1;
	case 0x31:
		{	// ADD
			accu = (accu + RAMRead(ramb)) & 15;
		} return 1;
	case 0x51: case 0x52: case 0x53: case 0x54: case 0x55: case 0x56: case 0x57:
	case 0x58: case 0x59: case 0x5A: case 0x5B: case 0x5C: case 0x5D: case 0x5E: case 0x5F:
		{	// AISC y
			accu = accu + (IR & 15);
			if (accu > 15) {
				accu &= 15;
				skipnext = true;
			}
		} return 1;
	case 0x00:
		{	// CLRA
			accu = 0;
		} return 1;
	case 0x40:
		{	// COMP
			accu = accu ^ 15;
		} return 1;
	case 0x44:
		{	// NOP
		} return 1;
	case 0x32:
		{	// RC
			carry = 0;
		} return 1;
	case 0x22:
		{	// SC
			carry = 1;
		} return 1;
	case 0x02:
		{	// XOR
			accu = accu ^ RAMRead(ramb);
		} return 1;
	case 0xFF:
		{	// JID
			pc.low = OnReadROM((pc.w & 0x100) | (accu << 4) | RAMRead(ramb));
		} return 2;
	case 0x60: case 0x61:
		{	// JMP a
			pc.w = ((IR & 1) << 8) | ReadPC();
		} return 2;
	case 0x80: case 0x81: case 0x82: case 0x83: case 0x84: case 0x85: case 0x86: case 0x87:
	case 0x88: case 0x89: case 0x8A: case 0x8B: case 0x8C: case 0x8D: case 0x8E: case 0x8F:
	case 0x90: case 0x91: case 0x92: case 0x93: case 0x94: case 0x95: case 0x96: case 0x97:
	case 0x98: case 0x99: case 0x9A: case 0x9B: case 0x9C: case 0x9D: case 0x9E: case 0x9F:
	case 0xA0: case 0xA1: case 0xA2: case 0xA3: case 0xA4: case 0xA5: case 0xA6: case 0xA7:
	case 0xA8: case 0xA9: case 0xAA: case 0xAB: case 0xAC: case 0xAD: case 0xAE: case 0xAF:
	case 0xB0: case 0xB1: case 0xB2: case 0xB3: case 0xB4: case 0xB5: case 0xB6: case 0xB7:
	case 0xB8: case 0xB9: case 0xBA: case 0xBB: case 0xBC: case 0xBD: case 0xBE:
		{	// JP a / JSRP a
			if ((pc.w >= 0x080) && (pc.w < 0x100)) {
				pc.low = (pc.low & 0x80) | (IR & 0x7F);
			} else {
				Push();
				pc.w = 0x80 | (IR & 0x3F);

			}
		} return 2;
	case 0xC0: case 0xC1: case 0xC2: case 0xC3: case 0xC4: case 0xC5: case 0xC6: case 0xC7:
	case 0xC8: case 0xC9: case 0xCA: case 0xCB: case 0xCC: case 0xCD: case 0xCE: case 0xCF:
	case 0xD0: case 0xD1: case 0xD2: case 0xD3: case 0xD4: case 0xD5: case 0xD6: case 0xD7:
	case 0xD8: case 0xD9: case 0xDA: case 0xDB: case 0xDC: case 0xDD: case 0xDE: case 0xDF:
	case 0xE0: case 0xE1: case 0xE2: case 0xE3: case 0xE4: case 0xE5: case 0xE6: case 0xE7:
	case 0xE8: case 0xE9: case 0xEA: case 0xEB: case 0xEC: case 0xED: case 0xEE: case 0xEF:
	case 0xF0: case 0xF1: case 0xF2: case 0xF3: case 0xF4: case 0xF5: case 0xF6: case 0xF7:
	case 0xF8: case 0xF9: case 0xFA: case 0xFB: case 0xFC: case 0xFD: case 0xFE:
		{	// JP a / JP a
			if ((pc.w >= 0x080) && (pc.w < 0x100)) {
				pc.low = (pc.low & 0x80) | (IR & 0x7F);
			} else {
				pc.low = (pc.low & 0xC0) | (IR & 0x3F);
			}
		} return 2;
	case 0x68: case 0x69:
		{	// JSR a
			uint8_t IR2 = ReadPC();
			Push();
			pc.w = ((IR & 1) << 8) | IR2;
		} return 2;
	case 0x48:
		{	// RET
			Pop();
		} return 1;
	case 0x49:
		{	// RETSK
			Pop();
			skipnext = true;
		} return 1;
	case 0x33:
		{	// $33 Expand
		} return ExecSingle_33Ex();
	case 0x05: case 0x15: case 0x25: case 0x35:
		{	// LD r
			accu = RAMRead(ramb);
			ramb = ramb ^ (IR & 0x30);
		} return 1;
	case 0xBF:
		{	// LQID
			qreg = OnReadROM((pc.w & 0x100) | (accu << 4) | RAMRead(ramb)) & 15;
		} return 2;
	case 0x4C:
		{	// RMB 0
			RAMWrite(ramb, RAMRead(ramb) & ~1);
		} return 1;
	case 0x45:
		{	// RMB 1
			RAMWrite(ramb, RAMRead(ramb) & ~2);
		} return 1;
	case 0x42:
		{	// RMB 2
			RAMWrite(ramb, RAMRead(ramb) & ~4);
		} return 1;
	case 0x43:
		{	// RMB 3
			RAMWrite(ramb, RAMRead(ramb) & ~8);
		} return 1;
	case 0x4D:
		{	// SMB 0
			RAMWrite(ramb, RAMRead(ramb) | 1);
		} return 1;
	case 0x47:
		{	// SMB 1
			RAMWrite(ramb, RAMRead(ramb) | 2);
		} return 1;
	case 0x46:
		{	// SMB 2
			RAMWrite(ramb, RAMRead(ramb) | 4);
		} return 1;
	case 0x4B:
		{	// SMB 3
			RAMWrite(ramb, RAMRead(ramb) | 8);
		} return 1;
	case 0x70: case 0x71: case 0x72: case 0x73: case 0x74: case 0x75: case 0x76: case 0x77:
	case 0x78: case 0x79: case 0x7A: case 0x7B: case 0x7C: case 0x7D: case 0x7E: case 0x7F:
		{	// STII y
			RAMWrite(ramb, IR & 15);
			ramb = (ramb & 0x30) | ((ramb + 1) & 0x0F);
		} return 1;
	case 0x06: case 0x16: case 0x26: case 0x36:
		{	// X r
			uint8_t tmp = accu;
			accu = RAMRead(ramb);
			RAMWrite(ramb, tmp);
			ramb = ramb ^ (IR & 0x30);
		} return 1;
	case 0x23:
		{	// XAD r, r
			uint8_t ramp = ReadPC() & 0x7F;
			uint8_t tmp = accu;
			accu = RAMRead(ramp);
			RAMWrite(ramp, tmp);
		} return 2;
	case 0x07: case 0x17: case 0x27: case 0x37:
		{	// XDS r
			uint8_t tmp = accu;
			accu = RAMRead(ramb);
			RAMWrite(ramb, tmp);
			uint8_t blow = ((ramb - 1) & 0x0F);
			ramb = ((ramb & 0x30) ^ (IR & 0x30)) | blow;
			if (blow == 15) skipnext = true;
		} return 1;
	case 0x04: case 0x14: case 0x24: case 0x34:
		{	// XIS r
			uint8_t tmp = accu;
			accu = RAMRead(ramb);
			RAMWrite(ramb, tmp);
			uint8_t blow = ((ramb + 1) & 0x0F);
			ramb = ((ramb & 0x30) ^ (IR & 0x30)) | blow;
			if (blow == 0) skipnext = true;
		} return 1;
	case 0x50:
		{	// CAB
			ramb = (ramb & 0x30) | accu;
		} return 1;
	case 0x4E:
		{	// CBA
			accu = ramb & 15;
		} return 1;
	case 0x08: case 0x09: case 0x0A:  case 0x0B: case 0x0C: case 0x0D: case 0x0E:  case 0x0F:
	case 0x18: case 0x19: case 0x1A:  case 0x1B: case 0x1C: case 0x1D: case 0x1E:  case 0x1F:
	case 0x28: case 0x29: case 0x2A:  case 0x2B: case 0x2C: case 0x2D: case 0x2E:  case 0x2F:
	case 0x38: case 0x39: case 0x3A:  case 0x3B: case 0x3C: case 0x3D: case 0x3E:  case 0x3F:
		{	// LBI r, d
			ramb = (IR & 0x30) | ((IR + 1) & 15);
			skiplbi = true;
		} return 1;
	case 0x20:
		{	// SKC
			if (carry) skipnext = true;
		} return 1;
	case 0x21:
		{	// SKE
			if (accu == RAMRead(ramb)) skipnext = true;
		} return 1;
	case 0x01:
		{	// SKMBZ 0
			if (!(RAMRead(ramb) & 1)) skipnext = true;
		} return 2;
	case 0x11:
		{	// SKMBZ 1
			if (!(RAMRead(ramb) & 2)) skipnext = true;
		} return 2;
	case 0x03:
		{	// SKMBZ 2
			if (!(RAMRead(ramb) & 4)) skipnext = true;
		} return 2;
	case 0x13:
		{	// SKMBZ 3
			if (!(RAMRead(ramb) & 8)) skipnext = true;
		} return 2;
	case 0x4F:
		{	// XAS
			uint8_t tmp = accu;
			accu = sioreg;
			sioreg = tmp;
			skl = carry;
		} return 2;
	default:
		OnException(INVALID_INSTRUCTION, IR);
		return 1;
	}
}

uint8_t CoreCOP411::ReadPC(void)
{
	uint8_t res = OnReadROM(pc.w);
	pc.w = ((pc.w + 1) & 0x1FF);
	return res;
}

uint8_t CoreCOP411::RAMRead(uint8_t addr)
{
	return ram[addr & 31];
}

void CoreCOP411::RAMWrite(uint8_t addr, uint8_t data)
{
	ram[addr & 31] = data & 15;
}

void CoreCOP411::Push(void)
{
	sb = sa;
	sa = pc.w;
}

void CoreCOP411::Pop(void)
{
	pc.w = sa;
	sa = sb;
}

int CoreCOP411::ExecSingle()
{
	uint8_t IR = OnReadROM(pc.w);
	if (skiplbi) {
		// Skip if next instruction is a LBI
		if ((IR & 8) && (IR < 0x40)) {
			skipnext = true;
		} else skiplbi = false;
	}
	if (skipnext) {
		// Skip instruction
		skipnext = false;
		pc.w = ((pc.w + InstructionSize[IR]) & 0x1FF);
		return InstructionCycle[IR];
	}
	return ExecSingle_Internal();
}
