/*
  AdViEmulator - AdventureVision emulator
  Copyright (C) 2012-2013  JustBurn

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <QFileDialog>
#include "optionswindow.h"
#include "ui_optionswindow.h"

OptionsWindow::OptionsWindow(QWidget *parent) :
	QDialog(parent),
	ui(new Ui::OptionsWindow)
{
	ui->setupUi(this);
	connect(ui->buttonBox, SIGNAL(clicked(QAbstractButton*)), this, SLOT(clicked(QAbstractButton*)));
	connect(ui->pushCPUFirmware, SIGNAL(clicked()), this, SLOT(OnCPUBrowse()));
	connect(ui->pushAPUFirmware, SIGNAL(clicked()), this, SLOT(OnAPUBrowse()));
	connect(ui->horizontalRed, SIGNAL(valueChanged(int)), this, SLOT(OnRedChange(int)));
	connect(ui->horizontalGreen, SIGNAL(valueChanged(int)), this, SLOT(OnGreenChange(int)));
	connect(ui->horizontalBlue, SIGNAL(valueChanged(int)), this, SLOT(OnBlueChange(int)));
	connect(ui->horizontalAnalogDecay, SIGNAL(valueChanged(int)), this, SLOT(OnAnalogDecayChange(int)));
	connect(ui->horizontalVolume, SIGNAL(valueChanged(int)), this, SLOT(OnVolumeChange(int)));
	connect(ui->horizontalAPUFreq, SIGNAL(valueChanged(int)), this, SLOT(OnAPUFreqChange(int)));
	mainwin = (MainWindow *)parent;
	loadSettings();
}

OptionsWindow::~OptionsWindow()
{
	delete ui;
}

void OptionsWindow::loadSettings(void)
{
	mainwin->LoadSettings();
	ui->checkAutostart->setChecked(Settings::o.autostart);
	ui->lineCPUFirmware->setText(Settings::o.customcpu_file);
	ui->lineAPUFirmware->setText(Settings::o.customapu_file);
	ui->checkCPUFirmware->setChecked(Settings::o.customcpu_enable);
	ui->checkAPUFirmware->setChecked(Settings::o.customapu_enable);
	ui->horizontalRed->setValue(Settings::o.colorRed);
	ui->horizontalGreen->setValue(Settings::o.colorGreen);
	ui->horizontalBlue->setValue(Settings::o.colorBlue);
	ui->checkAnalogDisplay->setChecked(Settings::o.analogDisplay);
	ui->horizontalAnalogDecay->setValue(Settings::o.analogDecay);
	ui->checkColumnMode->setChecked(Settings::o.columnMode);
	ui->checkPixelBar->setChecked(Settings::o.pixelBar);
	ui->checkSoundEnabled->setChecked(Settings::o.soundEnabled);
	ui->horizontalVolume->setValue(Settings::o.volume);
	if (Settings::o.soundSamples <= 1024) ui->comboBufferSize->setCurrentIndex(0);
	else if (Settings::o.soundSamples <= 2048) ui->comboBufferSize->setCurrentIndex(1);
	else if (Settings::o.soundSamples <= 4096) ui->comboBufferSize->setCurrentIndex(2);
	else if (Settings::o.soundSamples <= 8192) ui->comboBufferSize->setCurrentIndex(3);
	else if (Settings::o.soundSamples <= 16384) ui->comboBufferSize->setCurrentIndex(4);
	else if (Settings::o.soundSamples <= 32768) ui->comboBufferSize->setCurrentIndex(5);
	if (Settings::o.apuFreq) ui->horizontalAPUFreq->setValue((2933332 / Settings::o.apuFreq) - 40);
	ui->lineInputButton1->setAssignedKey(Settings::o.keyButton1);
	ui->lineInputButton2->setAssignedKey(Settings::o.keyButton2);
	ui->lineInputButton3->setAssignedKey(Settings::o.keyButton3);
	ui->lineInputButton4->setAssignedKey(Settings::o.keyButton4);
	ui->lineInputStickDown->setAssignedKey(Settings::o.keyStickDown);
	ui->lineInputStickUp->setAssignedKey(Settings::o.keyStickUp);
	ui->lineInputStickRight->setAssignedKey(Settings::o.keyStickRight);
	ui->lineInputStickLeft->setAssignedKey(Settings::o.keyStickLeft);
}

void OptionsWindow::saveSettings(void)
{
	Settings::o.autostart = ui->checkAutostart->isChecked();
	Settings::o.customcpu_file = ui->lineCPUFirmware->text();
	Settings::o.customapu_file = ui->lineAPUFirmware->text();
	Settings::o.customcpu_enable = ui->checkCPUFirmware->isChecked();
	Settings::o.customapu_enable = ui->checkAPUFirmware->isChecked();
	Settings::o.colorRed = ui->horizontalRed->value();
	Settings::o.colorGreen = ui->horizontalGreen->value();
	Settings::o.colorBlue = ui->horizontalBlue->value();
	Settings::o.analogDisplay = ui->checkAnalogDisplay->isChecked();
	Settings::o.analogDecay = ui->horizontalAnalogDecay->value();
	Settings::o.columnMode = ui->checkColumnMode->isChecked();
	Settings::o.pixelBar = ui->checkPixelBar->isChecked();
	Settings::o.soundEnabled = ui->checkSoundEnabled->isChecked();
	Settings::o.volume = ui->horizontalVolume->value();
	Settings::o.soundSamples = (1024 << ui->comboBufferSize->currentIndex());
	Settings::o.apuFreq = 2933332 / (40 + ui->horizontalAPUFreq->value());
	Settings::o.keyButton1 = ui->lineInputButton1->getAssignedKey();
	Settings::o.keyButton2 = ui->lineInputButton2->getAssignedKey();
	Settings::o.keyButton3 = ui->lineInputButton3->getAssignedKey();
	Settings::o.keyButton4 = ui->lineInputButton4->getAssignedKey();
	Settings::o.keyStickDown = ui->lineInputStickDown->getAssignedKey();
	Settings::o.keyStickUp = ui->lineInputStickUp->getAssignedKey();
	Settings::o.keyStickRight = ui->lineInputStickRight->getAssignedKey();
	Settings::o.keyStickLeft = ui->lineInputStickLeft->getAssignedKey();
	mainwin->SaveSettings();
}

void OptionsWindow::clicked(QAbstractButton *button)
{
	int stdbutton = ui->buttonBox->standardButton(button);
	if (stdbutton == QDialogButtonBox::RestoreDefaults) {
		// Restore defaults
		ui->checkAutostart->setChecked(Settings::def.autostart);
		ui->lineCPUFirmware->setText(Settings::def.customcpu_file);
		ui->lineAPUFirmware->setText(Settings::def.customapu_file);
		ui->checkCPUFirmware->setChecked(Settings::def.customcpu_enable);
		ui->checkAPUFirmware->setChecked(Settings::def.customapu_enable);
		ui->horizontalRed->setValue(Settings::def.colorRed);
		ui->horizontalGreen->setValue(Settings::def.colorGreen);
		ui->horizontalBlue->setValue(Settings::def.colorBlue);
		ui->checkAnalogDisplay->setChecked(Settings::def.analogDisplay);
		ui->horizontalAnalogDecay->setValue(Settings::def.analogDecay);
		ui->checkColumnMode->setChecked(Settings::def.columnMode);
		ui->checkPixelBar->setChecked(Settings::def.pixelBar);
		ui->checkSoundEnabled->setChecked(Settings::def.soundEnabled);
		ui->horizontalVolume->setValue(Settings::def.volume);
		if (Settings::def.soundSamples <= 1024) ui->comboBufferSize->setCurrentIndex(0);
		else if (Settings::def.soundSamples <= 2048) ui->comboBufferSize->setCurrentIndex(1);
		else if (Settings::def.soundSamples <= 4096) ui->comboBufferSize->setCurrentIndex(2);
		else if (Settings::def.soundSamples <= 8192) ui->comboBufferSize->setCurrentIndex(3);
		else if (Settings::def.soundSamples <= 16384) ui->comboBufferSize->setCurrentIndex(4);
		else if (Settings::def.soundSamples <= 32768) ui->comboBufferSize->setCurrentIndex(5);
		ui->horizontalAPUFreq->setValue((2933332 / Settings::def.apuFreq) - 40);
		ui->lineInputButton1->setAssignedKey(Settings::def.keyButton1);
		ui->lineInputButton2->setAssignedKey(Settings::def.keyButton2);
		ui->lineInputButton3->setAssignedKey(Settings::def.keyButton3);
		ui->lineInputButton4->setAssignedKey(Settings::def.keyButton4);
		ui->lineInputStickDown->setAssignedKey(Settings::def.keyStickDown);
		ui->lineInputStickUp->setAssignedKey(Settings::def.keyStickUp);
		ui->lineInputStickRight->setAssignedKey(Settings::def.keyStickRight);
		ui->lineInputStickLeft->setAssignedKey(Settings::def.keyStickLeft);
	} else if (stdbutton == QDialogButtonBox::Ok) {
		// Ok
		saveSettings();
		close();
	} else if (stdbutton == QDialogButtonBox::Cancel) {
		// Cancel
		close();
	}
}

void OptionsWindow::OnCPUBrowse(void)
{
	QString fileName = QFileDialog::getOpenFileName(this, tr("Open CPU (8048) firmware"), "", tr("Binary (*.bin);;Files (*.*)"));
	if (fileName != NULL) {
		ui->lineCPUFirmware->setText(fileName);
		ui->checkCPUFirmware->setChecked(true);
	}
}

void OptionsWindow::OnAPUBrowse(void)
{
	QString fileName = QFileDialog::getOpenFileName(this, tr("Open APU (COP411) firmware"), "", tr("Binary (*.bin);;Files (*.*)"));
	if (fileName != NULL) {
		ui->lineAPUFirmware->setText(fileName);
		ui->checkAPUFirmware->setChecked(true);
	}
}

void OptionsWindow::OnRedChange(int value)
{
	ui->labelPixelRed->setText("Pixel Red (" + QString::number(value) + "):");
}

void OptionsWindow::OnGreenChange(int value)
{
	ui->labelPixelGreen->setText("Pixel Green (" + QString::number(value) + "):");
}

void OptionsWindow::OnBlueChange(int value)
{
	ui->labelPixelBlue->setText("Pixel Blue (" + QString::number(value) + "):");
}

void OptionsWindow::OnAnalogDecayChange(int value)
{
	ui->labelAnalogDecay->setText("Analog Decay (" + QString::number(value) + "):");
}

void OptionsWindow::OnVolumeChange(int value)
{
	ui->labelVolume->setText("Volume (" + QString::number(value * 100 / 127) + "%):");
}

void OptionsWindow::OnAPUFreqChange(int value)
{
	ui->labelAPUFreq->setText("COP411 Frequency:\n(" + QString::number(2933332 / (40 + value)) + " Hz)");
}
