#include <nall/nall.hpp>
using namespace nall;

#include <ruby/ruby.hpp>
using namespace ruby;

#include <phoenix/phoenix.hpp>
using namespace phoenix;
struct ShaderSetting {
	string name;
	string value;
	float min=0.0;
	float max=0.0;
	float step=1.0;
	bool active=true;
	bool operator< ( const ShaderSetting& source ) {
		return name <  source.name;
	}
	bool operator== ( const ShaderSetting& source ) {
		return name == source.name;
	}
	ShaderSetting() {}
	ShaderSetting ( const string& name ) : name ( name ) {}
	ShaderSetting ( const string& name, const string& value ) : name ( name ), value ( value ) {}
};
struct ZoomSettings {
	float zoomX=1.0;
	float zoomY=1.0;
	int offsetX=0;
	int offsetY=0;
};

struct CheckButton_tagged : CheckButton {
	ShaderSetting* tag;
};
struct HorizontalSlider_tagged : HorizontalSlider {
	ShaderSetting* tag;
	Label*          label;
};
struct FILEINFO {
	string fileName;
	time_t timeStamp;
};

struct MainWindow : Window {
	HorizontalLayout    layout;
	Viewport            viewport;

	VerticalLayout      marginLayoutMain;
	VerticalLayout      listLayout;
	VerticalLayout      marginLayout;

	Button              screenShotButton;
	HorizontalLayout    widthHeightEditLayout;
	LineEdit            textEditWidth;
	Label               xLabel;
	LineEdit            textEditHeight;
	Button              setSizeButton;
	CheckButton         dockCheckButton;
	HorizontalLayout    scaleButtonsLayout;
	Button              scale1xButton;
	Button              scale2xButton;
	Button              scale3xButton;
	Button              scale4xButton;
	Button              scale5xButton;
	CheckButton         aspectRatioCheckButton;

	HorizontalLayout    offsetAndZoomLayout;
//	Label               zoomLabel;
//	LineEdit            textEditZoomX;
//	LineEdit            textEditZoomY;
//	Button              setZoomButton;
	Button              resetOffsetZoomButton;



	HorizontalLayout    optionsLayout;
	CheckButton         renderOnUpdateCheckButton;
	CheckButton         vSyncCheckButton;
	CheckButton         WatchFilesCheckButton;
	CheckButton         useUniformsCheckButton;

	ListView listShaders;
	ListView listImages;

	VerticalLayout                      settingsLayout;
	vector<Label*>                      settingsLabels;
	vector<CheckButton_tagged*>         settingsCheckButtons;
	vector<HorizontalSlider_tagged*>    settingsHorizontalSlider;
	vector<HorizontalLayout*>           settingsHorizontalLayout;

	phoenix::Window		controlWindow;
	HorizontalLayout	controlWindowLayout;
	VerticalLayout		controlWindowMarginLayout;

	Label myLabel;
	image buffer;
	string screenShotFileName;
	vector<FILEINFO>    shaderFilesInfo;
	bool screenShotRequested = false;
	int updateViewport = 1;
	lstring shaders;
	lstring images;
	string messageBuffer;
	int messageDisplayDuration = 0;
	ZoomSettings* zoomSettings;
	ZoomSettings previousZoomSettings;
	MainWindow();
	void mainLoop();
	void setShader();
	void setImage();
	void displayMessage ( const char* message, int duration = 1 );
	void setViewportSize ( void );
	void scaleViewport ( int scale );
	void updatePanelDockState ( void );
	void screenShot ( void );
	enum class DragMode{ NONE , DRAG, ZOOM};
	DragMode dragMode=DragMode::NONE;
	Position startPos;
	Position currentPos;
};

MainWindow::MainWindow() {
	setTitle ( "GLSLdev" );
	setSmartGeometry ( {800, 128, 230 + 640, 480} );

	shaders = directory::folders ( ".", "*.shader" );
	shaders.prepend ( "None" );
	for ( auto & shader : shaders ) listShaders.append ( string {shader} .rtrim<1> ( ".shader/" ) );
	listShaders.setSelection ( 0 );

	images = directory::files ( ".", "*.png" );
	for ( auto & image : images ) listImages.append ( string {image} .rtrim<1> ( ".png" ) );
	listImages.setSelection ( 0 );

	append ( layout );
	layout.append ( viewport, {~0, ~0}, 0 );
	layout.append ( marginLayoutMain, {5, ~0}, 0 );
	layout.append ( listLayout, {220, ~0},5 );
	listLayout.append ( marginLayout, {~0,0},5 );
	listLayout.append ( screenShotButton, {~0,25},5 );
	listLayout.append ( widthHeightEditLayout, {~0,25},5 );
	widthHeightEditLayout.append ( textEditWidth, {40,~0},3 );
	widthHeightEditLayout.append ( xLabel, {10,~0},0 );
	widthHeightEditLayout.append ( textEditHeight, {40,~0},5 );
	widthHeightEditLayout.append ( setSizeButton, {48,~0},0 );
	widthHeightEditLayout.append ( dockCheckButton, {~0,~0},0 );
	listLayout.append ( scaleButtonsLayout, {~0,25},5 );
	scaleButtonsLayout.append ( scale1xButton, {30,~0},3 );
	scaleButtonsLayout.append ( scale2xButton, {30,~0},3 );
	scaleButtonsLayout.append ( scale3xButton, {30,~0},3 );
	scaleButtonsLayout.append ( scale4xButton, {30,~0},3 );
	scaleButtonsLayout.append ( scale5xButton, {30,~0},5 );
	scaleButtonsLayout.append ( aspectRatioCheckButton, {~0,~0},5 );

	listLayout.append ( offsetAndZoomLayout, {~0,25},5 );
//	offsetAndZoomLayout.append ( zoomLabel, {30,~0},5 );
//	offsetAndZoomLayout.append ( textEditZoomX, {40,~0},3 );
//	offsetAndZoomLayout.append ( textEditZoomY, {40,~0},5 );
//	offsetAndZoomLayout.append ( setZoomButton, {40,~0},25 );
	offsetAndZoomLayout.append ( resetOffsetZoomButton, {~0,~0},0 );

//	zoomLabel.setText("Zoom:");
//	textEditZoomX.setText("1.0");
//	textEditZoomY.setText("1.0");
//	setZoomButton.setText("Set");
	resetOffsetZoomButton.setText("Reset");
	resetOffsetZoomButton.onActivate=[&]{
		zoomSettings->offsetX=0;
		zoomSettings->offsetY=0;
		zoomSettings->zoomX=1.0;
		zoomSettings->zoomY=1.0;
		updateViewport++;

	};

	listLayout.append ( optionsLayout, {~0,20},0 );
	optionsLayout.append ( renderOnUpdateCheckButton, {65,~0},5 );
	optionsLayout.append ( vSyncCheckButton, {55,~0},5 );
	optionsLayout.append ( WatchFilesCheckButton, {~0,~0},0 );
	listLayout.append ( useUniformsCheckButton, {~0,20},0 );
	listLayout.append ( listShaders, {~0, 100},5 );
	listLayout.append ( listImages, {~0, 100},5 );
	listLayout.append ( settingsLayout, {~0, ~0} );


	dockCheckButton.setText ( "Dock" );
	dockCheckButton.setChecked();
	renderOnUpdateCheckButton.setText ( "limit FPS" );
	renderOnUpdateCheckButton.setChecked();
	vSyncCheckButton.setText ( "vSync" );
	vSyncCheckButton.setChecked( false );
	vSyncCheckButton.onToggle=[&] {
		video.set ( Video::Synchronize, vSyncCheckButton.checked() );
		video.set ( Video::Handle, viewport.handle() );
		setImage();
		setShader();
	};
	WatchFilesCheckButton.setText ( "Watch Files" );
	WatchFilesCheckButton.setChecked ( false );
	useUniformsCheckButton.setText ( "Use Uniforms for floats" );
	useUniformsCheckButton.setChecked ( true );
	useUniformsCheckButton.onToggle=[&] {
		video.set ( Video::UseUniforms, useUniformsCheckButton.checked() );
		video.set ( Video::Shader, ( const char* ) shaders ( listShaders.selection() ) );
	};

	xLabel.setText ( "x" );
	setSizeButton.setText ( "Set" );

	scale1xButton.setText ( "1x" );
	scale2xButton.setText ( "2x" );
	scale3xButton.setText ( "3x" );
	scale4xButton.setText ( "4x" );
	scale5xButton.setText ( "5x" );
	aspectRatioCheckButton.setText ( "4:3" );
	aspectRatioCheckButton.setChecked();

	screenShotButton.setText ( "ScreenShot" );

	onClose = &Application::quit;

	controlWindow.append ( controlWindowLayout );
	controlWindowLayout.append ( controlWindowMarginLayout, {5,0},0 );
	controlWindow.setResizable ( false );
	controlWindow.setVisible ( false );
	controlWindow.onClose=[this] {
		dockCheckButton.setChecked();
		updatePanelDockState();
	};
	setVisible();
	Application::processEvents();
	video.driver ( "OpenGL" );
	zoomSettings= ( ZoomSettings* ) any_cast<uintptr_t> ( video.get ( Video::ZoomSettings ) ) ;
	video.set ( Video::Handle, viewport.handle() );
	if ( video.init() == false ) exit ( 0 );
	video.set ( Video::Filter, Video::FilterLinear );
	setImage();
	if ( shaders.size() > 1 ) {
		listShaders.setSelection ( 1 );
	}
	listShaders.setFocused();
	setShader();


	listShaders.onChange = {&MainWindow::setShader, this};
	listImages.onChange = {&MainWindow::setImage, this};

	scale1xButton.onActivate=[&] {scaleViewport ( 1 ); };
	scale2xButton.onActivate=[&] {scaleViewport ( 2 ); };
	scale3xButton.onActivate=[&] {scaleViewport ( 3 ); };
	scale4xButton.onActivate=[&] {scaleViewport ( 4 ); };
	scale5xButton.onActivate=[&] {scaleViewport ( 5 ); };

	screenShotButton.onActivate= {&MainWindow::screenShot, this};
	setSizeButton.onActivate= {&MainWindow::setViewportSize, this};
	dockCheckButton.onToggle= {&MainWindow::updatePanelDockState, this};
	renderOnUpdateCheckButton.onToggle=[this] {updateViewport=0;};

	viewport.onMousePress=[this](phoenix::Mouse::Button button){
		if (button!=phoenix::Mouse::Button::Left) return;
		if ((phoenix::Keyboard::pressed(phoenix::Keyboard::Scancode::ControlLeft))||
			(phoenix::Keyboard::pressed(phoenix::Keyboard::Scancode::ControlRight)))
			dragMode=DragMode::ZOOM;
		else
			dragMode=DragMode::DRAG;
		startPos=currentPos;
		previousZoomSettings=*zoomSettings;
	};
	viewport.onMouseRelease=[this](phoenix::Mouse::Button button){
		if (button!=phoenix::Mouse::Button::Left) return;
		dragMode=DragMode::NONE;
	};


	viewport.onMouseMove=[this] ( Position pos ) {
		Position& savedPos=* ( ( Position* ) any_cast<uintptr_t> ( video.get ( Video::MousePos ) ) );
		if ( ( savedPos.x!=pos.x ) || ( savedPos.y!=pos.y ) ) {
			updateViewport++;
		}
		savedPos=pos;		
		if (dragMode==DragMode::ZOOM){

			float newDist=sqrt((pos.x-startPos.x)*(pos.x-startPos.x)+(pos.y-startPos.y)*(pos.y-startPos.y));
			newDist*=(((pos.x-startPos.x)>(pos.y-startPos.y)))?1:-1;

			zoomSettings->zoomX=previousZoomSettings.zoomX+newDist/100.0;
			zoomSettings->zoomX=max(zoomSettings->zoomX,1.0);
			zoomSettings->zoomY=zoomSettings->zoomX;


			zoomSettings->offsetX=previousZoomSettings.offsetX*(zoomSettings->zoomX/previousZoomSettings.zoomX);
			zoomSettings->offsetY=previousZoomSettings.offsetY*(zoomSettings->zoomY/previousZoomSettings.zoomY);

			int maxOffsetX=viewport.geometry().width*(zoomSettings->zoomX-1.0)/2.0;
			int maxOffsetY=viewport.geometry().height*(zoomSettings->zoomY-1.0)/2.0;

			zoomSettings->offsetX=min(max(zoomSettings->offsetX,-maxOffsetX),maxOffsetX);
			zoomSettings->offsetY=min(max(zoomSettings->offsetY,-maxOffsetY),maxOffsetY);



		}
		if (dragMode==DragMode::DRAG){
			zoomSettings->offsetX=previousZoomSettings.offsetX+pos.x-startPos.x;
			zoomSettings->offsetY=previousZoomSettings.offsetY+pos.y-startPos.y;
			int maxOffsetX=viewport.geometry().width*(zoomSettings->zoomX-1.0)/2.0;
			int maxOffsetY=viewport.geometry().height*(zoomSettings->zoomY-1.0)/2.0;
			zoomSettings->offsetX=min(max(zoomSettings->offsetX,-maxOffsetX),maxOffsetX);
			zoomSettings->offsetY=min(max(zoomSettings->offsetY,-maxOffsetY),maxOffsetY);

		}
		currentPos=pos;
	};
	viewport.onMouseLeave=[this] {
		* ( ( Position* ) any_cast<uintptr_t> ( video.get ( Video::MousePos ) ) ) ={-1,-1};
		dragMode=DragMode::NONE;
		updateViewport++;
	};
	viewport.onPaint=[&] {
		updateViewport=1;
	};
//	onSize=[&]{
//		static int oldWidth=0;
//		static int oldHeight=0;
//		if (oldWidth)
//			zoomSettings->offsetX*=viewport.geometry().width/oldWidth;
//		if (oldHeight)
//			zoomSettings->offsetY*=viewport.geometry().height/oldHeight;

//		int maxOffsetX=viewport.geometry().width*(zoomSettings->zoomX-1.0)/2.0;
//		int maxOffsetY=viewport.geometry().height*(zoomSettings->zoomY-1.0)/2.0;

//		zoomSettings->offsetX=min(max(zoomSettings->offsetX,-maxOffsetX),maxOffsetX);
//		zoomSettings->offsetY=min(max(zoomSettings->offsetY,-maxOffsetY),maxOffsetY);
//		oldWidth=viewport.geometry().width;
//		oldHeight=viewport.geometry().height;
//		updateViewport=1;
//	};


}

void MainWindow::displayMessage ( const char* message,int duration ) {
	messageDisplayDuration=duration;
	messageBuffer=message;
}


void MainWindow::setViewportSize() {
	int newWidth,newHeight,oldX,oldY;
	newWidth=atoi ( textEditWidth.text().data() );
	newHeight=atoi ( textEditHeight.text().data() );
	if ( newHeight<60 ) {
		newHeight=60;
		textEditHeight.setText ( "60" );
	}
	if ( newWidth<1 ) {
		newWidth=1;
		textEditHeight.setText ( "1" );
	}
	oldX=geometry().x;
	oldY=geometry().y;


	//setGeometry({oldX,oldY,newWidth+230+geometry().width-frameGeometry().width,newHeight+geometry().height-frameGeometry().height});
	int offsetX=viewport.geometry().width;
	offsetX-=newWidth;
	if ( dockCheckButton.checked() ) {
		setGeometry ( {oldX+offsetX,oldY,newWidth+230,newHeight} );
	} else {
		setGeometry ( {oldX,oldY,newWidth,newHeight} );
	}

}

void MainWindow::setShader() {

	updateViewport++;
	unsigned selection = 0;
	if ( listShaders.selected() ) selection = listShaders.selection();
	string pathname = shaders ( selection );
	if ( pathname == "None" ) pathname = "";
	video.set ( Video::Shader, ( const char* ) pathname );
	shaderFilesInfo.reset();
	if ( ( !pathname.empty() ) && ( file::exists ( {pathname,"manifest.bml"} ) ) ) {
		time_t currentTime=time ( 0 );
		shaderFilesInfo.append ( {"manifest.bml",currentTime} );
		auto document = Markup::Document ( file::read ( { pathname, "manifest.bml" } ) );
		for ( auto& node : document.find ( "program" ) ) {
			if ( node["vertex"].exists() ) {
				shaderFilesInfo.append ( {node["vertex"].text(),currentTime} );
			}
			if ( node["fragment"].exists() ) {
				shaderFilesInfo.append ( {node["fragment"].text(),currentTime} );
			}
		}

	}


	vector<ShaderSetting>& settings = * ( ( vector<ShaderSetting>* ) any_cast<uintptr_t> ( video.get ( Video::ShaderSettings ) ) );




	for ( auto pHly:settingsHorizontalLayout ) {
		settingsLayout.remove ( *pHly );
		delete pHly;
	}
	settingsHorizontalLayout.reset();

	for ( auto pLbl:settingsLabels ) {
		settingsLayout.remove ( *pLbl );
		delete pLbl;
	}
	settingsLabels.reset();
	for ( auto pCBtn:settingsCheckButtons ) {
		settingsLayout.remove ( *pCBtn );
		delete pCBtn;
	}
	settingsCheckButtons.reset();
	for ( auto pHSld:settingsHorizontalSlider ) {
		//settingsLayout.remove(*pHSld);
		delete pHSld;
	}
	settingsHorizontalSlider.reset();

	for ( ShaderSetting& setting : settings ) {
		if ( setting.min < setting.max ) { // use slider
			settingsLabels.append ( new Label );
			settingsLabels.last()->setText ( setting.name );
			settingsLayout.append ( * ( settingsLabels.last() ), {~0,20} );

			settingsHorizontalLayout.append ( new HorizontalLayout );
			settingsLayout.append ( * ( settingsHorizontalLayout.last() ), {~0,20} );

			settingsLabels.append ( new Label );
			settingsLabels.last()->setText ( setting.value );
			settingsHorizontalLayout.last()->append ( * ( settingsLabels.last() ), {50,~0},0 );

			settingsHorizontalSlider.append ( new HorizontalSlider_tagged );
			settingsHorizontalSlider.last()->tag=&setting;
			settingsHorizontalSlider.last()->label=settingsLabels.last();
			settingsHorizontalSlider.last()->setLength ( 1.0+floor ( ( setting.max-setting.min ) /setting.step ) );
			settingsHorizontalSlider.last()->setPosition ( ( atof ( setting.value )-setting.min ) /setting.step );
			settingsHorizontalLayout.last()->append ( * ( settingsHorizontalSlider.last() ), {~0, ~0},0 );
			HorizontalSlider_tagged& lastHzSld= * ( settingsHorizontalSlider.last() );
			lastHzSld.onChange=[&] {
				updateViewport++;
				lastHzSld.tag->value=lastHzSld.tag->min+lastHzSld.position() *setting.step;
				lastHzSld.label->setText ( setting.value );
				if ( !useUniformsCheckButton.checked() ) {
					video.set ( Video::Shader, ( const char* ) shaders ( listShaders.selection() ) );
				}
			};
			string b;

		} else if ( setting.value.empty() ) { // only valueless toggles
			settingsCheckButtons.append ( new CheckButton_tagged );
			settingsCheckButtons.last()->tag=&setting;
			settingsCheckButtons.last()->setChecked ( setting.active );
			settingsCheckButtons.last()->setText ( setting.name );
			settingsLayout.append ( * ( settingsCheckButtons.last() ), {~0,20} );
			CheckButton_tagged& lastCBtn = * ( settingsCheckButtons.last() );
			lastCBtn.onToggle=[&] {
				updateViewport++;
				lastCBtn.tag->active=lastCBtn.checked();
				video.set ( Video::Shader, ( const char* ) shaders ( listShaders.selection() ) );
			};
		}

	}
	if ( !dockCheckButton.checked() ) {
		controlWindow.setGeometry ( {controlWindow.geometry().x,controlWindow.geometry().y,controlWindow.geometry().width,listLayout.minimumSize().height+10} );
	}


}

void MainWindow::setImage() {
	updateViewport++;
	unsigned selection = 0;
	if ( listImages.selected() ) selection = listImages.selection();

	buffer.free();
	buffer.load ( images ( selection ) );
	buffer.transform ( 0, 32, 255u << 24, 255u << 16, 255u << 8, 255u << 0 );

	uint32_t* data;
	unsigned pitch;
	if ( video.lock ( data, pitch, buffer.width, buffer.height ) ) {
		pitch >>= 2;
		for ( unsigned y = 0; y < buffer.height; y++ ) {
			memcpy ( data + y * pitch, buffer.data + y * buffer.pitch, buffer.width * sizeof ( uint32_t ) );
		}
		video.unlock();
	}
}

void MainWindow::scaleViewport ( int scale ) {
	textEditHeight.setText ( buffer.height*scale );
	if ( aspectRatioCheckButton.checked() ) {
		textEditWidth.setText ( ( int ) ( ( ( float ) ( buffer.height*scale ) *4.0 ) /3.0 ) );
	} else {
		textEditWidth.setText ( buffer.width*scale );
	}
	setViewportSize();
}

void MainWindow::updatePanelDockState() {
	Geometry vp=viewport.geometry();


	if ( dockCheckButton.checked() ) {

		//if (controlWindow.visible())  return;
		setGeometry ( {geometry().x,geometry().y,geometry().width+230,geometry().height} );
		controlWindowLayout.remove ( listLayout );
		//controlWindow.synchronizeLayout();
		//listLayout.synchronizeLayout();
		layout.append ( marginLayoutMain, {5,~0},0 );
		layout.append ( listLayout, {220,~0},5 );
		controlWindow.setVisible ( false );
	} else {

		//if (!controlWindow.visible())  return;
		controlWindow.setGeometry ( {geometry().x+vp.width+2,geometry().y,230,listLayout.minimumSize().height+10} );
		setGeometry ( {geometry().x,geometry().y, ( geometry().width>230 ) ?geometry().width-230:60,geometry().height} );
		layout.remove ( listLayout );
		layout.remove ( marginLayoutMain );
		synchronizeLayout();

		controlWindowLayout.append ( listLayout, {~0, ~0},5 );
		controlWindow.setVisible();
	}
	updateViewport++;


}

void MainWindow::screenShot() {
	updateViewport++;
	screenShotFileName="./screenShots";
	if ( !nall::directory::exists ( screenShotFileName ) ) {
		directory::create ( screenShotFileName,0777 );
	}
	screenShotFileName.append ( "/" );
	screenShotFileName.append ( string ( images ( listImages.selection() ) ).rtrim<1> ( ".png" ) );
	screenShotFileName.append ( "." );
	screenShotFileName.append ( string ( shaders ( listShaders.selection() ) ).rtrim<1> ( ".shader/" ) );
	string tempName=screenShotFileName;
	int tempCounter=1;
	while ( ( file::exists ( {tempName,".bmp"} ) ) || ( file::exists ( {tempName,".settings"} ) ) ) {
		tempName=string ( {screenShotFileName,"(",tempCounter++,")"} );
	}
	screenShotFileName=tempName;
	video.set ( Video::ScreenShotFileName, ( const char* ) screenShotFileName );

	screenShotRequested=true;


}

void MainWindow::mainLoop() {


	static int oldWidth=0;
	static int oldHeight=0;
	static unsigned frameCounter = 0;
	static int remainingSeconds = 0;

	if ((oldWidth)&&(oldWidth!=viewport.geometry().width)&&
		(oldHeight)&&(oldHeight!=viewport.geometry().height)){
		zoomSettings->offsetX*=float(viewport.geometry().width)/oldWidth;
		zoomSettings->offsetY*=float(viewport.geometry().height)/oldHeight;
	}




	int maxOffsetX=viewport.geometry().width*(zoomSettings->zoomX-1.0)/2.0;
	int maxOffsetY=viewport.geometry().height*(zoomSettings->zoomY-1.0)/2.0;

	zoomSettings->offsetX=min(max(zoomSettings->offsetX,-maxOffsetX),maxOffsetX);
	zoomSettings->offsetY=min(max(zoomSettings->offsetY,-maxOffsetY),maxOffsetY);

	static time_t cachedTime = time ( 0 );
	time_t currentTime = time ( 0 );

	if ( oldWidth!=viewport.geometry().width ) {
		oldWidth=viewport.geometry().width;
		textEditWidth.setText ( oldWidth );
		updateViewport=10;
	}
	if ( oldHeight!=viewport.geometry().height ) {
		oldHeight=viewport.geometry().height;
		textEditHeight.setText ( oldHeight );
		updateViewport=10;
	}

	if ( renderOnUpdateCheckButton.checked() ) {
		if ( updateViewport ) {
			video.refresh();
			frameCounter++;
			updateViewport--;

		} else {
			usleep ( 10000 );
		}


	} else {
		video.refresh();
		frameCounter++;
	}




	if ( screenShotRequested ) {

		if ( * ( ( string* ) any_cast<uintptr_t> ( video.get ( Video::LastError ) ) ) =="screenShotError" ) {
			displayMessage ( "cannot save ScreenShot" );
		} else {
			if ( settingsLabels.empty() ) {
				displayMessage ( string {"ScreenShot Saved to ",screenShotFileName,".bmp"},2 );
			} else {
				displayMessage ( string {"ScreenShot and Settings Saved to ",screenShotFileName,".bmp   +   .settings"},2 );
			}
		}
		screenShotRequested=false;
	}





	if ( messageDisplayDuration>0 ) {
		remainingSeconds=messageDisplayDuration;
		messageDisplayDuration=0;
		setTitle ( messageBuffer );
	}


	Position* pMousePos= ( ( Position* ) any_cast<uintptr_t> ( video.get ( Video::MousePos ) ) );

	if ( currentTime > cachedTime ) {
		if ( WatchFilesCheckButton.checked() ) {
			unsigned selection = 0;
			if ( listShaders.selected() ) selection = listShaders.selection();
			string pathname = shaders ( selection );
			if ( pathname != "None" ) {
				bool reloadShader=false;
				for ( FILEINFO fileInfo: shaderFilesInfo ) {
					time_t timeStamp=file::timestamp ( {pathname,fileInfo.fileName},file::time::modify );
					if ( timeStamp>fileInfo.timeStamp ) {
						reloadShader=true;
						fileInfo.timeStamp=timeStamp;
						if ( fileInfo.fileName=="manifest.bml" ) {
							video.set ( Video::Shader, "" );
						}

					}
				}
				if ( reloadShader ) {
					listShaders.onChange();
				}
			}
		}


		if ( remainingSeconds>0 ) {
			remainingSeconds-= ( currentTime-cachedTime );
		} else {
			static char tempBuffer[16];
			if (zoomSettings->zoomX > 1.0)
				sprintf(tempBuffer,"  ( * %.2f ) ",zoomSettings->zoomX);
			else
				tempBuffer[0]=0;
			setTitle ( {"GLSLdev - ", viewport.geometry().width, "x", viewport.geometry().height, tempBuffer, " - ", ( frameCounter/ ( currentTime-cachedTime ) ), " FPS","  X:",pMousePos->x,"  Y:",pMousePos->y} );
		}
		frameCounter = 0;
		cachedTime = currentTime;
	}


}

int main() {
	auto mainWindow = new MainWindow();

	Application::main = {&MainWindow::mainLoop, mainWindow};
	Application::run();
	delete mainWindow;
	return 0;
}
