/* Intel i960 CPU emulator
 *
 * Recompiler, and C instructions.
 *
 * Copyright (c) 1999 Richard Mitton
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <math.h>
#include "i960cpu.h"

/*
 * NOTE: PC refers to Process Controls, IP refers to Instruction Pointer.
 * Don't confuse the two.
 */

/* Number of on-chip register sets (excluding the main one) */
#define REGCACHE_NUM 3

/* CPU state and stuff */
static i960_context i960;
static int i960_regcache_pos, i960_regcache_size;
static iWORD i960_frame_addr[REGCACHE_NUM], i960_regcache[REGCACHE_NUM][16];
static int cycles_left;

/* A single instruction */
typedef struct {
   iWORD opcode, disp, *src1, *src2, *srcdst;
   iWORD (*GetEA)(void);
   opcode_t *op;
} instr_param;

/* The current instruction being emulated */
static instr_param *this_instr;

/* BIG table containing unpacked instructions */
static int i960_rom_size;
static instr_param *param_table=NULL;

/* A table so we can have a pointer to a literal */
static iWORD literal_table[32];
static iFLOAT80 literal_float_0 = 0.0, literal_float_1 = 1.0;

static void i_invalid(void);
static opcode_t op_invalid = { &i_invalid };

/* Debugging */
#ifdef __I960_DEBUGGER__
static iWORD i960_breakpoint;
static int trace_ip_pos;
static iWORD trace_ip[I960_IP_TRACE_SIZE];
#endif

/* Memory helper macros */
#define ReadMem8(a) (i960_ReadMem8(a) & 0xff)
#define ReadMem16(a) (i960_ReadMem16(a) & 0xffff)
#define ReadMem32(a) (i960_ReadMem32(a))
#define WriteMem8(a,v) (i960_WriteMem8(a,v & 0xff))
#define WriteMem16(a,v) (i960_WriteMem16(a,v & 0xffff))
#define WriteMem32(a,v) (i960_WriteMem32(a,v))

/* Internal helper functions */

/* Print a formatted message */
static void Message(int fatal, char *msg, ...)
{
   char txt[200];
   va_list list;
  
   va_start(list, msg);
   vsprintf(txt, msg, list);
   va_end(list);

   i960_Message(fatal, txt);

   if (fatal) cycles_left = 0; /* stop emulation */
}

static void CheckIAC(void)
{
   int msg;
   msg = (i960.IAC[0] >> 24) & 0xff;
   Message(FALSE, "Checking IAC at IP=0x%x (IAC #%x).", i960.IP, msg);
   switch(msg)
   {
   case 0x89: /* Purge Instruction Cache */
      Message(FALSE, "Purging instruction cache at IP=0x%x.", i960.IP);
      /* FIXME! - do this! */
      break;
   case 0x93: /* Re-initialise Processor */
      Message(FALSE, "Re-initialising processor at IP=0x%x (new IP=0x%x).",
              i960.IP, i960.IAC[3]);
      i960.SAT  = i960.IAC[1];
      i960.PRCB = i960.IAC[2];
      i960.IP   = i960.IAC[3];
      break;
   default:
      Message(TRUE, "Invalid IAC #%x at IP=0x%x!", msg, i960.IP);
      break;
   }
}

iWORD MakeAC(void)
{
   return (i960.AC & ~7) | (i960.flags & 7);
}

void SplitAC(iWORD ac)
{
   i960.AC = ac;
   i960.flags = ac & 7;
}

static void GetSrc1_REG(void)
{
   int m1;
   int val;

   m1 = (this_instr->opcode >> 11) & 1;
   val = this_instr->opcode & 0x1f;
   if (m1 == 0)   /* register */
   {
      if (0 && this_instr->op->operands & O_KEEPSRC1)
         this_instr->src1 = &literal_table[val];
      else
         this_instr->src1 = &i960.regs[val];
   } else {       /* literal */
      if (this_instr->op->fp)
      {
         switch(val)
         {
         case 0x0: case 0x1: case 0x2: case 0x3:
            this_instr->src1 = (iWORD *)&i960.fp[val]; break;
         case 0x10:
            this_instr->src1 = (iWORD *)&literal_float_0; break;
         case 0x16:
            this_instr->src1 = (iWORD *)&literal_float_1; break;
         default:
            this_instr->src1 = &literal_table[0]; break;
         }
      } else {
         this_instr->src1 = &literal_table[val];
      }
   }
}

static void GetSrc1_COBR(void)
{
   int m1;
   int val;

   m1 = (this_instr->opcode >> 13) & 1;
   val = (this_instr->opcode >> 19) & 0x1f;
   if (m1 == 0)   /* register */
   {
      this_instr->src1 = &i960.regs[val];
   } else {       /* literal */
      this_instr->src1 = &literal_table[val];
   }
}

static void GetSrc2_REG(void)
{
   int m2;
   int val;

   m2 = (this_instr->opcode >> 12) & 1;
   val = (this_instr->opcode >> 14) & 0x1f;
   if (m2 == 0)   /* register */
   {
      if (0 && this_instr->op->operands & O_KEEPSRC2)
         this_instr->src2 = &literal_table[val];
      else
         this_instr->src2 = &i960.regs[val];
   } else {       /* literal */
      if (this_instr->op->fp)
      {
         switch(val)
         {
         case 0x0: case 0x1: case 0x2: case 0x3:
            this_instr->src2 = (iWORD *)&i960.fp[val]; break;
         case 0x10:
            this_instr->src2 = (iWORD *)&literal_float_0; break;
         case 0x16:
            this_instr->src2 = (iWORD *)&literal_float_1; break;
         default:
            this_instr->src2 = &literal_table[0]; break;
         }
      } else {
         this_instr->src2 = &literal_table[val];
      }
   }
}

static void GetSrc2_COBR(void)
{
   int val;

   val = (this_instr->opcode >> 14) & 0x1f;
   if (0 && this_instr->op->operands & O_KEEPSRC2)
      this_instr->src2 = &literal_table[val];
   else
      this_instr->src2 = &i960.regs[val];
}

static void GetSrcDst_REG(void)
{
   int m3, val;

   val = (this_instr->opcode >> 19) & 0x1f;
   m3 = (this_instr->opcode >> 13) & 1;
   if (m3 == 0)   /* register */
   {
      this_instr->srcdst = &i960.regs[val];
   } else {          /* literal */
      if (this_instr->op->fp)
      {
         switch(val)
         {
         case 0x0: case 0x1: case 0x2: case 0x3:
            this_instr->srcdst = (iWORD *)&i960.fp[val]; break;
         default:
            this_instr->srcdst = &literal_table[0]; break;
         }
      } else {
         this_instr->srcdst = &literal_table[val];
      }
   }
}

static void GetSrcDst_MEM(void)
{
   int val;

   val = (this_instr->opcode >> 19) & 0x1f;
   this_instr->srcdst = &i960.regs[val];
}

static void GetDisp_COBR(void)
{
   iWORD disp;
   disp = this_instr->opcode & 0x1fff;
   if (disp & 0x1000) disp |= 0xfffff000;
   this_instr->disp = disp - 4;
}

static void GetDisp_CTRL(void)
{
   iWORD disp;
   disp = this_instr->opcode & 0xffffff;
   if (disp & 0x800000) disp |= 0xff000000;
   this_instr->disp = disp - 4;
}

static iWORD CalcEA(void)
{
   int abase, offset, mode, scale, index;
   iWORD opcode2;

   abase = (this_instr->opcode >> 14) & 0x1f;

   if (((this_instr->opcode >> 12) & 1) == 0) /* MEMA */
   {
      offset = this_instr->opcode & 0x1fff;
      mode = this_instr->opcode & 0x2000;
      if (!mode)
      {
         return offset;
      } else {
         return i960.regs[abase] + offset;
      }
   } else {                       /* MEMB */
      index = this_instr->opcode & 0x1f;
      scale = (this_instr->opcode >> 7) & 0x7;
      mode = (this_instr->opcode >> 10) & 0xf;
      switch(mode)
      {
      case 0x4:
         return i960.regs[abase];
      case 0x5:
         opcode2 = ReadMem32(i960.IP) + 4;
         i960.IP += 4;
         opcode2 += i960.IP;
         return opcode2;
      case 0x7:
         return i960.regs[abase] + (i960.regs[index] << scale);
      case 0xc:
         opcode2 = ReadMem32(i960.IP);
         i960.IP += 4;
         return opcode2;
      case 0xd:
         opcode2 = ReadMem32(i960.IP);
         i960.IP += 4;
         return i960.regs[abase] + opcode2;
      case 0xe:
         opcode2 = ReadMem32(i960.IP);
         i960.IP += 4;
         return (i960.regs[index] << scale) + opcode2;
      case 0xf:
         opcode2 = ReadMem32(i960.IP);
         i960.IP += 4;
         return i960.regs[abase] + (i960.regs[index] << scale) + opcode2;
      }
   }

   Message(TRUE, "Invalid EA at IP=0x%x!", i960.IP);
   return 0;
}

static void GetEA_MEM(void)
{
   this_instr->GetEA = &CalcEA;
}

static int ExtractOpcodeNumber(iWORD opcode)
{
   int opnum;
   /* get the op-num */
   opnum = (opcode & 0xff000000) >> 24;
   if (opnum >= 0x58 && opnum <= 0x79)
   {
      opnum <<= 4;
      opnum |= (opcode >> 7) & 0x0f;
   }
   return opnum;
}

/* Main functions */

static void FillInInstr(iWORD addr)
{
   int op_num;
   opcode_t *op;

   this_instr->opcode = ReadMem32(addr);

   op_num = ExtractOpcodeNumber(this_instr->opcode);

   /* find it */
   op = &i960_opcode_table[0];
   while(op->func)
   {
      if (op->opcode == op_num) break;
      op++;
   }
   this_instr->op = op;
   if (!op->func)
   {
      this_instr->op = &op_invalid;
      return;
   }

   /* fill in the info for it */
   switch(op->type)
   {
   case I_REG:
      if (op->operands & O_SRC1) GetSrc1_REG();
      if (op->operands & O_SRC2) GetSrc2_REG();
      if (op->operands & O_SRCDST) GetSrcDst_REG();
      break;
   case I_MEM:
      if (op->operands & O_SRCDST) GetSrcDst_MEM();
      if (op->operands & O_ADDR) GetEA_MEM();
      break;
   case I_COBR:
      if (op->operands & O_SRC1) GetSrc1_COBR();
      if (op->operands & O_SRC2) GetSrc2_COBR();
      if (op->operands & O_ADDR) GetDisp_COBR();
      break;
   case I_CTRL:
      if (op->operands & O_ADDR) GetDisp_CTRL();
      break;
   }
}

void i960_Init(int rom_size)
{
   int n;

   Message(FALSE, "Allocating param_table...");
   i960_rom_size = rom_size;
   rom_size /= 4;
   param_table = (instr_param *)malloc(sizeof(instr_param) * rom_size);
   if (!param_table) Message(TRUE, "Out of memory for param_table!");

   for (n=0;n<32;n++) literal_table[n] = n;

   Message(FALSE, "Re-compiling...");
   for (n=0;n<rom_size;n++)
   {
      this_instr = &param_table[n];
      FillInInstr(n*4);
   }
   Message(FALSE, "Re-compiled %i instructions.", rom_size);
}

void i960_Shutdown(void)
{
   Message(FALSE, "Shutting down CPU core.");
   if (param_table) free(param_table);
   param_table = NULL;
}

void i960_Reset(void)
{
   Message(FALSE, "Processor hard-reset.");
   i960.SAT  = ReadMem32(0x00);
   i960.PRCB = ReadMem32(0x04);
   i960.IP   = ReadMem32(0x0c);
   SplitAC(0);
   i960.PC = 0x001f2002;
   i960.regs[REG_FP] = ReadMem32(i960.PRCB+24); /* FP */
   i960.regs[REG_SP] = i960.regs[REG_FP];       /* SP */

   i960_regcache_size = 0;

#ifdef __I960_DEBUGGER__
   for (trace_ip_pos=0;trace_ip_pos<I960_IP_TRACE_SIZE;trace_ip_pos++)
      trace_ip[trace_ip_pos] = 0x00000000;
   trace_ip_pos = 0;
   i960_breakpoint = 0;
#endif
}

iWORD i960_GetIP(void)
{
   return i960.IP;
}

void i960_SendIAC(iWORD addr, iWORD value)
{
   i960.IAC[(addr - 0xff000010) >> 2] = value;
}

void i960_Interrupt(int pin)
{
   iBYTE vector;
   iWORD table, address, FP,SP,PFP;
   int n,priority;

   table = ReadMem32(i960.PRCB+20);

   vector = (i960.ICR >> (pin*8)) & 0xff;
   if (vector == 0x00)
   {
      /* FIXME - what about other ICR configurations? */
      Message(TRUE, "Invalid ICR vector on INT%i!", pin);
      return;
   }

   priority = vector / 8;
   if (priority < ((i960.PC >> 16) & 0x1f))
   {
      /* FIXME! - pending interrupts? */
      Message(FALSE, "INT%i not taken, priority is too low (%i vs %i).", pin, priority, (i960.PC>>16)&0x1f);
      return;
   }

   /* Preserve local registers */
   FP = i960.regs[REG_FP];
   i960.regs[REG_RIP] = i960.IP;    /* set RIP */

   /* see if we need to free a cache entry */
   if (i960_regcache_size == REGCACHE_NUM)
   {
      /* flush the regs already in the cache */
      PFP = i960_frame_addr[i960_regcache_pos];
      for (n=0;n<16;n++)
      {
         WriteMem32(PFP+n*4, i960_regcache[i960_regcache_pos][n]);
      }
      i960_regcache_size--;
   }

   /* preserve our regs */
   memcpy(&i960_regcache[i960_regcache_pos][0], &i960.regs[0], 16*sizeof(iWORD));
   i960_frame_addr[i960_regcache_pos] = i960.regs[REG_FP];
   i960_regcache_pos++;
   i960_regcache_pos %= REGCACHE_NUM;
   i960_regcache_size++;

   address = ReadMem32(table + 4 + (vector*4));

   if (!(i960.PC & 0x2000))
      SP = ReadMem32(i960.PRCB+24);    /* Switch to interrupt stack if not in
   else                              * interrupt mode already */
      SP = i960.regs[REG_SP];

   SP = (SP + 63) & (~63);          /* round up to 64 bytes */
   SP += 48;                        /* to make the new FP at a 64-byte interval */

   WriteMem32(SP, i960.PC);            /* Write PC */
   SP += 4;
   WriteMem32(SP, MakeAC());        /* Write AC */
   SP += 4;
   WriteMem32(SP, vector);          /* Vector number */
   SP += 8;
   i960.regs[REG_FP] = SP;             /* Set FP */

   i960.PC &= ~0x1f00;                 /* Clear old priority */
   i960.PC |= 0x2002 | (priority<<16); /* Set supervisor mode, interrupted mode,
                                     * and the new priority */
   i960.regs[REG_PFP] = FP|0x07;       /* Set PFP and return mode */

   i960.regs[REG_SP] = SP + 64;        /* put the stack after this frame */
   i960.IP = address;
}

int i960_CyclesLeft(void)
{
   return cycles_left;
}

int i960_Emulate(int cycles)
{
   int cycles_start;
   instr_param instr;

   cycles_start = cycles_left = cycles;

   while(cycles_left > 0)
   {
#ifdef __I960_DEBUGGER__
      /* update the IP log */
      trace_ip[trace_ip_pos] = i960.IP;
      trace_ip_pos++;
      trace_ip_pos %= I960_IP_TRACE_SIZE;

#endif

      //if (i960.IP >= i960_rom_size) {
      //   Message(TRUE, "Jump to 0x%x, outside ROM!", i960.IP);
      //   break;
      //}

      if (i960.IP >= i960_rom_size) {
         /* Outside ROM, so decode it now rather than using param_table */
         this_instr = &instr;
         FillInInstr(i960.IP);
      } else {
         /* Inside ROM, so already decoded */
         this_instr = &param_table[i960.IP / 4];
      }
      i960.IP += 4;

      this_instr->op->func();
      
      cycles_left -= this_instr->op->cycles;

#ifdef __I960_DEBUGGER__
      if (i960.IP == i960_breakpoint) break;
#endif
   }
   return cycles_start - cycles_left;
}

void i960_GetContext(i960_context *context)
{
   i960.AC = MakeAC();
   *context = i960;
}

void i960_SetContext(i960_context *context)
{
   i960 = *context;
   SplitAC(i960.AC);
}

/* Debugging functions */

#ifdef __I960_DEBUGGER__

int i960_SingleStep(void)
{
   return i960_Emulate(1);
}

iWORD i960_TraceIP(int trace_num)
{
   int n;
   n = trace_ip_pos - (trace_num + 1);
   if (n < 0) n += I960_IP_TRACE_SIZE;
   return trace_ip[n];
}

void i960_SetBreakPoint(iWORD bp)
{
   if (i960_breakpoint && !bp) Message(FALSE, "Breakpoint cleared.");
   i960_breakpoint = bp;
   if (i960_breakpoint) Message(FALSE, "Breakpoint set to 0x%x.", i960_breakpoint);
}

#endif

/* CPU opcode functions */

   /* some macros */
   #define GetEA() this_instr->GetEA()
   #define GetDisp() this_instr->disp
   #define GetSrc1() *(this_instr->src1)
   #define GetSrc2() *(this_instr->src2)
   #define GetSrcDst() *(this_instr->srcdst)
   #define GetPSrc1() (this_instr->src1)
   #define GetPSrc2() (this_instr->src2)
   #define GetPSrcDst() (this_instr->srcdst)

   /* Convert a stored IEEE float to it's proper type */
   static inline iFLOAT32 GetFloat32(iWORD *val) { return *((iFLOAT32 *)val); }
   static inline iFLOAT64 GetFloat64(iWORD *val) { return *((iFLOAT64 *)val); }
   static inline iFLOAT80 GetFloat80(iWORD *val) { return *((iFLOAT80 *)val); }

   /* Convert a proper float to it's stored IEEE representation */
   static inline void PutFloat32(iFLOAT32 val, iWORD *dst) { memcpy(dst, &val, sizeof(iFLOAT32)); }
   static inline void PutFloat64(iFLOAT64 val, iWORD *dst) { memcpy(dst, &val, sizeof(iFLOAT64)); }
   static inline void PutFloat80(iFLOAT80 val, iWORD *dst) { memcpy(dst, &val, sizeof(iFLOAT80)); }

   #define SRC1_IS_FP   (this_instr->opcode & (1 << 11))
   #define SRC2_IS_FP   (this_instr->opcode & (1 << 12))
   #define SRCDST_IS_FP (this_instr->opcode & (1 << 13))

   #define JumpCC(cc, disp) if ((i960.flags & cc) != 0) i960.IP += disp
   #define Cmp(a, b) if ((a) < (b)) i960.flags = 4; \
                     else if ((a) == (b)) i960.flags = 2; \
                     else i960.flags = 1;

static void i_invalid(void)
{
   Message(TRUE, "Invalid opcode %x at IP=0x%x!",
           ExtractOpcodeNumber(this_instr->opcode),
           i960.IP);
}

static void i_ldob(void)
{
   iWORD ea, *dst;

   ea = GetEA();
   dst = GetPSrcDst();
   *dst = ReadMem8(ea);
}

static void i_ldib(void)
{
   iWORD ea, *dst;
   iWORDs val;

   ea = GetEA();
   val = (iWORDs)(iBYTEs)ReadMem8(ea);
   dst = GetPSrcDst();
   *dst = val;
}

static void i_ldos(void)
{
   iWORD ea, val, *dst;

   ea = GetEA();
   val = ReadMem16(ea);
   dst = GetPSrcDst();
   *dst = val;
}

static void i_ldis(void)
{
   iWORD ea, val, *dst;

   ea = GetEA();
   val = (iWORDs)(iSHORTs)ReadMem16(ea);
   dst = GetPSrcDst();
   *dst = val;
}

static void i_ld(void)
{
   iWORD ea, val, *dst;

   ea = GetEA();
   val = ReadMem32(ea);
   dst = GetPSrcDst();
   *dst = val;
}

static void i_ldl(void)
{
   iWORD ea, *dst;

   ea = GetEA();
   dst = GetPSrcDst();
   *dst++ = ReadMem32(ea);
   *dst   = ReadMem32(ea + 4);
}

static void i_ldt(void)
{
   iWORD ea, *dst;

   ea = GetEA();
   dst = GetPSrcDst();
   *dst++ = ReadMem32(ea);
   *dst++ = ReadMem32(ea+4);
   *dst   = ReadMem32(ea+8);
}

static void i_ldq(void)
{
   iWORD ea, *dst;

   ea = GetEA();
   dst = GetPSrcDst();
   *dst++ = ReadMem32(ea);
   *dst++ = ReadMem32(ea+4);
   *dst++ = ReadMem32(ea+8);
   *dst   = ReadMem32(ea+12);
}

static void i_st(void)
{
   iWORD ea, val;

   ea = GetEA();
   val = GetSrcDst();
   WriteMem32(ea, val);
}

static void i_stob(void)
{
   iWORD ea, val;

   ea = GetEA();
   val = GetSrcDst();
   WriteMem8(ea, val);
}

static void i_stib(void)
{
   iWORD ea, val;

   ea = GetEA();
   val = GetSrcDst();
   WriteMem8(ea, val);
}

static void i_stos(void)
{
   iWORD ea, val;

   ea = GetEA();
   val = GetSrcDst();
   WriteMem16(ea, val);
}

static void i_stis(void)
{
   iWORD ea, val;

   ea = GetEA();
   val = GetSrcDst();
   WriteMem16(ea, val);
}

static void i_stl(void)
{
   iWORD ea, *src;

   ea = GetEA();
   src = GetPSrcDst();
   WriteMem32(ea, *src++);
   WriteMem32(ea+4, *src);
}

static void i_stt(void)
{
   iWORD ea, *src;

   ea = GetEA();
   src = GetPSrcDst();

   WriteMem32(ea, *src++);
   WriteMem32(ea+4, *src++);
   WriteMem32(ea+8, *src);
}

static void i_stq(void)
{
   iWORD ea, *src;

   ea = GetEA();
   src = GetPSrcDst();

   WriteMem32(ea, *src++);
   WriteMem32(ea+4, *src++);
   WriteMem32(ea+8, *src++);
   WriteMem32(ea+12, *src);
}

static void i_mov(void)
{
   iWORD src, *dst;
   src = GetSrc1();
   dst = GetPSrcDst();
   *dst = src;
}

static void i_movl(void)
{
   iWORD *src, *dst;
   src = GetPSrc1();
   dst = GetPSrcDst();

   /* test M1 bit. Damn exceptions-to-the-rule. */
   if (this_instr->opcode & (1 << 11))
   {
      *dst++ = *src;
      *dst   = *src;
   } else {
      memcpy(dst, src, sizeof(iWORD) * 2);
   }
}

static void i_movt(void)
{
   iWORD *src, *dst;
   src = GetPSrc1();
   dst = GetPSrcDst();

   /* test M1 bit. Damn exceptions-to-the-rule. */
   if (this_instr->opcode & (1 << 11))    
   {
      *dst++ = *src;
      *dst++ = *src;
      *dst   = *src;
   } else {
      memcpy(dst, src, sizeof(iWORD) * 3);
   }
}

static void i_movq(void)
{
   iWORD *src, *dst;
   src = GetPSrc1();
   dst = GetPSrcDst();

   /* test M1 bit. Damn exceptions-to-the-rule. */
   if (this_instr->opcode & (1 << 11))
   {
      *dst++ = *src;
      *dst++ = *src;
      *dst++ = *src;
      *dst   = *src;
   } else {
      memcpy(dst, src, sizeof(iWORD) * 4);
   }
}

static void i_lda(void)
{
   iWORD ea, *dst;
   ea = GetEA();
   dst = GetPSrcDst();
   *dst = ea;
}

static void i_notbit(void)
{
   iWORD src, *dst, bitpos;
   bitpos = GetSrc1();
   src = GetSrc2();
   dst = GetPSrcDst();

   src ^= (1 << bitpos);
   *dst = src;
}

static void i_setbit(void)
{
   iWORD bitpos, src, *dst;
   bitpos = GetSrc1();
   src = GetSrc2();
   dst = GetPSrcDst();

   src |= (1 << bitpos);
   *dst = src;
}

static void i_clrbit(void)
{
   iWORD bitpos, src, *dst;
   bitpos = GetSrc1();
   src = GetSrc2();
   dst = GetPSrcDst();

   src &= ~(1 << bitpos);
   *dst = src;
}

static void i_alterbit(void)
{
   iWORD bitpos, src, *dst;
   bitpos = GetSrc1();
   src = GetSrc2();
   dst = GetPSrcDst();

   if (i960.flags & 0x2)
      src |=  (1 << bitpos);
   else
      src &= ~(1 << bitpos);
   *dst = src;
}

static void i_chkbit(void)
{
   iWORD bitpos, src;
   bitpos = GetSrc1();
   src = GetSrc2();

   if (src & (1 << bitpos))
      i960.flags = 0x2;
   else
      i960.flags = 0x0;
}

static void i_addi(void)
{
   iWORDs src1, src2;
   iWORD *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 += src1;
   *dst = src2;
}

static void i_addo(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 += src1;
   *dst = src2;
}

static void i_subo(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 -= src1;
   *dst = src2;
}

static void i_subi(void)
{
   iWORDs src1, src2;
   iWORD *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 -= src1;
   *dst = src2;
}

static void i_addc(void)
{
   iWORD src1, src2, *dst;

   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   asm("
      shr $2, %4
      adc %3, %2
      pushf
      pop %1
      and $0x801, %1
      shl $5, %h1
      rol $1, %w1
   " : "=r" (src2), "=r" (i960.flags)
     : "0" (src2), "g" (src1), "r" (i960.flags)
     : "%cc", "%eax");
   *dst = src2;
}

static void i_subc(void)
{
   iWORD src1, src2, *dst;

   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   asm("
      shr $2, %4
      sbb %3, %2
      pushf
      pop %1
      and $0x801, %1
      shl $5, %h1
      rol $1, %w1
   " : "=r" (src2), "=r" (i960.flags)
     : "0" (src2), "g" (src1), "r" (i960.flags)
     : "%cc", "%eax");
   *dst = src2;
}

static void i_teste(void)
{
   iWORD *dst, result;
   dst = GetPSrc1();

   if (i960.flags & 2)
      result = 1;
   else
      result = 0;
   *dst = result;
}

static void i_testne(void)
{
   iWORD *dst, result;
   dst = GetPSrc1();

   if (i960.flags & 5)
      result = 1;
   else
      result = 0;
   *dst = result;
}

static void i_testl(void)
{
   iWORD *dst, result;
   dst = GetPSrc1();

   if (i960.flags & 4)
      result = 1;
   else
      result = 0;
   *dst = result;
}

static void i_testle(void)
{
   iWORD *dst, result;
   dst = GetPSrc1();

   if (i960.flags & 6)
      result = 1;
   else
      result = 0;
   *dst = result;
}

static void i_testg(void)
{
   iWORD *dst, result;
   dst = GetPSrc1();

   if (i960.flags & 1)
      result = 1;
   else
      result = 0;
   *dst = result;
}

static void i_testno(void)
{
   iWORD *dst, result;
   dst = GetPSrc1();

   if (i960.flags == 0)
      result = 1;
   else
      result = 0;
   *dst = result;
}

static void i_and(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 &= src1;
   *dst = src2;
}

static void i_or(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 |= src1;
   *dst = src2;
}

static void i_ornot(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 |= (~src1);
   *dst = src2;
}

static void i_not(void)
{
   iWORD src, *dst;
   src = GetSrc1();
   dst = GetPSrcDst();
   *dst = ~src;
}

static void i_andnot(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 &= (~src1);
   *dst = src2;
}

static void i_notand(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src1 &= (~src2);
   *dst = src1;
}

static void i_xor(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 ^= src1;
   *dst = src2;
}

static void i_modac(void)
{
   iWORD mask, src, *dst, temp;
   mask = GetSrc1();
   src = GetSrc2();
   dst = GetPSrcDst();

   temp = MakeAC();
   SplitAC((src & mask) | (temp & (~mask)));
   *dst = temp;

   /* FIXME - this should do something! */
   Message(FALSE, "modac at IP=0x%x.", i960.IP);
}

static void i_modpc(void)
{
   iWORD mask, srcdst;
   mask = GetSrc2();
   srcdst = GetSrcDst();

   i960.PC = (srcdst & mask) | (i960.PC & (~mask));

   /* FIXME!! */
   /* This should check pending ints etc */
   Message(FALSE, "modpc at IP=0x%x.", i960.IP);       
}

static void i_mulo(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 *= src1;
   *dst = src2;
}

static void i_muli(void)
{
   iWORDs src1, src2;
   iWORD *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 *= src1;
   *dst = src2;
}

static void i_divi(void)
{
   iWORDs src1, src2;
   iWORD *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 /= src1;
   *dst = src2;
}

static void i_divo(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 /= src1;
   *dst = src2;
}

static void i_remi(void)
{
   iWORDs src1, src2;
   iWORD *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 %= src1;
   *dst = src2;
}

static void i_remo(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 %= src1;
   *dst = src2;
}

static void i_ediv(void)
{
   iWORD src1, *src2, *dst;
   iDWORD val;

   src1 = GetSrc1();
   src2 = GetPSrc2();
   dst = GetPSrcDst();

   val  = *((iDWORD *)src2);

   *dst++ = val % src1;
   *dst   = val / src1;
}

static void i_synmov(void)
{
   iWORD val, src, dst;

   dst = GetSrc1();
   src = GetSrc2();

   val = ReadMem32(src);

   if (dst == 0xff000004)
   {
      i960.ICR = val;
      Message(FALSE, "ICR set to 0x%x, at IP=0x%x.", val, i960.IP);
   } else {
      WriteMem32(dst, val);
   }
   i960.flags = 0x02;
}

static void i_synmovq(void)
{
   iWORD val, src, dst, addr;

   addr = dst = GetSrc1();
   src = GetSrc2();

   val = ReadMem32(src);
   WriteMem32(dst, val);
   val = ReadMem32(src+4);
   WriteMem32(dst+4, val);
   val = ReadMem32(src+8);
   WriteMem32(dst+8, val);
   val = ReadMem32(src+12);
   WriteMem32(dst+12, val);

   if (addr == 0xff000010) CheckIAC();
   i960.flags = 0x02;
}

static void i_call(void)
{
   iWORD FP,SP,PFP;
   int n;
   FP = i960.regs[REG_FP];    /* FP is in g15 */
   SP = i960.regs[REG_SP];    /* SP is in r1 */
   i960.regs[REG_RIP] = i960.IP; /* set RIP */

   /* see if we need to free a cache entry */
   if (i960_regcache_size == REGCACHE_NUM)
   {
      /* flush the regs already in the cache */
      PFP = i960_frame_addr[i960_regcache_pos];
      for (n=0;n<16;n++)
      {
         WriteMem32(PFP+n*4, i960_regcache[i960_regcache_pos][n]);
      }
      i960_regcache_size--;
   }

   /* preserve our regs */
   memcpy(&i960_regcache[i960_regcache_pos][0], &i960.regs[0], 16*sizeof(iWORD));
   i960_frame_addr[i960_regcache_pos] = i960.regs[REG_FP];
   i960_regcache_pos++;
   i960_regcache_pos %= REGCACHE_NUM;
   i960_regcache_size++;

   i960.IP += GetDisp();
   i960.regs[REG_PFP] = FP;   /* set new PFP to old FP */
   SP = (SP + 63) & (~63); /* round up to 64 bytes */
   i960.regs[REG_FP] = SP;      /* set FP to SP */
   i960.regs[REG_SP] = SP + 64; /* set new SP */
#ifdef __I960_CALL_TRACE__
   Message(FALSE, "call %x (ret to %x).", i960.IP, i960.regs[REG_RIP]);
#endif
}

static void i_callx(void)
{
   iWORD FP,SP,PFP;
   int n;
   FP = i960.regs[REG_FP];    /* FP is in g15 */
   SP = i960.regs[REG_SP];    /* SP is in r1 */
   i960.regs[REG_RIP] = i960.IP; /* set RIP */

   /* see if we need to free a cache entry */
   if (i960_regcache_size == REGCACHE_NUM)
   {
      /* flush the regs already in the cache */
      PFP = i960_frame_addr[i960_regcache_pos];
      for (n=0;n<16;n++)
      {
         WriteMem32(PFP+n*4, i960_regcache[i960_regcache_pos][n]);
      }
      i960_regcache_size--;
   }

   /* preserve our regs */
   memcpy(&i960_regcache[i960_regcache_pos][0], &i960.regs[0], 16*sizeof(iWORD));
   i960_frame_addr[i960_regcache_pos] = i960.regs[REG_FP];
   i960_regcache_pos++;
   i960_regcache_pos %= REGCACHE_NUM;
   i960_regcache_size++;

   i960.IP = GetEA();
   i960.regs[REG_PFP] = FP;   /* set new PFP to old FP */
   SP = (SP + 63) & (~63); /* round up to 64 bytes */
   i960.regs[REG_FP] = SP;      /* set FP to SP */
   i960.regs[REG_SP] = SP + 64; /* set new SP */
#ifdef __I960_CALL_TRACE__
   Message(FALSE, "callx %x (ret to %x).", i960.IP, i960.regs[REG_RIP]);
#endif
}

static void i_ret(void)
{
   int status,n;
   iWORD PFP,AC,PC,addr;
   int found=FALSE;

   status = i960.regs[REG_PFP] & 7;
   switch(status)
   {
   case 0: /* standard ret */
      PFP = i960.regs[REG_PFP] & (~7);

      /* load the old regs back in */
      while (i960_regcache_size > 0)
      {
         i960_regcache_pos--;
         if (i960_regcache_pos < 0) i960_regcache_pos = REGCACHE_NUM-1;
         i960_regcache_size--;

         if (i960_frame_addr[i960_regcache_pos] == PFP)
         {
            memcpy(&i960.regs[0], &i960_regcache[i960_regcache_pos][0], 16*sizeof(iWORD));
            found = TRUE;
            break;
         }

         /* flush these regs, they're being bypassed */
         addr = i960_frame_addr[i960_regcache_pos];
         for (n=0;n<16;n++)
         {
            WriteMem32(addr+n*4, i960_regcache[i960_regcache_pos][n]);
         }
      }

      if (!found)
      {
         /* read the old regs back in */
         for (n=0;n<16;n++)
         {
            i960.regs[n] = ReadMem32(PFP+n*4);
         }
      }

      i960.regs[REG_FP] = PFP;
      i960.IP = i960.regs[REG_RIP]; /* RIP */
      break;
   case 7: /* interrupt ret */
      Message(FALSE, "interrupt ret at IP=0x%x.", i960.IP);
      PC = ReadMem32(i960.regs[REG_FP]-16);
      AC = ReadMem32(i960.regs[REG_FP]-12);
      PFP = i960.regs[REG_PFP] & (~7);

      /* load the old regs back in */
      while (i960_regcache_size > 0)
      {
         i960_regcache_pos--;
         if (i960_regcache_pos < 0) i960_regcache_pos = REGCACHE_NUM-1;
         i960_regcache_size--;

         if (i960_frame_addr[i960_regcache_pos] == PFP)
         {
            memcpy(&i960.regs[0], &i960_regcache[i960_regcache_pos][0], 16*sizeof(iWORD));
            found = TRUE;
            break;
         }

         /* flush these regs, they're being bypassed */
         addr = i960_frame_addr[i960_regcache_pos];
         for (n=0;n<16;n++)
         {
            WriteMem32(addr+n*4, i960_regcache[i960_regcache_pos][n]);
         }
      }

      if (!found)
      {
         /* read the old regs back in */
         for (n=0;n<16;n++)
         {
            i960.regs[n] = ReadMem32(PFP+n*4);
         }
      }

      i960.regs[REG_FP] = PFP;
      i960.IP = i960.regs[REG_RIP]; /* RIP */
      SplitAC(AC);
      if (i960.PC & 0x02) i960.PC = PC; /* check supervisor bit */
      /* FIXME! - check pending ints here */
      Message(FALSE, "FP=0x%x", i960.regs[REG_FP]);
      break;
   default: Message(TRUE, "Unsupported return-status %i from IP=0x%x.", status, i960.IP); break;
   }
#ifdef __I960_CALL_TRACE__
   Message(FALSE, "ret to 0x%x (g0=%x).", i960.IP, i960.regs[0x10]);
#endif
}

static void i_flushreg(void)
{
   iWORD PFP;
   int n;

   while (i960_regcache_size > 0)
   {
      i960_regcache_pos--;
      if (i960_regcache_pos < 0) i960_regcache_pos = REGCACHE_NUM-1;
      i960_regcache_size--;

      /* flush these regs, they're being bypassed */
      PFP = i960_frame_addr[i960_regcache_pos];
      for (n=0;n<16;n++)
      {
         WriteMem32(PFP+n*4, i960_regcache[i960_regcache_pos][n]);
      }
   }
}

static void i_b(void)
{
   i960.IP += GetDisp();
}

static void i_bx(void)
{
   i960.IP = GetEA();
}

static void i_be(void)
{
   JumpCC(2, GetDisp());
}

static void i_bne(void)
{
   JumpCC(5, GetDisp());
}

static void i_bl(void)
{
   JumpCC(4, GetDisp());
}

static void i_bg(void)
{
   JumpCC(1, GetDisp());
}

static void i_bge(void)
{
   JumpCC(3, GetDisp());
}

static void i_ble(void)
{
   JumpCC(6, GetDisp());
}

static void i_bno(void)
{
   if ((i960.flags & 7) == 0) i960.IP += GetDisp();
}

static void i_bal(void)
{
   i960.regs[0x1e] = i960.IP;
   i960.IP += GetDisp();
#ifdef __I960_CALL_TRACE__
   Message(FALSE, "bal to 0x%x, return to 0x%x.", i960.IP, i960.regs[0x1e]);
#endif
}

static void i_balx(void)
{
   iWORD *dst, addr;
   addr = GetEA();
   dst = GetPSrcDst();
   *dst = i960.IP;
   i960.IP = addr;
#ifdef __I960_CALL_TRACE__
   Message(FALSE, "balx to 0x%x, return to 0x%x.", i960.IP, *dst);
#endif
}

static void i_cmpo(void)
{
   iWORD src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
}

static void i_cmpi(void)
{
   iWORDs src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
}

static void i_concmpi(void)
{
   iWORDs src1, src2;
   if (!(i960.flags & 4))
   {
      src1 = GetSrc1();
      src2 = GetSrc2();

      if (src1 <= src2) i960.flags = 2;
      else i960.flags = 1;
   }
}

static void i_concmpo(void)
{
   iWORD src1, src2;
   if (!(i960.flags & 4))
   {
      src1 = GetSrc1();
      src2 = GetSrc2();

      if (src1 <= src2) i960.flags = 2;
      else i960.flags = 1;
   }
}

static void i_cmpinco(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   Cmp(src1, src2);
   src2++;
   *dst = src2;
}

static void i_cmpdeco(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   Cmp(src1, src2);
   src2--;
   *dst = src2;
}

static void i_cmpinci(void)
{
   iWORDs src1, src2;
   iWORD *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   Cmp(src1, src2);
   src2++;
   *dst = src2;
}

static void i_cmpdeci(void)
{
   iWORDs src1, src2;
   iWORD *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   Cmp(src1, src2);
   src2--;
   *dst = src2;
}

static void i_cmpobe(void)
{
   iWORD src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
   JumpCC(2, GetDisp());
}                    

static void i_cmpobne(void)
{
   iWORD src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
   JumpCC(5, GetDisp());
}                    

static void i_cmpobge(void)
{
   iWORD src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
   JumpCC(3, GetDisp());
}                    

static void i_cmpobl(void)
{
   iWORD src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
   JumpCC(4, GetDisp());
}                    

static void i_cmpoble(void)
{
   iWORD src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
   JumpCC(6, GetDisp());
}                    

static void i_cmpobg(void)
{
   iWORD src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
   JumpCC(1, GetDisp());
}                    

static void i_cmpibg(void)
{
   iWORDs src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
   JumpCC(1, GetDisp());
}                    

static void i_cmpibge(void)
{
   iWORDs src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
   JumpCC(3, GetDisp());
}                    

static void i_cmpibe(void)
{
   iWORDs src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
   JumpCC(2, GetDisp());
}                    

static void i_cmpibne(void)
{
   iWORDs src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
   JumpCC(5, GetDisp());
}                    

static void i_cmpibl(void)
{
   iWORDs src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
   JumpCC(4, GetDisp());
}                    

static void i_cmpible(void)
{
   iWORDs src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   Cmp(src1, src2);
   JumpCC(6, GetDisp());
}                    

static void i_bbs(void)
{
   iWORD src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   src1 &= 0x1f;
   if (src2 & (1 << src1))
   {
      i960.flags = 2;
      i960.IP += GetDisp();
   } else {
      i960.flags = 0;
   }
}                    

static void i_bbc(void)
{
   iWORD src1, src2;
   src1 = GetSrc1();
   src2 = GetSrc2();

   src1 &= 0x1f;
   if (!(src2 & (1 << src1)))
   {
      i960.flags = 2;
      i960.IP += GetDisp();
   } else {
      i960.flags = 0;
   }
}                    

static void i_shli(void)
{
   iWORDs src2;
   iWORD src1, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 <<= src1;
   *dst = src2;
}

static void i_shlo(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 <<= src1;
   *dst = src2;
}

static void i_shri(void)
{
   iWORDs src2;
   iWORD src1, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   /* FIXME: negative rounding error here? */
   src2 >>= src1;
   *dst = src2;
}

static void i_shro(void)
{
   iWORD src1, src2, *dst;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src2 >>= src1;
   *dst = src2;
}

static void i_shrdi(void)
{
   iWORD src1, *dst;
   iWORDs src2;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   /* FIXME: negative rounding error here (see manual) */
   src2 >>= src1;
   *dst = src2;
}

static void i_rotate(void)
{
   iWORD src1, src2, *dst, tmp;
   src1 = GetSrc1();
   src2 = GetSrc2();
   dst = GetPSrcDst();

   src1 &= 0x1f;

   tmp = src2 >> (32 - src1);
   src2 <<= src1;
   src2 |= tmp;
   *dst = src2;
}

static void i_scanbit(void)
{
   iWORD src, *dst, val, i, mask;
   src = GetSrc1();
   dst = GetPSrcDst();

   /* indicate failure first */
   val = 0xffffffff;
   i960.flags = 0x0;

   /* search for MSB */
   i = 31;
   mask = 0x80000000;
   do {
      if (src & mask)
      {
         val = i;
         i960.flags = 0x2;
         break;
      }
      mask >>= 1;
   } while(--i);

   *dst = val;
}

static void i_spanbit(void)
{
   iWORD src, *dst, val, i, mask;
   src = GetSrc1();
   dst = GetPSrcDst();

   /* indicate failure first */
   val = 0xffffffff;
   i960.flags = 0x0;

   /* search for MSB */
   i = 31;
   mask = 0x80000000;
   do {
      if (!(src & mask))
      {
         val = i;
         i960.flags = 0x2;
         break;
      }
      mask >>= 1;
   } while(--i);

   *dst = val;
}

static void i_cmpr(void)
{
   iFLOAT80 src1, src2;
   src1 = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat32(GetPSrc1());
   src2 = SRC2_IS_FP ? GetFloat80(GetPSrc2()) : GetFloat32(GetPSrc2());

   Cmp(src1, src2);
}

static void i_cmprl(void)
{
   iFLOAT80 src1, src2;
   src1 = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat64(GetPSrc1());
   src2 = SRC2_IS_FP ? GetFloat80(GetPSrc2()) : GetFloat64(GetPSrc2());

   Cmp(src1, src2);
}

static void i_addr(void)
{
   iFLOAT80 src1, src2, result;
   iWORD *dst;

   src1 = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat32(GetPSrc1());
   src2 = SRC2_IS_FP ? GetFloat80(GetPSrc2()) : GetFloat32(GetPSrc2());
   dst = GetPSrcDst();

   result = src2 + src1;

   if (SRCDST_IS_FP) PutFloat80(result, dst);
   else PutFloat32(result, dst);
}

static void i_addrl(void)
{
   iFLOAT80 src1, src2, result;
   iWORD *dst;

   src1 = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat64(GetPSrc1());
   src2 = SRC2_IS_FP ? GetFloat80(GetPSrc2()) : GetFloat64(GetPSrc2());
   dst = GetPSrcDst();

   result = src2 + src1;

   if (SRCDST_IS_FP) PutFloat80(result, dst);
   else PutFloat64(result, dst);
}

static void i_subr(void)
{
   iFLOAT80 src1, src2, result;
   iWORD *dst;

   src1 = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat32(GetPSrc1());
   src2 = SRC2_IS_FP ? GetFloat80(GetPSrc2()) : GetFloat32(GetPSrc2());
   dst = GetPSrcDst();

   result = src2 - src1;

   if (SRCDST_IS_FP) PutFloat80(result, dst);
   else PutFloat32(result, dst);
}

static void i_mulr(void)
{
   iFLOAT80 src1, src2, result;
   iWORD *dst;

   src1 = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat32(GetPSrc1());
   src2 = SRC2_IS_FP ? GetFloat80(GetPSrc2()) : GetFloat32(GetPSrc2());
   dst = GetPSrcDst();

   result = src2 * src1;

   if (SRCDST_IS_FP) PutFloat80(result, dst);
   else PutFloat32(result, dst);
}

static void i_mulrl(void)
{
   iFLOAT80 src1, src2, result;
   iWORD *dst;

   src1 = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat64(GetPSrc1());
   src2 = SRC2_IS_FP ? GetFloat80(GetPSrc2()) : GetFloat64(GetPSrc2());
   dst = GetPSrcDst();

   result = src2 * src1;

   if (SRCDST_IS_FP) PutFloat80(result, dst);
   else PutFloat64(result, dst);
}

static void i_divr(void)
{
   iFLOAT80 src1, src2, result;
   iWORD *dst;

   src1 = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat32(GetPSrc1());
   src2 = SRC2_IS_FP ? GetFloat80(GetPSrc2()) : GetFloat32(GetPSrc2());
   dst = GetPSrcDst();

   result = src2 / src1;

   if (SRCDST_IS_FP) PutFloat80(result, dst);
   else PutFloat32(result, dst);
}

static void i_divrl(void)
{
   iFLOAT80 src1, src2, result;
   iWORD *dst;

   src1 = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat64(GetPSrc1());
   src2 = SRC2_IS_FP ? GetFloat80(GetPSrc2()) : GetFloat64(GetPSrc2());
   dst = GetPSrcDst();

   result = src2 / src1;

   if (SRCDST_IS_FP) PutFloat80(result, dst);
   else PutFloat64(result, dst);
}

static void i_cvtir(void)
{
   iWORD src, *dst;
   iFLOAT80 val;

   // FIXME: set FP rounding mode first!
   src = GetSrc1();
   dst = GetPSrcDst();
   val = (iFLOAT80)src;

   if (SRCDST_IS_FP) PutFloat80(val, dst);
   else PutFloat32(val, dst);
}

static void i_cvtri(void)
{
   iFLOAT80 src;
   iWORD *dst;

   // FIXME: set FP rounding mode first!
   src = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat32(GetPSrc1());
   dst = GetPSrcDst();

   *dst = (iWORDs)src;
}

static void i_cvtzri(void)
{
   iFLOAT80 src;
   iWORD *dst;

   // FIXME: set FP rounding mode first!
   src = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat32(GetPSrc1());
   dst = GetPSrcDst();

   *dst = (iWORDs)src;
}

static void i_cvtzril(void)
{
   iFLOAT80 src;
   iWORD *dst;

   // FIXME: set FP rounding mode first!
   src = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat64(GetPSrc1());
   dst = GetPSrcDst();

   *dst = (iWORDs)src;
}

static void i_logbnr(void)
{
   iFLOAT80 src, result;
   iWORD *dst;

   src = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat32(GetPSrc1());
   dst = GetPSrcDst();

   result = log2(src);

   if (SRCDST_IS_FP) PutFloat80(result, dst);
   else PutFloat32(result, dst);
}

static void i_sqrtr(void)
{
   iFLOAT80 src, result;
   iWORD *dst;

   src = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat32(GetPSrc1());
   dst = GetPSrcDst();

   result = sqrt(src);

   if (SRCDST_IS_FP) PutFloat80(result, dst);
   else PutFloat32(result, dst);
}

static void i_atanr(void)
{
   iFLOAT80 src1, src2, result;
   iWORD *dst;

   src1 = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat32(GetPSrc1());
   src2 = SRC2_IS_FP ? GetFloat80(GetPSrc2()) : GetFloat32(GetPSrc2());
   dst = GetPSrcDst();

   result = atan2(src2, src1);

   if (SRCDST_IS_FP) PutFloat80(result, dst);
   else PutFloat32(result, dst);
}

static void i_movr(void)
{
   iFLOAT32 src;
   iWORD *dst;

   src = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat32(GetPSrc1());
   dst = GetPSrcDst();

   if (SRCDST_IS_FP) PutFloat80(src, dst);
   else PutFloat32(src, dst);
}

static void i_movrl(void)
{
   iFLOAT64 src;
   iWORD *dst;

   src = SRC1_IS_FP ? GetFloat80(GetPSrc1()) : GetFloat64(GetPSrc1());
   dst = GetPSrcDst();

   if (SRCDST_IS_FP) PutFloat80(src, dst);
   else PutFloat64(src, dst);
}

static void i_movre(void)
{
   iFLOAT80 src;
   iWORD *dst;

   src = GetFloat80(GetPSrc1());
   dst = GetPSrcDst();

   PutFloat80(src, dst);
}



/* Auto-prototype the assembler functions */
//#define INSTR(name, etc...) void i_##name##_start(void); void i_##name##_end(void);
//#include "i960opc.h"
//#undef INSTR

/* Construct the opcode table */
static opcode_t i960_opcode_table[] = {
#define INSTR(name, type, opcode, etc...) \
   { &i_##name##, /*&i_##name##_start, &i_##name##_end,*/ #name, I_##type, 0x##opcode, etc },
#include "i960opc.h"
#undef INSTR
   { NULL }
};

