/* Intel i960 CPU emulator
 *
 * CPU disassembler (80960KB only).
 *
 * NOTE: Invalid EA modes get handled strangely.
 *
 * Copyright (c) 1999 Richard Mitton
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "i960cpu.h"
#include "i960dasm.h"

char *MakeReg(int x)
{
   /* Use two static buffers so we can be called twice in a row */
   static int n=0;
   static char buf[4], buf2[4];
   char *p;
   if (n) p = &buf[0];
   else   p = &buf2[0];
   n ^= 1;

   switch(x) {
   case REG_PFP: return "pfp";
   case REG_SP:  return "sp";
   case REG_RIP: return "rip";
   case REG_FP:  return "fp";
   default:
      sprintf(p, "%c%i", ((x) & 0x10) ? 'g' : 'r', (x) & 0xf);
      return p;
   }
}

static opcode_t *this_op;

/* Construct the opcode table */
static opcode_t opcode_table[] = {
#define INSTR(name, type, opcode, etc...) \
   { NULL, /*NULL, NULL,*/ #name, I_##type, 0x##opcode, etc },
#include "i960opc.h"
#undef INSTR
   { NULL }
};             

static int GetEA(char *buf, iWORD pc, iWORD opcode, iWORD opcode2)
{
   int abase, offset, mode, scale, index;

   abase = (opcode >> 14) & 0x1f;

   if (((opcode >> 12) & 1) == 0) /* MEMA */
   {
      offset = opcode & 0x1fff;
      mode = opcode & 0x2000;
      if (mode == 0)
      {
         sprintf(buf, "[0x%x]", offset);
      } else {
         sprintf(buf, "[%s+0x%x]", MakeReg(abase), offset);
      }
      return 0;
   } else {                       /* MEMB */
      index = opcode & 0x1f;
      scale = 1 << ((opcode >> 7) & 0x7);
      mode = (opcode >> 10) & 0xf;
      switch(mode)
      {
      case 0x4:
         sprintf(buf, "[%s]", MakeReg(abase));
         return 0;
      case 0x5:
         opcode2 += pc+8;
         sprintf(buf, "[0x%x]", opcode2);
         return 1;
      case 0x7:
         if (scale == 1)
         {
            sprintf(buf, "[%s+%s]", MakeReg(abase),
                                    MakeReg(index));
         } else {
            sprintf(buf, "[%s+%s*%i]", MakeReg(abase),
                                       MakeReg(index),
                                       scale);
         }
         return 0;
      case 0xc:
         sprintf(buf, "[0x%x]", opcode2);
         return 1;
      case 0xd:
         if (opcode2 & 0x80000000)
            sprintf(buf, "[%s-0x%x]", MakeReg(abase), -opcode2);
         else
            sprintf(buf, "[%s+0x%x]", MakeReg(abase), opcode2);
         return 1;
      case 0xe:
         if (scale != 1)
         {
            if (opcode2 & 0x80000000)
               sprintf(buf, "[%s*%i-0x%x]", MakeReg(index), scale, -opcode2);
            else
               sprintf(buf, "[%s*%i+0x%x]", MakeReg(index), scale, opcode2);
         } else {
            if (opcode2 & 0x80000000)
               sprintf(buf, "[%s-0x%x]", MakeReg(index), -opcode2);
            else
               sprintf(buf, "[%s+0x%x]", MakeReg(index), opcode2);
         }
         return 1;
      case 0xf:
         if (scale == 1)
         {
            if (opcode2 & 0x80000000)
               sprintf(buf, "[%s+%s-0x%x]", MakeReg(abase),
                                            MakeReg(index),
                                            -opcode2);
            else
               sprintf(buf, "[%s+%s+0x%x]", MakeReg(abase),
                                            MakeReg(index),
                                            opcode2);
         } else {
            if (opcode2 & 0x80000000)
               sprintf(buf, "[%s+%s*%i-0x%x]", MakeReg(abase),
                                               MakeReg(index),
                                               scale, -opcode2);
            else
               sprintf(buf, "[%s+%s*%i+0x%x]", MakeReg(abase),
                                               MakeReg(index),
                                               scale, opcode2);
         }
         return 1;
      }
   }

   return -1;
}

static void GetSrc1_REG(char *buf, iWORD opcode)
{
   int m1;
   int val;

   m1 = (opcode >> 11) & 1;
   val = opcode & 0x1f;
   if (m1 == 0)   /* register */
   {
      sprintf(buf, "%s", MakeReg(val));
   } else {       /* literal */
      if (this_op->fp)
      {
         switch(val)
         {
         case 0x0: case 0x1: case 0x2: case 0x3:
            sprintf(buf, "fp%i", val); break;
         case 0x10:
            sprintf(buf, "+0.0"); break;
         case 0x16:
            sprintf(buf, "+1.0"); break;
         default:
            sprintf(buf, "<invalid>"); break;
         }
      } else {
         sprintf(buf, "0x%x", val);
      }
   }
}

static void GetSrc1_COBR(char *buf, iWORD opcode)
{
   int m1;
   int val;

   m1 = (opcode >> 13) & 1;
   val = (opcode >> 19) & 0x1f;
   if (m1 == 0)   /* register */
   {
      sprintf(buf, "%s", MakeReg(val));
   } else {       /* literal */
      if (this_op->fp)
      {
         switch(val)
         {
         case 0x0: case 0x1: case 0x2: case 0x3:
            sprintf(buf, "fp%i", val); break;
         case 0x10:
            sprintf(buf, "+0.0"); break;
         case 0x16:
            sprintf(buf, "+1.0"); break;
         default:
            sprintf(buf, "<invalid>"); break;
         }
      } else {
         sprintf(buf, "0x%x", val);
      }
   }
}

static void GetSrc2_REG(char *buf, iWORD opcode)
{
   int m2;
   int val;

   m2 = (opcode >> 12) & 1;
   val = (opcode >> 14) & 0x1f;
   if (m2 == 0)   /* register */
   {
      sprintf(buf, "%s", MakeReg(val));
   } else {       /* literal */
      if (this_op->fp)
      {
         switch(val)
         {
         case 0x0: case 0x1: case 0x2: case 0x3:
            sprintf(buf, "fp%i", val); break;
         case 0x10:
            sprintf(buf, "+0.0"); break;
         case 0x16:
            sprintf(buf, "+1.0"); break;
         default:
            sprintf(buf, "<invalid>"); break;
         }
      } else {
         sprintf(buf, "0x%x", val);
      }
   }
}

static void GetSrc2_COBR(char *buf, iWORD opcode)
{
   int val;

   val = (opcode >> 14) & 0x1f;
   sprintf(buf, "%s", MakeReg(val));
}

static void GetSrcDst_REG(char *buf, iWORD opcode)
{
   int m3, val;

   val = (opcode >> 19) & 0x1f;
   m3 = (opcode >> 13) & 1;
   if (m3 == 0)   /* register */
   {
      sprintf(buf, "%s", MakeReg(val));
   } else {          /* literal */
      if (this_op->fp)
      {
         switch(val)
         {
         case 0x0: case 0x1: case 0x2: case 0x3:
            sprintf(buf, "fp%i", val); break;
         default:
            sprintf(buf, "<invalid>"); break;
         }
      } else {  
         sprintf(buf, "%s", MakeReg(val));
      }
   }
}

static void GetSrcDst_MEM(char *buf, iWORD opcode)
{
   int val;

   val = (opcode >> 19) & 0x1f;
   sprintf(buf, "%s", MakeReg(val));
}

static void GetDisp_COBR(char *buf, iWORD pc, iWORD opcode)
{
   iWORDs disp;
   disp = opcode & 0x1fff;
   if (disp & 0x1000) disp |= ~0x1fff;
   sprintf(buf, "0x%x", disp+pc);

#ifdef __I960_FANCY_ARROWS__
   if (disp < 0)
      strcat(buf, " ; ");
   else if (disp > 0)
      strcat(buf, " ; ");
#else
   if (disp < 0)
      strcat(buf, " ; up");
   else if (disp > 0)
      strcat(buf, " ; down");
#endif
}

static void GetDisp_CTRL(char *buf, iWORD pc, iWORD opcode)
{
   iWORDs disp;
   disp = opcode & 0xffffff;
   if (disp & 0x800000) disp |= ~0xffffff;
   sprintf(buf, "0x%x", disp+pc);

#ifdef __I960_FANCY_ARROWS__
   if (disp < 0)
      strcat(buf, " ; ");
   else if (disp > 0)
      strcat(buf, " ; ");
#else
   if (disp < 0)
      strcat(buf, " ; up");
   else if (disp > 0)
      strcat(buf, " ; down");
#endif
}

int i960_Dasm(char *buf, iWORD pc, int *end)
{
   int n, op_num;
   char src1[30], src2[30], srcdst[30], ea[30];
   opcode_t *op;
   iWORD opcode, opcode2;
   int operands;

   opcode = i960_ReadMem32(pc);

   /* Get the opcode number */
   op_num = (opcode >> 24) & 0x000000ff;
   if (op_num >= 0x50 && op_num < 0x80)
   {
      op_num <<= 4;
      op_num |= (opcode >> 7) & 0xf;
   }

   /* Find the information for it */
   op = &opcode_table[0];
   while(op->opcode != op_num)
   {
      op++;
      if (op->desc == NULL)
      {
         if (isprint(opcode & 0xff))
         {
            sprintf(buf, ".dw 0x%x ; (op %x)\t%c%c%c%c", opcode, op_num,
                    opcode & 0xff,
                    (opcode >> 8) & 0xff,
                    (opcode >> 16) & 0xff,
                    (opcode >> 24) & 0xff);
         } else {
            sprintf(buf, ".dw 0x%x ; (op %x)", opcode, op_num);
         }
         *end = FALSE;
         return 1;
      }
   }

   n = 1;
   this_op = op;
   *end = op->dasm_gap;
   sprintf(buf, "<internal bug!>");

   /* ignore the O_KEEPSRCx flag, it's no use for disassembly */
   operands = op->operands & (~(O_KEEPSRC1|O_KEEPSRC2));

   switch(op->type)
   {
   case I_MEM:
      opcode2 = i960_ReadMem32(pc+4);
      if (operands == O_ADDR)
      {
         n += GetEA(&ea[0], pc, opcode, opcode2);
         sprintf(buf, "%s %s", op->desc, ea);
      } else if (operands == (O_ADDR|O_SRCDST)) {
         n += GetEA(&ea[0], pc, opcode, opcode2);
         GetSrcDst_MEM(&srcdst[0], opcode);
         sprintf(buf, "%s %s, %s", op->desc, ea, srcdst);
      }
      if (!n)
      {
         sprintf(buf, ".dw 0x%x (op %x)", opcode, op_num);
         *end = FALSE;
         return 1;
      }
      break;
   case I_REG:
      if (operands == (O_SRC1|O_SRC2|O_SRCDST)) {
         GetSrc1_REG(&src1[0], opcode);
         GetSrc2_REG(&src2[0], opcode);
         GetSrcDst_REG(&srcdst[0], opcode);
         sprintf(buf, "%s %s, %s, %s", op->desc, src1, src2, srcdst);
      } else if (operands == (O_SRC1|O_SRC2)) {
         GetSrc1_REG(&src1[0], opcode);
         GetSrc2_REG(&src2[0], opcode);
         sprintf(buf, "%s %s, %s", op->desc, src1, src2);
      } else if (operands == (O_SRC1|O_SRCDST)) {
         GetSrc1_REG(&src1[0], opcode);
         GetSrcDst_REG(&srcdst[0], opcode);
         sprintf(buf, "%s %s, %s", op->desc, src1, srcdst);
      } else if (operands == O_NONE) {
         sprintf(buf, "%s", op->desc);
      }
      break;
   case I_COBR:
      if (operands == (O_SRC1|O_SRC2|O_ADDR)) {
         GetSrc1_COBR(&src1[0], opcode);
         GetSrc2_COBR(&src2[0], opcode);
         GetDisp_COBR(&ea[0], pc, opcode);
         sprintf(buf, "%s %s, %s, %s", op->desc, src1, src2, ea);
      } else if (operands == O_SRC1) {
         GetSrc1_COBR(&src1[0], opcode);
         sprintf(buf, "%s %s", op->desc, src1);
      }
      break;
   case I_CTRL:
      if (operands == O_ADDR) {
         GetDisp_CTRL(&ea[0], pc, opcode);
         sprintf(buf, "%s %s", op->desc, ea);
      } else if (operands == O_NONE) {
         sprintf(buf, "%s", op->desc);
      }
      break;
   }
   return n;
}
