#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <fps.h>

#include "virtua.h"
#include "data.h"
#include "unzip.h"

Model2 *model2;
MessageWindow *io;
FONT *bios_font;
iWORD *ROM, *DATA;

/***** Polygon Test - this currently is wrong/useless *****/
Bitmap *foo;
iFLOAT80 preg[4];
bool got[4];
int lx,ly;

PolygonTest::PolygonTest(Model2 *emu)
 : Window("PolygonTest", W_CLOSE, 200, 200)
{
   log->Print("PolygonTest: started.");
   this->emu = emu;
   foo = new Bitmap(200, 200);
   SetTitle("Polygon test");
}

PolygonTest::~PolygonTest()
{
   log->Print("PolygonTest: ended.");
   delete foo;
   emu->polygontest = NULL;
}

void PolygonTest::AddPoint(int reg, iFLOAT80 value)
{
   int x,y,z;
   bool used[4];

   preg[reg] = value;
   got[reg] = true;
   for (x=0;x<4;x++) used[x] = false;

   if (got[1] && got[2] && got[3])
   {
      x = (int)(50 + 40*preg[1]);
      y = (int)(50 + 40*preg[2]);
      z = (int)(128 + 127*preg[3]);
      line(foo->bmp, lx,ly, x,y, GUI::c_text);
      used[1] = used[2] = used[3] = true;
      lx = x;
      ly = y;
   }

   for (x=0;x<4;x++) if (used[x]) got[x] = false;
}

void PolygonTest::Draw(BITMAP *bmp)
{
   blit(foo->bmp, bmp, 0,0,0,0,bmp->w,bmp->h);
   clear_to_color(foo->bmp, GUI::c_background);
   RequestRedraw();
}

/***** End of Polygon Test *****/

void FileException::Report(void)
{
   switch(error)
   {
   case EBADSIZE:
      log->Error("Error with file '%s', File is the wrong size (EBADSIZE).", filename);
      break;
   case EBADCRC:
      log->Error("Error with file '%s', File is corrupt (EBADCRC).", filename);
      break;
   case EBADINFO:
      log->Error("Error with file '%s', File does not have proper info (EBADINFO).", filename);
      break;
   default:
      log->Error("Error with file '%s', %s.", filename, sys_errlist[error]);
      break;
   }
}

Model2::Model2(char *gamename)
 : Window("Model2", W_NONE, 496, 384)
{
   char *name;
   int program_size, data_size;
   iWORD addr, data;

   // display some info
   log->Error("Model2: init '%s'.", gamename);
   name = get_config_string(gamename, "name", "Unknown game");
   SetTitle(name);
   log->Print("Model2: '%s'.", name);
   log->Print("Model2: by %s, %s.",
      get_config_string(gamename, "company", "unknown company"),
      get_config_string(gamename, "year", "year unknown"));
   log->Print("Model2: dumped by %s.",
      get_config_string(gamename, "dumper", "unknown"));

   // clear out all the variables
   model2 = this;
   this->gamename = gamename;
   paused = false;
   stopped = true;
   io = NULL;
   debugger = NULL;
   registers = NULL;
   textureviewer = NULL;
   polygontest = NULL;
   ROM = DATA = NULL;

   // clear the RAM
   memset(&RAM_50,   0x00, sizeof(RAM_50));
   memset(&RAM_90,   0x00, sizeof(RAM_90));
   memset(&RAM_100,  0x00, sizeof(RAM_100));
   memset(&RAM_101,  0x00, sizeof(RAM_101));
   memset(&PAL_DATA, 0x00, sizeof(PAL_DATA));
   memset(&RAM_181,  0x00, sizeof(RAM_181));
   memset(&TEXTURE,  0x00, sizeof(TEXTURE));

   // allocate buffers
   log->Print("Model2: allocating video buffers...");
   layer[0] = new Bitmap(512, 384);
   layer[1] = new Bitmap(512, 384);
   for (int x=0;x<512;x++) pal_changed[x] = true;

   // load the program ROMs
   Rom rom(gamename);
   program_size = get_config_int(gamename, "program_size", 0) * 1024;
   log->Print("Model2: loading PROGRAM ROMs...");
   log->Print("Model2: program size is %iKb total.", program_size / 1024);
   ROM = new iWORD[program_size >> 2];
   rom.Init(ROM, program_size);
   rom.Append("ROM_IC12", "ROM_IC13");
   if (get_config_int(gamename, "split_program", 0)) {
      // program is split over two pairs
      rom.Append("ROM_IC14", "ROM_IC15");
   }

   // load the data ROMs
   data_size = get_config_int(gamename, "data_size", 0) * 0x100000;
   log->Print("Model2: loading DATA ROMs...");
   log->Print("Model2: data size is %iMb total.", data_size / 0x100000);
   DATA = new iWORD[data_size >> 2];
   rom.Init(DATA, data_size);
   rom.Append("ROM_IC10", "ROM_IC11");
   if (rom.SpaceLeft()) rom.Append("ROM_IC8",  "ROM_IC9");
   if (rom.SpaceLeft()) rom.Append("ROM_IC6",  "ROM_IC7");
   if (rom.SpaceLeft()) rom.Append("ROM_IC4",  "ROM_IC5");

   // apply a patch if needed
   addr = get_config_hex(gamename, "patch_addr", 0);
   if (addr != 0) {
      data = get_config_hex(gamename, "patch_data", 0);
      ROM[addr >> 2] = data;
      log->Print("Model2: applying ROM patch 0x%x=0x%x.", addr, data);
   }

   // try loading the backup RAM, if it fails just carry on anyway
   try {
      LoadBackupRAM();
   } catch (Exception &e) {
      log->Error("Model2: error loading backup RAM.");
      memset(&BACKUP[0], 0, 0x4000);
      e.Report();
   }

   // make the I/O log
   io = new MessageWindow("I/O", W_NONE, 496, 134, "io.txt");

   // make the debuggers etc. if they fail, never mind, we can live without

   if (get_config_int("Debugger", "visible", 0))
   {
      try {
         debugger = new Debugger(this);
      } catch (Exception &e) { e.Report(); }
   }

   if (get_config_int("Registers", "visible", 0))
   {
      try {
         registers = new Registers(this);
      } catch (Exception &e) { e.Report(); }
   }

   if (get_config_int("TextureViewer", "visible", 0))
   {
      try {
         textureviewer = new TextureViewer(this);
      } catch (Exception &e) { e.Report(); }
   }

   if (get_config_int("PolygonTest", "visible", 0))
   {
      try {
         polygontest = new PolygonTest(this);
      } catch (Exception &e) { e.Report(); }
   }

   i960_Init(program_size);
}

Model2::~Model2()
{
   log->Print("Model2: finishing.");
   try {
      SaveBackupRAM();
   } catch (Exception &e) {
      log->Error("Model2: error saving backup RAM!");
   }
   DumpRAM();
   i960_Shutdown();
   log->Print("Model2: freeing memory...");
   delete layer[0]; delete layer[1];
   delete[] ROM; delete[] DATA;
   if (debugger) delete debugger;
   if (registers) delete registers;
   if (textureviewer) delete textureviewer;
   if (polygontest) delete polygontest;
   if (io) delete io;
}

void Model2::Reset(void)
{
   log->Print("Model2: hardware reset.");
   intreq = intena = 0x00000000;
   i960_Reset();
   if (debugger)
   {
      debugger->Reset();
      debugger->RequestRedraw();
   }
   if (registers) registers->RequestRedraw();
   stopped = false;
}

void Model2::IntReq(int pin, int num)
{
   io->Print("causing INT%i, bit%i (IP=0x%x)", pin, num, i960_GetIP());
   intreq |= (1 << num);
   if (intena & (1 << num))
      i960_Interrupt(pin);

   if (debugger) debugger->RequestRedraw();
   if (registers) registers->RequestRedraw();
}

bool Model2::KeyPress(int scancode)
{
   switch(scancode) {
   case KEY_R:
      Reset();
      RequestRedraw();
      break;
   case KEY_P:
      paused = !paused;
      if (paused) log->Print("Model2: paused.");
      else log->Print("Model2: unpaused.");
      RequestRedraw();
      break;
   case KEY_D:
      log->Print("Model2: debug hack enabled.");
      RAM_50[0x8000 >> 2] |= 0x4000; // debug cheat
      break;
   case KEY_H:
      log->Print("Model2: debug hack 2 enabled.");
      RAM_50[0x8000 >> 2] |= 0x8; // debug cheat
      break;
   default: return false;
   }
   return true;
}

void Model2::SingleStep(void)
{
   if (stopped) Reset();
#ifdef __I960_DEBUGGER__
   i960_SingleStep();
#endif
   RequestRedraw();
   if (debugger) debugger->RequestRedraw();
   if (registers) registers->RequestRedraw();
}

void Model2::DrawTile(BITMAP *bmp, int x, int y, iWORD *data, int *pal)
{
   int count;
   iWORD v;
   iWORD *pix;
   count = 8;
   do {
      pix = (iWORD *)&(bmp->line[y][x*sizeof(iSHORT)]);
      v = *data++;
      *pix     = (pal[(v >>  8) & 0xf] << 16) | pal[(v >> 12) & 0xf];
      *(pix+1) = (pal[ v        & 0xf] << 16) | pal[(v >>  4) & 0xf];
      *(pix+2) = (pal[(v >> 24) & 0xf] << 16) | pal[(v >> 28) & 0xf];
      *(pix+3) = (pal[(v >> 16) & 0xf] << 16) | pal[(v >> 20) & 0xf];
      y++;
   } while(--count);
}

void Model2::RecalcPalette(int code)
{
   int *pos1, *pos2, count;
   iSHORT val, *pal;
   pal = (iSHORT *)&PAL_DATA[code * 8];
   pos1 = &palette_solid[code * 16];
   pos2 = &palette_trans[code * 16];
   count = 16;
   do {
      val = *pal++;
      *pos1++ = makecol( (val         & 31)*8,
                         ((val >> 5)  & 31)*8,
                         ((val >> 10) & 31)*8);
      if (count == 16) *pos2++ = MASK_COLOR_16;
      else {
         *pos2++ = makecol( (val         & 31)*8,
                            ((val >> 5)  & 31)*8,
                            ((val >> 10) & 31)*8);
      }
   } while(--count);
}

void Model2::Draw(BITMAP *bmp)
{
   int x,y,c,tile,colour;
   iWORD val,offset;
   int frameskip;

   if (!paused)
   {
      if (key[KEY_TAB]) frameskip = 8;
      else frameskip = 1;
      for (x=0;x<frameskip;x++)
      {
      if (paused) break;
      if (stopped) Reset();
      io->Print("IP=0x%x", i960_GetIP());
      i960_Emulate(25000000/60);
      if (debugger) debugger->CheckBreakPoint();
      if (!paused)
      {
         IntReq(0, 0);  // video
         IntReq(3, 10); // sound
      }
      if (debugger) debugger->RequestRedraw();
      if (registers) registers->RequestRedraw();
      }
      //log->Print("0x508000=0x%x", RAM_50[0x8000 >> 2]);
   }

   for (x=0;x<512;x++) if (pal_changed[x]) RecalcPalette(x);

   frame_count++;
   offset = 0;
   for (y=0;y<48;y++)
   {
      for (x=0;x<64;x+=2,offset+=2)
      {
         val = RAM_100[(offset*2 + 0x4000) >> 2];
         tile = val & 0x3fff;
         colour = (val >> 7) & 0x7f;
         if (tilecode[0][offset] != tile || pal_changed[colour])
         {
            tilecode[0][offset] = tile;
            DrawTile(layer[0]->bmp, x*8, y*8, &GFX_DATA[tile * 8], &palette_solid[colour * 16]);
         }

         tile = (val >> 16) & 0x3fff;
         colour = (val >> 23) & 0x7f;
         if (tilecode[0][offset+1] != tile || pal_changed[colour])
         {
            tilecode[0][offset+1] = tile;
            DrawTile(layer[0]->bmp, x*8+8, y*8, &GFX_DATA[tile * 8], &palette_solid[colour * 16]);
         }

         val = RAM_100[(offset*2) >> 2];
         tile = val & 0x3fff;
         colour = (val >> 7) & 0x7f;
         if (tilecode[1][offset] != tile || pal_changed[colour])
         {
            tilecode[1][offset] = tile;
            DrawTile(layer[1]->bmp, x*8, y*8, &GFX_DATA[tile * 8], &palette_trans[colour * 16]);
         }

         tile = (val >> 16) & 0x3fff;
         colour = (val >> 23) & 0x7f;
         if (tilecode[1][offset+1] != tile || pal_changed[colour])
         {
            tilecode[1][offset+1] = tile;
            DrawTile(layer[1]->bmp, x*8+8, y*8, &GFX_DATA[tile * 8], &palette_trans[colour * 16]);
         }
      }
   }
   for (x=0;x<512;x++) pal_changed[x] = false;

   blit(layer[0]->bmp, bmp, 0,0, 0,0, 512, 384);
   if (!key[KEY_F1]) masked_blit(layer[1]->bmp, bmp, 0,0, 0,0, 512, 384);

   text_mode(0);
   textprintf(bmp, font, 0,0, makecol(255, 255, 255), "%i FPS", fps_count);

   if (paused)
   {
      for (y=0;y<bmp->h;y++)
      {
         for (x=0;x<bmp->w;x++)
         {
            int r,g,b;
            c = getpixel(bmp, x, y);
            r = getr(c); g = getg(c); b = getb(c);
            c = makecol(r / 2, g / 2, b / 2);
            putpixel(bmp, x, y, c);
         }
      }
      x = bmp->w / 2;
      y = (bmp->h - text_height(font)) / 2;
      text_mode(-1);
      textout_centre(bmp, font, "Paused", x, y, makecol(255, 255, 255));
   }

   if (!paused) RequestRedraw();
}

void Model2::Stop(void)
{
   stopped = paused = true;
   log->Print("Model2: emulation stopped.");
   RequestRedraw();
   if (debugger) debugger->RequestRedraw();
   if (registers) registers->RequestRedraw();
}

iBYTE InputRead(int reg)
{
   iBYTE val;
   switch(reg)
   {
   case 0x24: val = 0x01; break; // IP=0x1dc
   case 0x10: val = 0xff; if (key[KEY_T]) val -= 0x04;
                          if (key[KEY_S]) val -= 0x08;
                          if (key[KEY_1]) val -= 0x10;
                          if (key[KEY_2]) val -= 0x20;
                          break;
   case 0x12: val = 0xff; if (key[KEY_UP]) val -= 0x20;
                          if (key[KEY_DOWN]) val -= 0x10;
                          if (key[KEY_LEFT]) val -= 0x80;
                          if (key[KEY_RIGHT]) val -= 0x40;
                          if (key[KEY_LCONTROL]) val -= 0x01;
                          if (key[KEY_ALT]) val -= 0x02;
                          if (key[KEY_SPACE]) val -= 0x04;
                          break;
   case 0x14: val = 0xff; if (key[KEY_8_PAD]) val -= 0x20;
                          if (key[KEY_2_PAD]) val -= 0x10;
                          if (key[KEY_4_PAD]) val -= 0x80;
                          if (key[KEY_6_PAD]) val -= 0x40;
                          if (key[KEY_0_PAD]) val -= 0x01;
                          if (key[KEY_DEL_PAD]) val -= 0x02;
                          if (key[KEY_ENTER_PAD]) val -= 0x04;
                          break;
   case 0x16: val = 0xff; if (key[KEY_3]) val -= 0x80;
                          if (key[KEY_4]) val -= 0x40;
                          if (key[KEY_5]) val -= 0x20;
                          if (key[KEY_6]) val -= 0x10;
                          if (key[KEY_7]) val -= 0x08;
                          if (key[KEY_8]) val -= 0x04;
                          if (key[KEY_9]) val -= 0x02;
                          if (key[KEY_0]) val -= 0x01;
                          break;
   case 0x02: val = 0xff; if (key[KEY_T]) val -= 0x04;
                          if (key[KEY_S]) val -= 0x08;
                          if (!model2->input_selector)
                             if (key[KEY_V]) val -= 0x40;
                          break;
   case 0x04: val = 0xff; if (key[KEY_1]) val -= 0x01;
                          if (key[KEY_2]) val -= 0x02;
                          if (key[KEY_PGUP]) val -= 0x10;
                          if (key[KEY_PGDN]) val -= 0x20;
                          break;
   case 0x40: val = 0x00; break; // IP=0x1008
   case 0x42: val = 0x40; break; // IP=0xfdc
   default: val = 0xff; break;
   }
   return val;
}

void InputWrite(int reg, iBYTE value)
{
   switch(reg)
   {
   case 0x00: model2->input_selector = value & 1; break;
   }
}

iWORD i960_ReadMem32(iWORD addr)
{
   if (addr & 3)
   {
      iWORD a,v1,v2;
      a = addr & ~3;
      v1 = i960_ReadMem32(a);
      v2 = i960_ReadMem32(a+4);
      switch(addr & 3)
      {
      case 1: return (v1 >>  8) | (v2 << 24);
      case 2: return (v1 >> 16) | (v2 << 16);
      case 3: return (v1 >> 24) | (v2 <<  8);
      }
   }

   switch(addr >> 16)
   {
   case 0x0000 ... 0x001f: return ROM[addr >> 2];
   case 0x0020 ... 0x0021: return model2->RAM_20[(addr & 0x1ffff) >> 2];
   case 0x0022 ... 0x0023: return ROM[(addr & 0x3ffff) >> 2];
   case 0x0050 ... 0x005f: return model2->RAM_50[(addr & 0xfffff) >> 2];
   case 0x0088 ... 0x0088:
      io->Print("TGP: Read from 0x%x (IP=0x%x)", addr, i960_GetIP());
      return 0x00000008;
   //case 0x0090 ... 0x0091: return model2->RAM_90[(addr & 0x1ffff) >> 2];
#if 1
   case 0x0098 ... 0x0098:
      if (addr == 0x980004) {
         io->Print("TGP - Read status from 0x%x (IP=0x%x)", addr, i960_GetIP());
         return 0xffffffff;
      } else if (addr == 0x98000c) {
         io->Print("TGP - Read status2? from 0x%x (IP=0x%x)", addr, i960_GetIP());
         return rand() & 0x00000004;
      } else {
         io->Print("TGP: Read from 0x%x (IP=0x%x)", addr, i960_GetIP());
         return 0x00000000;
      }
#endif
   case 0x00e8:
      if (addr == 0xe80000)
      {
         io->Print("Read interrupt request (IP=0x%x)", i960_GetIP());
         return model2->intreq;
      } else {
         io->Print("Read interrupt enable (IP=0x%x)", i960_GetIP());
         return model2->intena;
      }
   case 0x00f0:
      io->Print("Read from 0x%x (IP=0x%x)", addr, i960_GetIP());
      if (addr == 0xf00008) return i960_CyclesLeft();
      else return 0xffffffff;
   case 0x01d0 ... 0x01d0: return model2->BACKUP[(addr & 0xffff) >> 2];
   case 0x0200 ... 0x02ff: return DATA[(addr - 0x2000000) >> 2];
   default: io->Print("Read from 0x%x (IP=0x%x)", addr, i960_GetIP());
      return 0xffffffff;
   }
}

// Note: TGP commands are only really for VF2 so far, and even then
// are basically just a guess... (i.e. testing only)
enum {
   TGP_CMD, TGP_DATA_BLOCK, TGP_BURST_POS, TGP_BURST_LEN,
   TGP_CMD_19003232, TGP_CMD_31006262
};

void i960_WriteMem32(iWORD addr, iWORD value)
{
   static int state=TGP_CMD, count;
   switch(addr >> 16)
   {
   case 0x0020 ... 0x0021: model2->RAM_20[(addr & 0x1ffff) >> 2] = value; break;
   case 0x0050 ... 0x005f: model2->RAM_50[(addr & 0xfffff) >> 2] = value; break;
#if 0
   case 0x0080: case 0x0088:
      union {
         iWORD i;
         iFLOAT32 f;
      } fconv;
      fconv.i = value;
      io->Print("Write to 0x%x=0x%x, %f (IP=0x%x)", addr, value, fconv.f, i960_GetIP());
      if (addr >= 0x804000 && addr < 0x804010 && model2->polygontest)
         model2->polygontest->AddPoint((addr - 0x804000) / 4, fconv.f);
      break;
#endif
   case 0x0080 ... 0x009f:
      if (addr == 0x980000) {
         if (value&0x80000000) {
            io->Print("TGP - sending data block (IP=0x%x)", i960_GetIP());
            state = TGP_DATA_BLOCK;
         } else {
            io->Print("TGP - done. (IP=0x%x)", i960_GetIP());
            state = TGP_CMD;
         }
      } else if (addr == 0x800140) {
         io->Print("TGP - starting burst transfer...");
         state = TGP_BURST_POS;
      } else if (addr == 0x804000) {
         switch(state) {
         default: io->Print("TGP: Write to 0x%x=0x%x (IP=0x%x)", addr, value, i960_GetIP()); break;
         case TGP_BURST_POS: io->Print("TGP - position=0x%x", value); state++; break;
         case TGP_BURST_LEN: io->Print("TGP - length=0x%x", value*4); state=TGP_CMD; break;
         }
      } else if (addr >= 0x884000 && addr < 0x88400f) {
         union {
            iWORD i;
            iFLOAT32 f;
         } fconv;
         fconv.i = value;
         switch(state) {
         default: io->Print("TGP: Write to 0x%x=0x%x (IP=0x%x)", addr, value, i960_GetIP()); break;
         case TGP_CMD:
            io->Print("TGP - cmd write to 0x%x=0x%x (IP=0x%x)", addr, value, i960_GetIP());
            count = 0;
            if (value == 0x19003232) state = TGP_CMD_19003232;
            if (value == 0x31006262) state = TGP_CMD_31006262;
            break;
         case TGP_CMD_19003232:
            io->Print("TGP - val%i=0x%.8x (%f)", count, value, fconv.f);
            count++; if (count == 6) state = TGP_CMD;
            break;
         case TGP_CMD_31006262:
            if (count < 3)
               io->Print("TGP - val%i=0x%.8x (%f)", count, value, fconv.f);
            else
               io->Print("TGP - val%i=0x%.4x", count, value & 0xffff);
            count++; if (count == 9) state = TGP_CMD;
            break;
         }
      } else {
         io->Print("TGP: Write to 0x%x=0x%x (IP=0x%x)", addr, value, i960_GetIP());
      }
      break;
   //case 0x0090 ... 0x0091: model2->RAM_90[(addr & 0x1ffff) >> 2] = value; break;
   case 0x0100 ... 0x0100: model2->RAM_100[(addr & 0xffff) >> 2] = value; break;
   case 0x0101 ... 0x0102: model2->RAM_101[(addr & 0x1ffff) >> 2] = value; break;
   case 0x0108 ... 0x010f: model2->GFX_DATA[(addr & 0x7ffff) >> 2] = value; break;
   case 0x0180 ... 0x0180: model2->PAL_DATA[(addr & 0x3fff) >> 2] = value;
                           model2->pal_changed[(addr & 0x3fff) / 32] = true; break;
   case 0x0181 ... 0x0181: model2->RAM_181[(addr & 0xffff) >> 2] = value; break;
   case 0x01d0 ... 0x01d0: model2->BACKUP[(addr & 0xffff) >> 2] = value; break;
   case 0x1200 ... 0x127f: model2->TEXTURE[(addr & 0x7fffff) >> 2] = value; break;
   case 0x1280 ... 0x128f: break;
   //case 0x1200 ... 0x12ff: break;
   case 0x00e8:
      if (addr == 0xe80000)
      {
         model2->intreq &= value;
         io->Print("Acknowledged interrupts 0x%x (IP=0x%x)", ~value, i960_GetIP());
      } else if (addr == 0xe80004)
      {
         model2->intena = value;
         io->Print("Enabled interrupts 0x%x (IP=0x%x)", value, i960_GetIP());
      }
      break;
   case 0xff00:
      if ((addr & 0xfffffff0) == 0xff000010)
         i960_SendIAC(addr, value);
      break;
   default: io->Print("Write to 0x%x=0x%x (IP=0x%x)", addr, value, i960_GetIP());
      break;
   }
}

iBYTE i960_ReadMem8(iWORD addr)
{
   iWORD val;
   iBYTE ret;

   switch(addr >> 16)
   {
   case 0x0000 ... 0x001f: val = ROM[addr >> 2]; break;
   case 0x0020 ... 0x0021: val = model2->RAM_20[(addr & 0x1ffff) >> 2]; break;
   case 0x0022 ... 0x0023: val = ROM[(addr & 0x3ffff) >> 2]; break;
   case 0x0050 ... 0x005f: val = model2->RAM_50[(addr & 0xfffff) >> 2]; break;
   //case 0x0090 ... 0x0091: val = model2->RAM_90[(addr & 0x1ffff) >> 2]; break;
   case 0x01a1: // daytona network
      io->Print("BYTE Read from 0x%x (IP=0x%x)", addr, i960_GetIP());
      return 0x00;
   case 0x01c0:
      io->Print("BYTE Read from 0x%x (IP=0x%x)", addr, i960_GetIP());
      val = InputRead(addr & 0xffff);
      return val;
   case 0x01d0 ... 0x01d0: val = model2->BACKUP[(addr & 0xffff) >> 2]; break;
   case 0x0200 ... 0x02ff: val = DATA[(addr - 0x2000000) >> 2]; break;
   default: io->Print("BYTE Read from 0x%x (IP=0x%x)", addr, i960_GetIP());
      return 0xff;
   }

   switch(addr & 3)
   {
   default:
   case 0: ret = val & 0xff; break;
   case 1: ret = (val >> 8) & 0xff; break;
   case 2: ret = (val >> 16) & 0xff; break;
   case 3: ret = (val >> 24) & 0xff; break;
   }
   return ret;
}

void i960_WriteMem8(iWORD addr, iBYTE value)
{
   iWORD val, mask;
   switch(addr & 3)
   {
   default:
   case 0: val = value;       mask = 0xffffff00; break;
   case 1: val = value <<  8; mask = 0xffff00ff; break;
   case 2: val = value << 16; mask = 0xff00ffff; break;
   case 3: val = value << 24; mask = 0x00ffffff; break;
   }

   switch(addr >> 16)
   {
   case 0x0020 ... 0x0021: model2->RAM_20[(addr & 0x1ffff) >> 2] &= mask;
                           model2->RAM_20[(addr & 0x1ffff) >> 2] |= val; break;
   case 0x0050 ... 0x005f: model2->RAM_50[(addr & 0xfffff) >> 2] &= mask;
                           model2->RAM_50[(addr & 0xfffff) >> 2] |= val; break;
   //case 0x0090 ... 0x0091: model2->RAM_90[(addr & 0x1ffff) >> 2] &= mask;
   //                        model2->RAM_90[(addr & 0x1ffff) >> 2] |= val; break;
   case 0x0108 ... 0x010f: model2->GFX_DATA[(addr & 0x7ffff) >> 2] &= mask;
                           model2->GFX_DATA[(addr & 0x7ffff) >> 2] |= val; break;
   case 0x01c0 ... 0x01c0:
      io->Print("BYTE write to 0x%x=0x%x (IP=0x%x)", addr, value, i960_GetIP());
      InputWrite(addr & 0xffff, value);
      break;
   case 0x01d0 ... 0x01d0: model2->BACKUP[(addr & 0xffff) >> 2] &= mask;
                           model2->BACKUP[(addr & 0xffff) >> 2] |= val; break;
   default: io->Print("BYTE write to 0x%x=0x%x (IP=0x%x)", addr, value,
                      i960_GetIP());
            break;
   }
}

iSHORT i960_ReadMem16(iWORD addr)
{
   iWORD val;
   switch(addr >> 16)
   {
   case 0x0000 ... 0x001f: val = ROM[addr >> 2]; break;
   case 0x0020 ... 0x0021: val = model2->RAM_20[(addr & 0x1ffff) >> 2]; break;
   case 0x0022 ... 0x0023: val = ROM[(addr & 0x3ffff) >> 2]; break;
   case 0x0050 ... 0x005f: val = model2->RAM_50[(addr & 0xfffff) >> 2]; break;
   //case 0x0090 ... 0x0091: val = model2->RAM_90[(addr & 0x1ffff) >> 2]; break;
   case 0x0100 ... 0x0100: val = model2->RAM_100[(addr & 0xffff) >> 2]; break;
   case 0x0101 ... 0x0102: val = model2->RAM_101[(addr & 0x1ffff) >> 2]; break;
   case 0x01c0:
      io->Print("SHORT Read from 0x%x (IP=0x%x)", addr, i960_GetIP());
      val = InputRead(addr & 0xffff);
      return val;
   case 0x01d0 ... 0x01d0: val = model2->BACKUP[(addr & 0xffff) >> 2]; break;
   case 0x01d8 ... 0x01d8:
      io->Print("SHORT Read from 0x%x (IP=0x%x)", addr, i960_GetIP());
      return 0xfff0;
   case 0x0200 ... 0x02ff: val = DATA[(addr - 0x2000000) >> 2]; break;
   default: io->Print("SHORT Read from 0x%x (IP=0x%x)", addr, i960_GetIP());
      return 0xffff;
   }

   switch(addr & 2)
   {
   default:
   case 0: return val & 0xffff;
   case 2: return (val >> 16) & 0xffff;
   }
}

void i960_WriteMem16(iWORD addr, iSHORT value)
{
   iWORD val, mask;
   switch(addr & 2)
   {
   default:
   case 0: val = value;       mask = 0xffff0000; break;
   case 2: val = value << 16; mask = 0x0000ffff; break;
   }

   switch(addr >> 16)
   {
   case 0x0020 ... 0x0021: model2->RAM_20[(addr & 0x1ffff) >> 2] &= mask;
                           model2->RAM_20[(addr & 0x1ffff) >> 2] |= val; break;
   case 0x0050 ... 0x005f: model2->RAM_50[(addr & 0xfffff) >> 2] &= mask;
                           model2->RAM_50[(addr & 0xfffff) >> 2] |= val; break;
   //case 0x0090 ... 0x0091: model2->RAM_90[(addr & 0x1ffff) >> 2] &= mask;
   //                        model2->RAM_90[(addr & 0x1ffff) >> 2] |= val; break;
   case 0x0100 ... 0x0100: model2->RAM_100[(addr & 0xffff) >> 2] &= mask;
                           model2->RAM_100[(addr & 0xffff) >> 2] |= val; break;
   case 0x0101 ... 0x0102: model2->RAM_101[(addr & 0x1ffff) >> 2] &= mask;
                           model2->RAM_101[(addr & 0x1ffff) >> 2] |= val; break;
   case 0x0108 ... 0x010f: model2->GFX_DATA[(addr & 0x7ffff) >> 2] &= mask;
                           model2->GFX_DATA[(addr & 0x7ffff) >> 2] |= val; break;
   case 0x0180 ... 0x0180: model2->PAL_DATA[(addr & 0x3fff) >> 2] &= mask;
                           model2->PAL_DATA[(addr & 0x3fff) >> 2] |= val;
                           model2->pal_changed[(addr & 0x3fff) / 32] = true;
                           break;
   case 0x0181 ... 0x0181: model2->RAM_181[(addr & 0xffff) >> 2] &= mask;
                           model2->RAM_181[(addr & 0xffff) >> 2] |= val; break;
   case 0x01d0 ... 0x01d0: model2->BACKUP[(addr & 0xffff) >> 2] &= mask;
                           model2->BACKUP[(addr & 0xffff) >> 2] |= val; break;

   case 0x1200 ... 0x127f: model2->TEXTURE[(addr & 0x7fffff) >> 2] &= mask;
                           model2->TEXTURE[(addr & 0x7fffff) >> 2] |= val; break;
   case 0x1280 ... 0x128f: break;
   //case 0x1200 ... 0x12ff: break;
   default: io->Print("SHORT write to 0x%x=0x%x (IP=0x%x)", addr, value, i960_GetIP());
            break;
   }
}

void Model2::DumpRAM(void)
{
   FILE *file;

#ifdef __I960_DEBUGGER__
   int n;
   file = fopen("trace.txt", "w");
   fprintf(file, "IP trace:\n");
   for(n=I960_IP_TRACE_SIZE-1;n>=0;n--)
   {
      fprintf(file, "IP=0x%x\n", i960_TraceIP(n));
   }
   fprintf(file, "\n*** end ***\n");
   fclose(file);
#endif

   if (get_config_int("Settings", "dump_tex_ram", 0) == 1) {
      log->Print("Model2: dumping texture RAM to disk.");
      file = fopen("ram1200.bin", "wb");
      fwrite(&TEXTURE[0], 1, 0x800000, file);
      fclose(file);
   }

   if (get_config_int("Settings", "dump_ram", 1) != 1) return;

   log->Print("Model2: dumping system RAM to disk.");
   file = fopen("ram50.bin", "wb");
   fwrite(&RAM_50[0], 1, 0x100000, file);
   fclose(file);
   file = fopen("ram90.bin", "wb");
   fwrite(&RAM_90[0], 1, 0x20000, file);
   fclose(file);
   file = fopen("ram100.bin", "wb");
   fwrite(&RAM_100[0], 1, 0x10000, file);
   fclose(file);
   file = fopen("ram101.bin", "wb");
   fwrite(&RAM_101[0], 1, 0x20000, file);
   fclose(file);
   file = fopen("ram108.bin", "wb");
   fwrite(&GFX_DATA[0], 1, 0x80000, file);
   fclose(file);
   file = fopen("ram180.bin", "wb");
   fwrite(&PAL_DATA[0], 1, 0x4000, file);
   fclose(file);
   file = fopen("ram181.bin", "wb");
   fwrite(&RAM_181[0], 1, 0xc000, file);
   fclose(file);
}

void i960_Message(int fatal, char *msg)
{
   if (fatal)
   {
      log->Error("i960: %s", msg);
      model2->Stop();
   } else {
      io->Print("i960: %s", msg);
   }
}

iBYTE *Rom::ZipLoadFile(char *zipname, char *filename, unsigned int len, iWORD crc32)
{
   unsigned char *buf, *tmp;
   unsigned int actual_len;
   iWORD actual_crc32;
   char path[200];

   strcpy(path, romdir);
   strcat(path, zipname);
   strcat(path, ".zip");

   log->Print("Model2: loading file '%s'@'%s'...", filename, path);

   if (checksum_zipped_file(path, filename, &actual_len, &actual_crc32))
      return NULL;
   else {
      // see if this is a bad dump or not
      if (get_config_int(zipname, "bad_dump", 0) != 1)
      {
         if (crc32 != actual_crc32) throw FileException(filename, EBADCRC);
         if (len != actual_len) throw FileException(filename, EBADSIZE);
      } else {
         // if ROM size does not match up, read as much as we can and
         // just pad the rest.
         if (len != actual_len)
         {
            if (load_zipped_file(path, filename, &buf, &actual_len))
               return NULL;
            tmp = (unsigned char *)malloc(len);
            if (!tmp) { free(buf); throw OutOfMemoryException(); }
            memcpy(tmp, buf, actual_len);
            free(buf);
            return tmp;
         }
      }
   }

   if (load_zipped_file(path, filename, &buf, &actual_len))
      return NULL;
   else
      return buf;
}

void Rom::Append(char *name1, char *name2)
{
   char filename1[20], filename2[20], *t;
   unsigned int size1, size2, len;
   unsigned char *buf1, *buf2;
   iSHORT *d1, *d2;
   iWORD crc1, crc2, v;

   // parse the ROM info 1
   t = get_config_string(gamename, name1, NULL);
   if (!t) t = get_config_string(romof, name1, NULL);
   if (!t) throw FileException(name1, EBADINFO);
   if (sscanf(t, "%*s %s %x %iK %*s", filename1, &crc1, &size1) != 3)
      throw FileException(name1, EBADINFO);
   size1 *= 1024;

   // parse the ROM info 2
   t = get_config_string(gamename, name2, NULL);
   if (!t) t = get_config_string(romof, name2, NULL);
   if (!t) throw FileException(name2, EBADINFO);
   if (sscanf(t, "%*s %s %x %iK %*s", filename2, &crc2, &size2) != 3)
      throw FileException(name2, EBADINFO);
   size2 *= 1024;

   // check both ROM pairs are the same size
   if (size1 != size2) throw FileException(name2, EBADINFO);

   // load file 1
   buf1 = ZipLoadFile(gamename, filename1, size1, crc1);
   if (!buf1 && romof) buf1 = ZipLoadFile(romof, filename1, size1, crc1);
   if (!buf1) throw FileException(filename1, errno);

   // load file 2
   buf2 = ZipLoadFile(gamename, filename2, size2, crc2);
   if (!buf2 && romof) buf2 = ZipLoadFile(romof, filename2, size2, crc2);
   if (!buf2) {
      free(buf1);
      throw FileException(filename2, errno);
   }

   // sort out pointers/length
   len = size1 / 2;
   d1 = (iSHORT *)buf1;
   d2 = (iSHORT *)buf2;

   // interleave ROM pairs
   do {
      v = ((*d2++) << 16) | (*d1++);
      *mem++ = v;
   } while(--len);
   free(buf1);
   free(buf2);
   spaceleft -= (size1*2);
}

void Model2::LoadBackupRAM(void)
{
   FILE *file;
   char *dirname, name[500];

   log->Print("Model2: loading backup RAM.");
   dirname = get_config_string("Settings", "backup_ram_dir", "backup");

   // construct filename
   strcpy(name, dirname);
   put_backslash(name);
   strcat(name, gamename);
   strcat(name, ".bbr");

   file = fopen(name, "rb");
   if (!file) throw FileException(name, errno);

   if (fread(&BACKUP[0], 1, 0x4000, file) != 0x4000)
   {
      fclose(file);
      throw FileException(name, EBADSIZE);
   }

   fclose(file);
}

void Model2::SaveBackupRAM(void)
{
   FILE *file;
   char *dirname, name[500];

   log->Print("Model2: saving backup RAM.");
   dirname = get_config_string("Settings", "backup_ram_dir", "backup");

   // make the directory if not already there
   if (access(dirname, D_OK))
   {
      log->Print("Model2: making directory '%s' for backup RAM.", dirname);
      mkdir(dirname, S_IWUSR);
   }

   // construct filename
   strcpy(name, dirname);
   put_backslash(name);
   strcat(name, gamename);
   strcat(name, ".bbr");

   file = fopen(name, "wb");
   if (!file) throw FileException(name, errno);

   if (fwrite(&BACKUP[0], 1, 0x4000, file) != 0x4000)
   {
      fclose(file);
      throw FileException(name, errno);
   }

   fclose(file);
}

int main(int argc, char *argv[])
{
   allegro_init();
   set_config_file("virtua.ini");
   override_config_file("roms.ini");
   install_fps();

   if (argc != 2)
   {
      printf("VIRTUA Copyright (c) 2000 Richard Mitton (richard.mitton@bigfoot.com)\n\n"
             "Usage: VIRTUA gamename\n");
      return 1;
   }

   if (get_config_string(argv[1], "name", NULL) == NULL)
   {
      printf("Unknown game '%s'.\n", argv[1]);
      return 1;
   }

   try {
      gui = new GUI(800, 600);
   } catch (Exception &e) {
      return 1;
   }

   log->Error("Welcome to Virtua");

   fixup_datafile(data_data);
   set_palette(data_palette);
   gui->SetDesktop(&data_desktop);
   bios_font = font;
   font = &data_font;
   gui_font_baseline = 2;

   log->Print("Loading, please wait...");
   gui->Refresh();

   try {
      log->Print("Allocating memory...");
      model2 = new Model2(argv[1]);
   } catch (Exception &e) {
      log->Error("Cannot start Model 2 emulation!");
      model2 = NULL;
      e.Report();
   }
   gui->Run();

   log->Error("Quitting...");
   gui->Refresh(); // make sure it appears before the big long wait

   if (model2) delete model2;
   delete gui;
   allegro_exit();
   printf("Thank you for using Virtua.\n");
   return 0;
}
