/*
 * gui.cc - GUI code
 */

#include "gui.h"

GUI *gui;
MessageWindow *log;
int GUI::c_active[5], GUI::c_inactive[5], GUI::c_shade[5];
int GUI::c_label, GUI::c_disabled, GUI::c_text, GUI::c_urgent, GUI::c_background;

bool GUI::infoline, GUI::infoline_error;
char GUI::infoline_msg[100];
int GUI::infoline_colour;

void OutOfMemoryException::Report(void)
{
   log->Error("Error: Out of memory!");
}

GUI::GUI(int width, int height, int depth)
{
   gui = this;
   width = get_config_int("GUI", "width", width);
   height = get_config_int("GUI", "height", height);
   depth = get_config_int("GUI", "depth", depth);

   printf("GUI: init at %ix%ix%i.\n", width, height, depth);

   if (install_keyboard() == -1)
   {
      fprintf(stderr, "GUI: cannot install keyboard handler.\n");
      throw AllegroException();
   }

   if (install_timer() == -1)
   {
      fprintf(stderr, "GUI: cannot install timer handler.\n");
      throw AllegroException();
   }

   if (install_mouse() == -1)
   {
      fprintf(stderr, "GUI: cannot install mouse handler.\n");
      throw AllegroException();
   }

   set_color_depth(depth);
   if (set_gfx_mode(GFX_AUTODETECT, width, height, 0, 0))
   {
      fprintf(stderr, "GUI: error setting graphics mode %ix%ix%i:\n%s\n",
              width, height, depth, allegro_error);
      throw AllegroException();
   }

   show_mouse(screen);

   windows = NULL;
   desktop = NULL;
   need_total_refresh = true;

   viewport.x = viewport.y = 0;
   viewport.x2 = SCREEN_W - 1;
   viewport.y2 = (SCREEN_H - INFOLINE_HEIGHT) - 1;

   /* Load the generic colours */
   c_text          = ReadColour("text",               0x000000);
   c_background    = ReadColour("background",         0xffffff);
   c_urgent        = ReadColour("urgent",             0x0000ff);
   c_label         = ReadColour("label",              0x000000);
   c_disabled      = ReadColour("disabled",           0x808080);

   /* Load the active-window colour */
   c_active[0]      = ReadColour("active_shade1",     0xc0c0ff);
   c_active[1]      = ReadColour("active_shade2",     0xa0a0c0);
   c_active[2]      = ReadColour("active_shade3",     0x8080a0);
   c_active[3]      = ReadColour("active_shade4",     0x303040);
   c_active[4]      = ReadColour("active_shade5",     0x202030);

   /* Load the inactive-window colour */
   c_inactive[0]    = ReadColour("inactive_shade1",   0xd0d0d0);
   c_inactive[1]    = ReadColour("inactive_shade2",   0xc0c0c0);
   c_inactive[2]    = ReadColour("inactive_shade3",   0xa0a0a0);
   c_inactive[3]    = ReadColour("inactive_shade4",   0x606060);
   c_inactive[4]    = ReadColour("inactive_shade5",   0x202020);

   /* Load the shade colours */
   c_shade[0]       = ReadColour("shade1",            0xd0d0d0);
   c_shade[1]       = ReadColour("shade2",            0xc0c0c0);
   c_shade[2]       = ReadColour("shade3",            0xa0a0a0);
   c_shade[3]       = ReadColour("shade4",            0x606060);
   c_shade[4]       = ReadColour("shade5",            0x202020);

   int m_w = get_config_int("Message", "width", 400);
   int m_h = get_config_int("Message", "height", 100);
   log = new MessageWindow("Message", W_NONE, m_w, m_h, "debug.txt");
   log->Print("GUI: started OK. Gfx is %ix%ix%i.", width, height, depth);
}

GUI::~GUI()
{
   log->Print("GUI: shutdown.");

   log->Print("GUI: saving colours.");
   /* Save the generic colours */
   SaveColour("text",               c_text);
   SaveColour("background",         c_background);
   SaveColour("urgent",             c_urgent);
   SaveColour("label",              c_label);
   SaveColour("disabled",           c_disabled);

   /* Save the active-window colours */
   SaveColour("active_shade1",      c_active[0]);
   SaveColour("active_shade2",      c_active[1]);
   SaveColour("active_shade3",      c_active[2]);
   SaveColour("active_shade4",      c_active[3]);
   SaveColour("active_shade5",      c_active[4]);

   /* Save the inactive-window colours */
   SaveColour("inactive_shade1",    c_inactive[0]);
   SaveColour("inactive_shade2",    c_inactive[1]);
   SaveColour("inactive_shade3",    c_inactive[2]);
   SaveColour("inactive_shade4",    c_inactive[3]);
   SaveColour("inactive_shade5",    c_inactive[4]);
}

int GUI::ReadColour(char *name, int def)
{
   int c,r,g,b;

   c = get_config_hex("GUI_colours", name, def);

   r = (c >> 16) & 255;
   g = (c >>  8) & 255;
   b =  c        & 255;
   return makecol(r, g, b);
}

void GUI::SaveColour(char *name, int c)
{
   int val;

   val = (getr(c) << 16)
       | (getg(c) << 8)
       |  getb(c);

   set_config_hex("GUI_colours", name, val);
}

void GUI::AddWindow(Window *window)
{
   window->next = windows;
   window->prev = NULL;
   if (windows)
   {
      windows->prev = window;
      windows->RequestFrameRedraw();
   }
   windows = window;
   window->RequestFrameRedraw();
}

void GUI::RemoveWindow(Window *window)
{
   if (window->prev) window->prev->next = window->next;
   if (window->next) window->next->prev = window->prev;
   if (window == windows) windows = window->next;
   need_total_refresh = true;
   if (windows) windows->RequestFrameRedraw();
}                

void GUI::SetDesktop(BITMAP *bmp)
{
   BITMAP *tmp;
   if (desktop)
   {
      log->Print("GUI: removing old desktop.");
      destroy_bitmap(desktop);
      desktop = NULL;
   }
   log->Print("GUI: creating new desktop, %ix%i.", bmp->w, bmp->h);
   desktop = create_bitmap(SCREEN_W, SCREEN_H);
   if (!desktop) throw OutOfMemoryException();
   tmp = create_bitmap(SCREEN_W, SCREEN_H);
   if (!tmp)
   {
      destroy_bitmap(desktop);
      throw OutOfMemoryException();
   }
   blit(bmp, tmp, 0,0, 0,0, bmp->w, bmp->h);
   stretch_blit(tmp, desktop, 0,0, tmp->w, tmp->h, 0,0, desktop->w, desktop->h);
   destroy_bitmap(tmp);
   need_total_refresh = true;
}

void GUI::MoveWindow(Window *window, int dx, int dy)
{
   if (dx != 0 || dy != 0)
   {
      window->x += dx;
      window->y += dy;
      need_total_refresh = true;
   }
}

Window *GUI::FindWindow(int x, int y)
{
   Window *win;
   win = windows;
   while(win)
   {
      if (x >= win->x && y >= win->y && x < win->x+win->w && y < win->y+win->h)
         return win;
      win = win->next;
   }
   return NULL;
}

void GUI::SelectWindow(Window *window)
{
   if (window == windows) return;
   if (window->prev) window->prev->next = window->next;
   if (window->next) window->next->prev = window->prev;
   windows->prev = window;
   window->next = windows;
   window->prev = NULL;
   windows->RequestFrameRedraw();
   windows = window;
   window->RequestFrameRedraw();
}

bool GUI::DoKeyPress(int k)
{
   Window *w, *next;
   w = windows;
   while(w)
   {
      next = w->next;
      if (w->KeyPress(k)) return false;
      w = w->next;
   }
   if (k == KEY_F12) ScreenGrab();
   if (k == KEY_ESC) return true;
   return false;
}

void GUI::Run(void)
{
   int mx,my, ox,oy;
   int k;
   bool mb, ob;
   Window *window;
   bool dragging;
   FrameClickMode mode;
   bool quit=false;

   log->Print("GUI: running.");

   window = NULL;
   mx = mouse_x;
   my = mouse_y;
   mb = mouse_b & 1;
   dragging = false;

   while(!quit)
   {
      // redraw
      Refresh();

      // update infoline
      text_mode(makecol(0,0,0));
      if (infoline)
      {
         int c = infoline_colour * 255 / INFOLINE_SPEED;
         int h = text_height(font);
         textout(screen, font, infoline_msg,
            4, SCREEN_H-((INFOLINE_HEIGHT+h) / 2),
            makecol(c,c,c));
         if (!infoline_error) infoline_colour--;
         if (infoline_colour < 0) infoline = false;
      }

      // process key presses
      while(keypressed())
      {
         k = readkey() >> 8;
         quit = DoKeyPress(k);
      }

      // process mouse clicks
      ox = mx;
      oy = my;
      ob = mb;
      mx = mouse_x; my = mouse_y; mb = mouse_b & 1;
      if (mb)
      {
         if (dragging)
         {
            MoveWindow(window, mx - ox, my - oy);
            ox = mx;
            oy = my;
         } else {
            if (!ob)
            {
               window = FindWindow(mx, my);
               if (window)
               {
                  SelectWindow(window);
                  mode = window->FrameClick(mx - window->x, my - window->y);
                  switch(mode) {
                  case NONE: break;
                  case CLOSE:
                     delete window;
                     break;
                  case DRAG:
                     dragging = true;
                     break;
                  }
               }
            }
         }
      } else {
         dragging = false;
      }
   }
}

void GUI::ScreenGrab(void)
{
   PALETTE pal;
   int x;
   char filename[50];
   BITMAP *bmp = create_sub_bitmap(screen, 0,0, SCREEN_W, SCREEN_H);
   get_palette(pal);

   x = 0;
   do {
      x++;
      sprintf(filename, "grab%.4i.pcx", x);
   } while(!access(filename, F_OK) && x < 9999);

   if (save_bitmap(filename, bmp, pal))
      log->Error("GUI: Error writing screen-grab to file '%s'.", filename);
   else
      log->Print("GUI: Saved screen-grab to file '%s'.", filename);
}

void GUI::Redraw(void)
{
   Window *w, *next;

   w = windows;
   while(w)
   {
      next = w->next;
      if (w->need_frame_redraw)
      {
         w->need_frame_redraw = false;
         w->DrawFrame(w == windows);
         w->need_refresh = true;
      }

      if (w->need_redraw)
      {
         w->need_redraw = false;
         w->Draw(w->client_bmp);
         w->need_refresh = true;
      }
      w = next;
   }
}

void GUI::Refresh(void)
{
   Window *w;

   Redraw();
   vsync_done = false;
   RecursiveRefresh(HIGH, windows, viewport);

   // might as well vsync now if no time-critical stuff is running,
   // just so that the normal windows get a smoother run.
   if (!vsync_done && get_config_int("GUI", "always_vsync", 1) == 1)
   {
      vsync();
      vsync_done = true;
   }
                              
   RecursiveRefresh(NORMAL, windows, viewport);
   RecursiveRefresh(LOW, windows, viewport);
   need_total_refresh = false;
   w = windows;
   while(w)
   {
      w->need_refresh = false;
      w = w->next;
   }
}

void GUI::RecursiveRefresh(Priority level, Window *window, Rect cliprect)
{
   Rect dest, newclip;
   Window *next;

   /* first pick what to draw */
   if (window)
   {
      window->Outside(&dest);
   } else {
      /* draw desktop background */
      dest = viewport;
   }

   /* limit to the clipping area */
   dest.LimitTo(cliprect);

   /* don't draw if no drawing area left */
   if (dest.x <= dest.x2 && dest.y <= dest.y2)
   {
      if (window)
      {
         if (window->vsync == level)
         {
            if (need_total_refresh || window->need_refresh)
            {
               if (!vsync_done && window->vsync != LOW)
               {
                  vsync_done = true;
                  vsync();
               }

               scare_mouse_area(dest.x, dest.y, dest.x2 - dest.x + 1, dest.y2 - dest.y + 1);
               blit(window->bmp, screen,
                    dest.x - window->x,
                    dest.y - window->y,
                    dest.x,
                    dest.y,
                    dest.x2 - dest.x + 1,
                    dest.y2 - dest.y + 1);
#if 0
               rect(screen, dest.x, dest.y, dest.x2, dest.y2, rand());
#endif
               unscare_mouse();
            }
         }
      } else {
         if (need_total_refresh && level == LOW)
         {
            scare_mouse_area(dest.x, dest.y, dest.x2 - dest.x + 1, dest.y2 - dest.y + 1);
            if (desktop)
               blit(desktop, screen,
                 dest.x, dest.y,
                 dest.x, dest.y,
                 dest.x2 - dest.x + 1,
                 dest.y2 - dest.y + 1);
            else
               rectfill(screen, dest.x, dest.y, dest.x2, dest.y2, makecol(0,0,0));
#if 0
            rect(screen, dest.x, dest.y, dest.x2, dest.y2, rand());
#endif
            unscare_mouse();
         }
      }
   }
   
   /* now recurse into all the lower windows */

   /* don't recurse down if this is the last window to do */
   if (!window) return;

   next = window->next;

   /* everything to the left (and top-left / bottom-left) */
   newclip = cliprect;
   newclip.x2 = MIN(window->x - 1, cliprect.x2);
   if (newclip.x2 >= newclip.x)
      RecursiveRefresh(level, next, newclip);

   /* chop the bit we just did off the clipping area
      (stop if no clipping area remains) */
   cliprect.x = MAX(window->x, cliprect.x);
   if (cliprect.x > cliprect.x2) return;

   /* everything to the right (and top-right / bottom-right)*/
   newclip = cliprect;
   newclip.x = MAX(window->x + window->w, cliprect.x);
   if (newclip.x <= newclip.x2)
      RecursiveRefresh(level, next, newclip);

   /* chop the bit we just did off the clipping area
      (stop if no clipping area remains) */
   cliprect.x2 = MIN(window->x + window->w - 1, cliprect.x2);
   if (cliprect.x2 < cliprect.x) return;

   /* everything to the top */
   newclip = cliprect;
   newclip.y2 = MIN(window->y - 1, cliprect.y2);
   if (newclip.y2 >= newclip.y)
      RecursiveRefresh(level, next, newclip);

   /* chop the bit we just did off the clipping area
      (stop if no clipping area remains) */
   cliprect.y = MAX(window->y, cliprect.y);
   if (cliprect.y > cliprect.y2) return;

   /* everything to the bottom */
   newclip = cliprect;
   newclip.y = MAX(window->y + window->h, cliprect.y);
   if (newclip.y <= newclip.y2)
      RecursiveRefresh(level, next, newclip);
}
