/*
 * gui.h - GUI header
 */

#ifndef __GUI_H_INCLUDED__
#define __GUI_H_INCLUDED__

#include <stdio.h>
#include <allegro.h>

#define PRINTF(x, y) __attribute__ ((format (printf, x+1, y+1)))
#define INFOLINE_HEIGHT 16
#define INFOLINE_SPEED 120

class Rect;
class GUI;
class Window;
class BitmapWindow;
class MessageWindow;
class Exception;
class OutOfMemoryException;
class AllegroException;

typedef enum {
   LOW, NORMAL, HIGH
} Priority;

// window flags
#define W_NONE    0
#define W_CLOSE   1

extern MessageWindow *log;
extern GUI *gui;

typedef enum {
   NONE, CLOSE, DRAG
} FrameClickMode;

class Rect
{
public:
   int x, y, x2, y2;

   void LimitTo(Rect &r) {
      if (x < r.x) x = r.x;
      if (y < r.y) y = r.y;
      if (x2 > r.x2) x2 = r.x2;
      if (y2 > r.y2) y2 = r.y2;
   }
};

class GUI
{
public:
   GUI(int width=640, int height=480, int depth=16);
   ~GUI();

   void AddWindow(Window *window);
   void RemoveWindow(Window *window);
   void Run(void);
   void Refresh(void);
   void SetDesktop(BITMAP *bmp);
   Window *FindWindow(int x, int y);
   void MoveWindow(Window *window, int dx, int dy);
   void SelectWindow(Window *window);
   void ScreenGrab(void);

   static int c_active[5], c_inactive[5], c_shade[5];
   static int c_label, c_disabled, c_text, c_urgent, c_background;

protected:
   bool vsync_done, need_total_refresh;
   Window *windows;
   Rect viewport;
   BITMAP *desktop;
   static bool infoline, infoline_error;
   static char infoline_msg[100];
   static int infoline_colour;

   friend class Window;
   friend class MessageWindow;

   void Redraw(void);
   void RecursiveRefresh(Priority level, Window *win, Rect cliprect);
   int ReadColour(char *name, int def);
   void SaveColour(char *name, int c);
   bool DoKeyPress(int k);
};

class Window
{
public:
   virtual ~Window();

   void RequestRedraw(void) { need_redraw = true; }

   Priority vsync;

private:
   int x, y, w, h;
   int flags;
   BITMAP *bmp, *client_bmp;
   Window *next, *prev;
   bool need_redraw, need_frame_redraw, need_refresh;

   friend class GUI;

   void Outside(Rect *r) { r->x = x; r->y = y; r->x2 = x+w-1; r->y2 = y+h-1; }
   void DrawFrame(bool active);
   FrameClickMode FrameClick(int x, int y);

protected:
   char title[256], section[256];

   Window(char *section, int flags, int w, int h);
   void Inside(Rect *r);
   void SetTitle(char *title);
   void RequestFrameRedraw(void) { need_frame_redraw = true; }
   void Select(void) { if (gui) gui->SelectWindow(this); }
   virtual void Draw(BITMAP *bmp) = NULL;
   virtual void Click(int x, int y) { }
   virtual bool KeyPress(int scancode) { return false; }
};

class BitmapWindow : public Window
{
public:
   BitmapWindow(char *section, int flags, int w, int h);
   ~BitmapWindow();

   void SetBitmap(BITMAP *bmp) { image = bmp; }
   void Draw(BITMAP *bmp);
   void SetTitle(char *title) { Window::SetTitle(title); }

protected:
   BITMAP *image;
};

class MessageWindow : public Window
{
public:
   MessageWindow(char *section, int flags, int w, int h, char *filename);
   ~MessageWindow();

   void Print(char *msg, ...) PRINTF(1, 2);
   void Error(char *msg, ...) PRINTF(1, 2);
   void Draw(BITMAP *bmp);

protected:
   const static int LOG_MAX=100, LOG_SIZE=100;
   int head, tail;
   char msgs[LOG_MAX][LOG_SIZE];
   bool urgent[LOG_MAX];
   FILE *file;
};

class Exception
{
public:
   virtual void Report(void) = NULL;
};

class OutOfMemoryException : public Exception
{
public:
   void Report(void);
};

class AllegroException : public Exception
{
public:
   void Report(void) { }
};

#endif /* __GUI_H_INCLUDED__ */
