/*
 * window.cc - window code
 */

#include "gui.h"

const int BORDER_SIZE = 4;
const int TITLE_HEIGHT = 15;

Window::Window(char *section, int flags, int width, int height)
{
   w = width + BORDER_SIZE * 2;
   h = height + BORDER_SIZE * 2 + TITLE_HEIGHT;
   strcpy(this->section, section);
   this->flags = flags;

   bmp = create_bitmap(w, h);
   if (!bmp) throw OutOfMemoryException();
   client_bmp = create_sub_bitmap(bmp,
                                  BORDER_SIZE, BORDER_SIZE+TITLE_HEIGHT,
                                  width, height);
   if (!client_bmp)
   {
      destroy_bitmap(bmp);
      throw OutOfMemoryException();
   }

   x = get_config_int(section, "x_pos", (SCREEN_W - w) / 2);
   y = get_config_int(section, "y_pos", (SCREEN_H - h) / 2);
   vsync = LOW;

   RequestRedraw();
   RequestFrameRedraw();
   gui->AddWindow(this);
}

Window::~Window()
{
   destroy_bitmap(client_bmp);
   destroy_bitmap(bmp);
   set_config_int(section, "x_pos", x);
   set_config_int(section, "y_pos", y);
   gui->RemoveWindow(this);
}

void Window::DrawFrame(bool active)
{
   int *c;
   int bg, hi, sh;

   if (active) c = &GUI::c_active[0];
   else c = &GUI::c_inactive[0];

   hi = c[1];
   bg = c[2];
   sh = c[3];

   // background
   rectfill(bmp, BORDER_SIZE-1, 0, w-(BORDER_SIZE+1), TITLE_HEIGHT+BORDER_SIZE-1, bg);
   rectfill(bmp, 0, 0, BORDER_SIZE-1, h-1, bg);
   rectfill(bmp, w-BORDER_SIZE, 0, w-1, h-1, bg);
   rectfill(bmp, BORDER_SIZE, h-BORDER_SIZE, w-1, h-1, bg);

   // highlight
   hline(bmp, 0, 0, w-1, hi);
   vline(bmp, 0, 0, h-1, hi);
   hline(bmp, 0, TITLE_HEIGHT, w-1, hi);
   hline(bmp, BORDER_SIZE, h-BORDER_SIZE, w-BORDER_SIZE, hi);
   vline(bmp, w-BORDER_SIZE, TITLE_HEIGHT+BORDER_SIZE, h-BORDER_SIZE, hi);

   // shadow
   hline(bmp, 0, TITLE_HEIGHT-1, w-1, sh);
   vline(bmp, w-1, 0, h-1, sh);
   hline(bmp, 0, h-1, w-1, sh);
   hline(bmp, BORDER_SIZE-1, TITLE_HEIGHT+BORDER_SIZE-1, w-BORDER_SIZE, sh);
   vline(bmp, BORDER_SIZE-1, TITLE_HEIGHT+BORDER_SIZE,   h-BORDER_SIZE, sh);

   // corners
   putpixel(bmp, 0, 0, c[0]);
   putpixel(bmp, w-1, h-1, c[4]);
   putpixel(bmp, w-1, TITLE_HEIGHT-1, c[4]);
   putpixel(bmp, 0, TITLE_HEIGHT, c[0]);
   putpixel(bmp, w-1, 0, bg);
   putpixel(bmp, 0, h-1, bg);
   putpixel(bmp, w-BORDER_SIZE, TITLE_HEIGHT+BORDER_SIZE-1, bg);
   putpixel(bmp, BORDER_SIZE-1, TITLE_HEIGHT+BORDER_SIZE-1, c[4]);
   putpixel(bmp, w-BORDER_SIZE, h-BORDER_SIZE, c[0]);
   putpixel(bmp, BORDER_SIZE-1, h-BORDER_SIZE, bg);

   text_mode(-1);
   textout(bmp, font, title, 2, 2, GUI::c_label);

   // close box
   if (flags & W_CLOSE)
      textout_right(bmp, font, "x", w-3, 2, GUI::c_label);
}

void Window::SetTitle(char *title)
{
   strcpy(this->title, title);
   RequestFrameRedraw();
}

FrameClickMode Window::FrameClick(int x, int y)
{
   if (y < TITLE_HEIGHT)
   {
      if (!(flags & W_CLOSE)) return DRAG;
      if (x < w-(text_length(font, "x") + 3)) return DRAG;
      return CLOSE;
   }

   if (x >= BORDER_SIZE && x < w-BORDER_SIZE
    && y >= TITLE_HEIGHT+BORDER_SIZE && y < h-BORDER_SIZE)
   {
      Click(x - BORDER_SIZE, y - (TITLE_HEIGHT+BORDER_SIZE));
   }
   return NONE;
}

void Window::Inside(Rect *r)
{
   r->x = r->y = 0;
   r->x2 = w - (BORDER_SIZE * 2 + 1);
   r->y2 = h - (BORDER_SIZE * 2 + TITLE_HEIGHT + 1);
}
