/******************************************************************************/
/*                                                                            */
/*                        RAINE FILE ACCESS/ZIP SUPPORT                       */
/*                                                                            */
/******************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "raine.h"
#include "config.h"
#include "debug.h"
#include "games.h"
#include "unzip.h"
#include "files.h"

int load_file(char *filename, UBYTE *dest, ULONG size)
{
   FILE *file_ptr;

   file_ptr = fopen(filename,"rb");

   if(file_ptr){
      fread(dest, 1, size, file_ptr);
      fclose(file_ptr);
      return 1;		// Success
   }
   else{
      return 0;		// Failure
   }
}

int save_file(char *filename, UBYTE *source, ULONG size)
{
   FILE *file_ptr;

   file_ptr = fopen(filename,"wb");

   if(file_ptr){
      fwrite(source, 1, size, file_ptr);
      fclose(file_ptr);
      return 1;		// Success
   }
   else{
      return 0;		// Failure
   }
}

int size_file(char *filename)
{
   int len;
   FILE *file_ptr;

   len = 0;

   file_ptr = fopen(filename,"rb");

   if(file_ptr){

      if(!(fseek(file_ptr,0,SEEK_END))){
         len = ftell(file_ptr);
         if(len<0) len=0;
      }

      fclose(file_ptr);
   }

   return len;
}

/*

find a file by filename

*/

static int unz_locate_file_name(unzFile file, char *name)
{
	int err;

    	if(!name)
		return UNZ_PARAMERROR;

	err = unzGoToFirstFile(file);

	while (err == UNZ_OK)
	{
		char current[256+1];

		unzGetCurrentFileInfo(file,NULL,current,256,NULL,0,NULL,0);

		if(!unzStringFileNameCompare(current,name,2))
			return UNZ_OK;

		err = unzGoToNextFile(file);
	}

	return err;
}

/*

find a file by crc32

*/

static int unz_locate_file_crc32(unzFile file, ULONG crc32)
{
	int err;

	if (!crc32)
		return UNZ_PARAMERROR;

	err = unzGoToFirstFile(file);

	while (err == UNZ_OK)
	{
		unz_file_info file_info;

		unzGetCurrentFileInfo(file,&file_info,NULL,0,NULL,0,NULL,0);

		if (file_info.crc == crc32)
			return UNZ_OK;

		err = unzGoToNextFile(file);
	}

	return err;
}

static int load_zipped(char *zipfile, ROM_INFO *rom_info, UBYTE *dest)
{
   unzFile uf;
   int err;

   uf = unzOpen(zipfile);

   if(!uf)			// Fail: Unable to find/open zipfile
      return 0;

   err = unz_locate_file_crc32(uf,rom_info->crc32);
   if(err!=UNZ_OK){
      #ifdef HACKS
      print_debug("unz_locate_file_crc32(): Error #%d\nNow trying with file name...\n",err);
      #endif
      unzClose(uf);

      uf = unzOpen(zipfile);

      if(!uf)			// Fail: Unable to find/open zipfile
         return 0;

      err = unz_locate_file_name(uf,rom_info->name);
      if(err!=UNZ_OK){
         #ifdef HACKS
         print_debug("unz_locate_file_name(): Error #%d\nNow giving up...\n",err);
         #endif
         unzClose(uf);
         return 0;		// Fail: File not in zip
      }
   }

   err = unzOpenCurrentFile(uf);
   if(err!=UNZ_OK){
      #ifdef HACKS
      print_debug("unzOpenCurrentFile(): Error #%d\n",err);
      #endif
      unzCloseCurrentFile(uf);	// Is this needed when open failed?
      unzClose(uf);
      return 0;			// Fail: Something internal
   }

   err = unzReadCurrentFile(uf,dest,rom_info->size);
   if(err<0){
      #ifdef HACKS
      print_debug("unzReadCurrentFile(): Error #%d\n",err);
      #endif
      unzCloseCurrentFile(uf);
      unzClose(uf);
      return 0;			// Fail: Something internal
   }

   err = unzCloseCurrentFile(uf);
   if(err!=UNZ_OK){
      #ifdef HACKS
      print_debug("unzCloseCurrentFile(): Error #%d\n",err);
      #endif
      unzClose(uf);
      return -1;		// Clean up Failed: Something internal
   }

   unzClose(uf);
   return -1;
}

static int size_zipped(char *zipfile, ROM_INFO *rom_info)
{
   unzFile uf;
   unz_file_info file_info;
   int err;

   uf = unzOpen(zipfile);

   if(!uf)			// Fail: Unable to find/open zipfile
      return 0;

   err = unz_locate_file_crc32(uf,rom_info->crc32);
   if(err!=UNZ_OK){
      #ifdef HACKS
      print_debug("unz_locate_file_crc32(): Error #%d\nNow trying with file name...\n",err);
      #endif
      unzClose(uf);

      uf = unzOpen(zipfile);

      if(!uf)			// Fail: Unable to find/open zipfile
         return 0;

      err = unz_locate_file_name(uf,rom_info->name);
      if(err!=UNZ_OK){
         #ifdef HACKS
         print_debug("unz_locate_file_name(): Error #%d\nNow giving up...\n",err);
         #endif
         unzClose(uf);
         return 0;		// Fail: File not in zip
      }
   }

   err = unzGetCurrentFileInfo(uf,&file_info,NULL,0,NULL,0,NULL,0);
   if(err!=UNZ_OK){
      #ifdef HACKS
      print_debug("unzGetCurrentFileInfo(): Error #%d\n",err);
      #endif
      unzClose(uf);
      return 0;			// Fail: Something internal
   }

   unzClose(uf);

   if( file_info.uncompressed_size > 0 )
      return file_info.uncompressed_size;
   else
      return 0;
}

/*

find a game based on main_name (useful for looking up romof games)

*/

static GAME_MAIN *find_game(UBYTE *main_name)
{
   ULONG i;

   for(i=0 ; i<game_count ; i++){

      if(! stricmp(main_name, game_list[i]->main_name) )

         return game_list[i];

   }

   return NULL;
}

/*

recursively search for a rom. recursion allows rom inheritance to
any depth (eg. game a uses roms from game b, game b gets the roms
from game c and game c gets them from game d! yes, I know it's a
bit unlikely, but you never know and recursion makes it look nice).

*/

static ROM_INFO  rec_rom_info;      // the rom to load
static UBYTE    *rec_dest;          // destination memory buffer

ULONG recursive_rom_load(DIR_INFO *head)
{
   UBYTE *dir;

   dir = head[0].maindir;

   if( dir ){

      if( IS_ROMOF(dir) ){

         GAME_MAIN *game_romof;

         game_romof = find_game(dir+1);

         if(game_romof){

            ULONG len;

            len = recursive_rom_load( game_romof->dir_list );

            if(len)

               return len;

         }

      }
      else{

         ULONG ta;
         UBYTE path[512];

         for(ta = 0; ta < 4; ta ++){

            if(dir_cfg.rom_dir[ta][0]){

               sprintf(path, "%s%s.zip", dir_cfg.rom_dir[ta], dir);
               if((load_zipped(path, &rec_rom_info, rec_dest)))
                  return 1;

               sprintf(path, "%s%s/%s", dir_cfg.rom_dir[ta], dir, rec_rom_info.name);
               if((load_file(path, rec_dest, rec_rom_info.size)))
                  return 1;

            }
         }

      }

      return recursive_rom_load( head+1 );

   }
   else{

      return 0;

   }
}

ULONG recursive_rom_size(DIR_INFO *head)
{
   UBYTE *dir;

   dir = head[0].maindir;

   if( dir ){

      if( IS_ROMOF(dir) ){

         GAME_MAIN *game_romof;

         game_romof = find_game(dir+1);

         if(game_romof){

            ULONG len;

            len = recursive_rom_size( game_romof->dir_list );

            if(len)

               return len;

         }

      }
      else{

         ULONG ta,len;
         UBYTE path[512];

         for(ta = 0; ta < 4; ta ++){

            if(dir_cfg.rom_dir[ta][0]){

               sprintf(path, "%s%s.zip", dir_cfg.rom_dir[ta], dir);
               if( ( len=size_zipped(path, &rec_rom_info) ) )
                  return len;

               sprintf(path, "%s%s/%s", dir_cfg.rom_dir[ta], dir, rec_rom_info.name);
               if( ( len=size_file(path) ) )
                  return len;

            }
         }

      }

      return recursive_rom_size( head+1 );

   }
   else{

      return 0;

   }
}

ULONG find_alternative_file_names(ROM_INFO *rom_info, DIR_INFO *dir_list)
{
   GAME_MAIN *game_romof;
   ROM_INFO  *rom_list_romof;
   UBYTE     *dir;
   ULONG      alt_name_count;

   alt_name_count = 0;

   if((!rom_info->crc32) || (!rom_info->size))

      return 0;

   while(dir_list->maindir){

      dir = dir_list->maindir;

      if( IS_ROMOF(dir) ){

         game_romof = find_game(dir+1);

         if(game_romof){

            rom_list_romof = game_romof->rom_list;

            while(rom_list_romof->name){

               if( (rom_list_romof->size  == rom_info->size )
                && (rom_list_romof->crc32 == rom_info->crc32) ){

                  alt_names[alt_name_count] = rom_list_romof->name;
                  alt_name_count = (alt_name_count+1)&7;

               }

               rom_list_romof ++;

            }

         }

      }

      dir_list ++;
   }

   return alt_name_count;
}

/*

dumps the search path for a dir_list. it recursively lists
any romof path list(s)

*/

void dump_search_path(DIR_INFO *dir_list)
{
   UBYTE *dir;

   while(dir_list->maindir){

      dir = dir_list->maindir;

      if( IS_ROMOF(dir) ){

         GAME_MAIN *game_romof;

         game_romof = find_game(dir+1);

         if(game_romof)

            dump_search_path(game_romof->dir_list);

      }
      else{

         if( ! IS_CLONEOF(dir) ){

            ULONG i;

            for(i = 0; i < 4; i ++){

               if(dir_cfg.rom_dir[i][0]){

                  sprintf(load_debug+strlen(load_debug),"%s%s [.zip]\n",dir_cfg.rom_dir[i], dir);

               }
            }
         }
      }

      dir_list++;
   }
}

/*

load rom from a filename

*/

int load_rom(UBYTE *rom, UBYTE *dest, ULONG size)
{
   DIR_INFO *dir_list;
   ROM_INFO *rom_list;
   ULONG     ta,tb,tc;

   dir_list = current_game->dir_list;

   // locate the full rom info (ie. crc32)

   rec_rom_info.name  = rom;
   rec_rom_info.size  = size;
   rec_rom_info.crc32 = 0;

   rom_list = current_game->rom_list;

   while((rom_list->name) && (!rec_rom_info.crc32)){

      if(! stricmp(rom_list->name, rom) )

         rec_rom_info.crc32 = rom_list->crc32;

      rom_list++;
   }

   // now try loading it

   rec_dest = dest;

   ta = recursive_rom_load( dir_list );

   if(!ta){

   // try to locate alternative filenames via the crc32 & size data (merged sets)

   tc = find_alternative_file_names( &rec_rom_info, dir_list );

   // try loading with any alternative file names we found

   ta = 0;

   for(tb=0; tb<tc; tb++){

      rec_rom_info.name = alt_names[tb];

      ta = recursive_rom_load( dir_list );

      if(ta)
         tb = tc;

   }

   }

   // Error Logging

   if(!ta){

      sprintf(load_debug+strlen(load_debug),"Unable to open '%s'\n",rom);
      sprintf(load_debug+strlen(load_debug),"\n");
      sprintf(load_debug+strlen(load_debug),"Search path:\n");
      sprintf(load_debug+strlen(load_debug),"\n");

      dump_search_path(dir_list);

      sprintf(load_debug+strlen(load_debug),"\n");

      load_error |= LOAD_FATAL_ERROR;

   }

   return ta;

}

/*

load rom from an index in the game rom_list[]

*/

int load_rom_index(ULONG num, UBYTE *dest, ULONG size)
{
   ROM_INFO *rom_list;

   rom_list = current_game->rom_list;

   return load_rom(rom_list[num].name, dest, size);
}

// User specified dir_list, no error log

int load_rom_dir(DIR_INFO *dir_list, UBYTE *rom, UBYTE *dest, ULONG size, ULONG crc32)
{
   ULONG ta,tb,tc;

   // locate the full rom info (ie. crc32)

   rec_rom_info.name  = rom;
   rec_rom_info.size  = size;
   rec_rom_info.crc32 = crc32;

   // now try loading it

   rec_dest = dest;

   ta = recursive_rom_load( dir_list );

   if(!ta){

   // try to locate alternative filenames via the crc32 & size data (merged sets)

   tc = find_alternative_file_names( &rec_rom_info, dir_list );

   // try loading with any alternative file names we found

   ta = 0;

   for(tb=0; tb<tc; tb++){

      rec_rom_info.name = alt_names[tb];

      ta = recursive_rom_load( dir_list );

      if(ta)
         tb = tc;

   }

   }

   return ta;
}

int rom_size_dir(DIR_INFO *dir_list, UBYTE *rom, ULONG size, ULONG crc32)
{
   ULONG ta,tb,tc;

   // locate the full rom info (ie. crc32)

   rec_rom_info.name  = rom;
   rec_rom_info.size  = size;
   rec_rom_info.crc32 = crc32;

   // now try loading it

   ta = recursive_rom_size( dir_list );

   if(!ta){

   // try to locate alternative filenames via the crc32 & size data (merged sets)

   tc = find_alternative_file_names( &rec_rom_info, dir_list );

   // try loading with any alternative file names we found

   ta = 0;

   for(tb=0; tb<tc; tb++){

      rec_rom_info.name = alt_names[tb];

      ta = recursive_rom_size( dir_list );

      if(ta)
         tb = tc;

   }

   }

   return ta;
}

#ifdef HACKS

void save_debug(UBYTE *name, UBYTE *src, ULONG size, ULONG mode)
{
   if(debug_mode){
      char str[256];

      if(src){

         if(mode)
            ByteSwap(src,size);

         sprintf(str,"%sdebug/%s", dir_cfg.exe_path, name);
         save_file(str, src, size);

         #ifdef HACKS
         print_debug("Debug Save: '%s' saved.\n", name);
         #endif

      }
      else{

         #ifdef HACKS
         print_debug("Debug Save: '%s' is NULL.\n", name);
         #endif

      }

   }

}

#endif
