#ifndef __VIRTUA_H_INCLUDED__
#define __VIRTUA_H_INCLUDED__

#include <allegro.h>
#include <gui.h>
#include "i960cpu.h"

class Model2;
class Debugger;
class Registers;
class TextureViewer;
class PolygonTest;
class Bitmap;
class Rom;
class FileException;

class Model2 : public Window
{
public:
   Model2(char *gamename);
   ~Model2();

   void Reset(void);
   void SingleStep(void);
   void IntReq(int pin, int num);
   void DumpRAM(void);
   void Stop(void);

protected:
   iWORD RAM_20[0x8000];
   iWORD RAM_50[0x40000];
   iWORD RAM_90[0x8000];
   iWORD RAM_100[0x4000];
   iWORD RAM_101[0x8000];
   iWORD GFX_DATA[0x20000];
   iWORD PAL_DATA[0x1000];
   iWORD RAM_181[0x3000];
   iWORD BACKUP[0x1000];
   iWORD TEXTURE[0x200000];

   iSHORT tilecode[2][64*48];
   bool pal_changed[512];
   Bitmap *layer[2];
   int palette_solid[512*16], palette_trans[512*16];
   iWORD intreq, intena;
   bool input_selector;

   char *gamename;
   bool paused, stopped;
   Debugger *debugger;
   Registers *registers;
   TextureViewer *textureviewer;
   PolygonTest *polygontest;

   friend class Debugger;
   friend class Registers;
   friend class TextureViewer;
   friend class PolygonTest;

   friend iBYTE  i960_ReadMem8(iWORD addr);
   friend iSHORT i960_ReadMem16(iWORD addr);
   friend iWORD  i960_ReadMem32(iWORD addr);
   friend void   i960_WriteMem8(iWORD addr, iBYTE value);
   friend void   i960_WriteMem16(iWORD addr, iSHORT value);
   friend void   i960_WriteMem32(iWORD addr, iWORD value);
   friend iBYTE  InputRead(int addr);
   friend void   InputWrite(int addr, iBYTE value);

   bool KeyPress(int scancode);
   void Draw(BITMAP *bmp);
   void LoadBackupRAM(void);
   void SaveBackupRAM(void);

   void DrawTile(BITMAP *bmp, int x, int y, iWORD *data, int *pal);
   void DrawTileSolid(BITMAP *bmp, int x, int y, iWORD *data, int *pal);
   void RecalcPalette(int code);
};

class TextureViewer : public Window
{
public:
   TextureViewer(Model2 *emu);
   ~TextureViewer();

protected:
   Model2 *emu;
   void Draw(BITMAP *bmp);
};

class Debugger : public Window
{
public:
   Debugger(Model2 *emu);
   ~Debugger();

   void Reset(void);
   void CheckBreakPoint(void);
   void SetBreakPoint(iWORD bp);

protected:
   Model2 *emu;
   iWORD breakpoint;

   bool KeyPress(int scancode);
   void Draw(BITMAP *bmp);
};

class Registers : public Window
{
public:
   Registers(Model2 *emu);
   ~Registers();

protected:
   Model2 *emu;

   void Draw(BITMAP *bmp);
};

class PolygonTest : public Window
{
public:
   PolygonTest(Model2 *emu);
   ~PolygonTest();

   void AddPoint(int reg, iFLOAT80 value);

protected:
   Model2 *emu;

   void Draw(BITMAP *bmp);
};

class Bitmap
{
public:
   Bitmap::Bitmap(int width, int height) {
      bmp = create_bitmap(width, height);
      if (!bmp) throw OutOfMemoryException();
   }

   Bitmap::Bitmap(int width, int height, int depth) {
      bmp = create_bitmap_ex(width, height, depth);
      if (!bmp) throw OutOfMemoryException();
   }

   Bitmap::~Bitmap() {
      destroy_bitmap(bmp);
   }

   void *GetLine(int y) { return bmp->line[y]; }

   BITMAP *bmp;
};

class Rom
{
   char *gamename, *romof, romdir[200];
   iWORD *mem;
   int spaceleft;

   iBYTE *ZipLoadFile(char *zipname, char *filename, unsigned int len, iWORD crc32);

public:
   Rom(char *gamename) {
      this->gamename = gamename;
      romof = get_config_string(gamename, "romof", NULL);
      strcpy(romdir, get_config_string("Settings", "rom_dir", "roms"));
      put_backslash(romdir);
   }

   void Init(iWORD *mem, int size) { this->mem = mem; this->spaceleft = size; }
   void Append(char *file1, char *file2);
   bool SpaceLeft(void) { return spaceleft > 0; }
};

#define EBADSIZE -1
#define EBADCRC  -2
#define EBADINFO -3

class FileException : public Exception
{
public:
   FileException(char *filename, int error) {
      strcpy(this->filename, filename);
      this->error = error;
   }

   void Report(void);
protected:
   char filename[100];
   int error;
};

extern Model2 *model2;
extern MessageWindow *io;
extern FONT *bios_font;

#endif /* __VIRTUA_H_INCLUDED__ */
