/*  TilEm, TI-Linux Emulator
 *  Copyright (C) 2001 Solignac Julien <x1cygnus@xcalc.org>
 *  Portions copyright (C) 2004 Benjamin Moody <benjamin@ecg.mit.edu>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

extern byte lcdactive, lcdmode, lcdinc;

void handlelink(byte line);

int xs_protectport(byte portaddr);
byte xs_calc_md5(byte mode, unsigned long *regs, byte shift, byte oshift);

byte xs_Z80_In(byte Port)
{
	byte v, i;
	
	switch(Port) {
		case 0x00:
			handlelink((~tilnk & pclnk) & 0x03);
			return((~tilnk & pclnk) & 0x03);

		case 0x01:
			updatekb();

			v = 0xFF;
			for (i = 0; i < 7; i++)
				v &= ((keymask >> i) & 0x01)?0xFF:scode[i];

			return(v);

		case 0x02:
		case 0x03:
			return(0xc3 | (flashlock << 2));

		case 0x04:
			if (port4) {
				port4--;
				return(0x01);
			}

			if (onkey) {
				onkey--;
				return(0x01);
			}

			return(0x0A);

		case 0x05:
			return(pager[3-(2*memmode)] & 0x07);

		case 0x06:
			return(pager[1+memmode]);

		case 0x07:
			return(pager[2+memmode]);

		case 0x08:
			return(dbusstat & DBUS_DISABLED);

		case 0x09:
			handlelink((~tilnk & pclnk) & 0x03);
			return(dbusstat & 0x78);

		case 0x0A:
			handlelink((~tilnk & pclnk) & 0x03);
			if (dbusstat & DBUS_DATA) {
				pclnk = 3;
				tilnk = 0;
				dbusstat = DBUS_READY;
				/* printf("read %02X\n",dbusread); */
			}
			return(dbusread);

		case 0x10:
		  	return(0x40 | lcdmode << 6| lcdactive << 5 | (lcdinc & 3));

		case 0x11:
			return(LCDIn());

		case 0x1C:
			return(xs_calc_md5(md5mode, md5regs, md5shift, 0));

		case 0x1D:
			return(xs_calc_md5(md5mode, md5regs, md5shift, 8));

		case 0x1E:
			return(xs_calc_md5(md5mode, md5regs, md5shift, 16));

		case 0x1F:
			return(xs_calc_md5(md5mode, md5regs, md5shift, 24));

		case 0x20:
			return(port20);

		case 0x21:
			return(port21);

		case 0x22:
			return(port22);

		case 0x23:
			return(port23);

		case 0x27:
			return(port27);

		case 0x28:
			return(port28);

		case 0x30:
		case 0x31:
		case 0x32:
		case 0x33:
		case 0x34:
		case 0x35:
		case 0x36:
		case 0x37:
		case 0x38:
			return 0;
	}
	printf("** unexpected ** %d\n", Port);
	return(0x00);
}


void xs_Z80_Out(byte Port, byte Value)
{
	extern byte lcdshow;

	byte x;

	switch(Port) {
		case 0x00:
			tilnk = Value;
			handlelink((~tilnk & pclnk) & 0x03);
			break;

		case 0x01:
			keymask = Value;
			break;

		case 0x04:
		  	if (memmode != (Value & 1)) {
			  if (memmode == 0) {
			    x = pager[3];
			    pager[3] = pager[2]; /* old port 7 value */
			    pager[2] = pager[1]; /* old port 6 value */
			    pager[1] = x; /* old port 5 value */
			  }
			  else {
			    x = pager[1];
			    pager[1] = pager[2]; /* old port 6 value */
			    pager[2] = pager[3]; /* old port 7 value */
			    pager[3] = x; /* old port 5 value */
			  }
			}
			memmode = Value & 1;
			break;
	
		case 0x05:
		  	pager[3-(2*memmode)] = (Value & 0x07) | 0x80;
			break;

		case 0x06:
		  	pager[1+memmode] = Value;
			break;

		case 0x07:
		  	pager[2+memmode] = Value;
			break;

		case 0x08:
			if (Value & 0x80)
				dbusstat |= DBUS_DISABLED;
			else
				dbusstat = DBUS_READY;
			break;

		case 0x0D:
		  /* printf("write %02X\n",Value); */
			if (!(dbusstat & DBUS_READY)) {
			  printf("** writing data too fast to DBUS **\n");
			  dbusstat |= DBUS_ERROR;
			}
			else {
			  dbusstat &= ~DBUS_READY;
			  dbusstat |= DBUS_WRITE;
			  dbuswrite = Value;
			}
			handlelink((~tilnk & pclnk) & 0x03);
			break;

		case 0x10:
			LCDDriver(Value);
			break;

		case 0x11:
			lcdshow = 1;
			LCDOut(Value);
			break;

		case 0x14:
			if (xs_protectport(0x14))
			  flashlock = Value&1;
			break;

		case 0x18:
		case 0x19:
		case 0x1A:
		case 0x1B:
		case 0x1C:
		case 0x1D:
			md5regs[Port-0x18] = (md5regs[Port-0x18] >> 8) |
				(Value << 24);
			break;

		case 0x1E:
			md5shift = Value & 0x1f;
			break;

		case 0x1F:
			md5mode = Value & 3;
			break;

		case 0x20:
			/* Speed control - NYI */
			port20 = Value;
			break;

		case 0x21:
			if (xs_protectport(0x21))
			  port21 = Value;
			break;

		case 0x22:
			if (xs_protectport(0x22))
			  port22 = Value;
			break;

		case 0x23:
		  	if (xs_protectport(0x23))
			  port23 = Value;
			break;

		case 0x27:
			if (Value < 0x13)
				port27 = Value;
			else
				port27 = 0x12;
			break;

		case 0x28:
			port28 = Value;
			break;
	}

	return;
}


/* void xs_emkeyin(unsigned short keyv, byte i) */
/* { */
/* 	if (i == 0xFF) { */

/* 		i = 0; */
/* 		while ((keyv != keycodes[i]) && (i < sizeof keycodes)) */
/* 			i++; */

/* /\* Support for 2nd Enter Key *\/ */
/* 		if (keyv == GDK_KP_Enter) */
/* 			i = 0x08; */
/* 	} */

/* 	if (i == 40) { */
/* 		onkey = -1; */
/* 		return; */
/* 	} */

/* 	if (i != sizeof keycodes) */
/* 		scode[i >> 3] &= ~(1 << (i & 0x07)); */

/* 	return; */
/* } */


/* void xs_emkeyup(unsigned short keyv, byte i) */
/* { */
/* 	if (i == 0xFF) { */
/* 		i = 0; */
/* 		while ((keyv != keycodes[i]) && (i < sizeof keycodes)) */
/* 			i++; */

/* /\* Support for 2nd Enter Key *\/ */
/* 		if (keyv == GDK_KP_Enter) */
/* 			i = 0x08; */
/* 	} */

/* 	if (i == 40) { */
/* 	  onkey = 0; */
/* 	  return; */
/* 	} */

/* 	if (i != sizeof keycodes) */
/* 		scode[i >> 3] |= 1 << (i & 0x07); */

/* 	return; */
/* } */


int xs_protectport(byte portaddr)
{
  /* Note this is not as strong as TI's protection (which monitors the
   * data bus and requires you to actually execute the instructions
   * NOP, NOP, IM 1, DI.)  This only requires the appropriate sequence
   * of instructions to be present in memory.
   */

  unsigned long pa;

  byte unprotect_opstring[] = {0x00,0x00,0xed,0x56,0xf3,0xd3};

  pa = (R.PC.W.l & 0x3FFF) + 0x4000*pager[(R.PC.W.l)>>14];

  if (pa < (0x7c*0x4000) || pa >= (0x80*0x4000)) {
    printf("** writing to port %02x from illegal address %06lx **\n",
	   portaddr,pa);
    return 0;
  }

  if (memcmp(mpages+pa-7,unprotect_opstring,6) || mpages[pa-1]!=portaddr) {
    printf("** writing to port %02x without proper unlock sequence **\n",
	   portaddr);
    return 0;
  }

  return 1;
}

byte xs_calc_md5(byte mode, unsigned long *regs, byte shift, byte oshift)
{
	/* Return the result of a complete MD5 operation:
	   b + ((a + f(b,c,d) + X + T) <<< s) */

	unsigned long x;

	if (mode==0)
		/* F(X,Y,Z) = XY v not(X) Z */
		x = (regs[1] & regs[2]) | ((~regs[1]) & regs[3]);
	else if (mode==1)
		/* G(X,Y,Z) = XZ v Y not(Z) */
		x = (regs[1] & regs[3]) | (regs[2] & (~regs[3]));
	else if (mode==2)
		/* H(X,Y,Z) = X xor Y xor Z */
		x = regs[1] ^ regs[2] ^ regs[3];
	else
		/* I(X,Y,Z) = Y xor (X v not(Z)) */
		x = regs[2] ^ (regs[1] | (~regs[3]));

	x += regs[0] + regs[4] + regs[5];
	x &= 0xffffffff;
	x = (x<<shift)|(x>>(32-shift));
	x += regs[1];

	return ((x >> oshift) & 0xff);
}
