/*  Tilem, TI-Linux Emulator
 *  Copyright (C) 2001 Solignac Julien <x1cygnus@xcalc.org>
 *  Portions copyright (C) 2004 Benjamin Moody <benjamin@ecg.mit.edu>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "../../config.h"

#include<gdk/gdkkeysyms.h>
#include<gtk/gtk.h>
#include<stdio.h>


/* >>> dependencies */
#ifndef _z80
#include "../core/Z80.h"
#endif
/* dependencies <<< */


/* >>> Imported */
#include "menus.h"
#include "outils.h"
#include "../core/dep/inter.h"
/* Imported <<< */

/*
 * The Keypad...
 *
 * is described by one or more 'kmapsto' structures.  A HW model
 * *should* only need one, but the 84+ has a stupid keyboard which
 * means we allow more than one.
 *
 *                margin        rmod              rx
 *                 left      |<horiz >|        |<void>|
 *                  |      | |      | |      | |      | |      |
 *  menu_top      --+------+-+------+-+------+-+------+-+------+-
 *                  |  F1  | |  F2  | |  F3  | |  F4  | |  F5  |
 *  menu_bas      --+------+-+------+-+------+-+------+-+------+-
 *                  |      | |      | |      | |      | |      |
 *                  |      | |      | |      |  +----+---+----+
 *  margin_top    --+------+-+------+-+------+- +--+ |   | +--+
 *                  | 2nd  | |      | |      |  |  | +---+ |  |
 *                --+------+-+------+-+------+- |  |       |  |
 *            -   --+------+-+------+-+------+- |  | +---+ |  |
 *   rmod     |     |Alpha | |      | |      |  +--+ |   | +--+
 *   vert     |   --+------+-+------+-+------+- +----+---+----+
 *            -   --+------+-+------+-+------+-+------+-+------+-
 *                  |      | |      | |      | |      | |      |
 *                --+------+-+------+-+------+-+------+-+------+-
 *            -   --+------+-+------+-+------+-+------+-+------+-
 *   ry void  |     |      | |      | |      | |      | |      |
 *            -   --+------+-+------+-+------+-+------+-+------+-
 *                  |      | |      | |      | |      | |      |
 *                --+------+-+------+-+------+-+------+-+------+-
 *                --+------+-+------+-+------+-+------+-+------+-
 *                  |      | |      | |      | |      | |      |
 *
 *                            td      td
 *              boxleft  left min    max right  boxright
 *                 |       |  |        |  |       |
 * ar_boxtop     --+-------+--+--------+--+-------+-
 *                 |       |  |        |  |       |
 *                 |       |  |   UP   |  |       |
 *                 |       |  |        |  |       |
 * ar_top        --+-------|--+--------+----------+-
 *                 |       |  |        |  |       |
 * ar_lrmin      --+-------+--|-----------+-------+-
 *                 |       |  |        |  |       |
 *                 | LEFT  |  |        |  | RIGHT |
 *                 |       |  |        |  |       |
 * ar_lrmax      --+-------+-----------|--+-------+-
 *                 |       |  |        |  |       |
 * ar_down       --+----------+--------+--|-------+-
 *                 |       |  |        |  |       |
 *                 |       |  |  DOWN  |  |       |
 *                 |       |  |        |  |       |
 * ar_boxbase    --+-------+--+--------+--+-------+-
 *
 */


/* >>> Static */
static gint show_menu(GtkWidget *widget, GdkEvent *event)
{
	extern struct hardware *hw;
	int i;

	byte tx = 0xFF, ty = 0xFF, val = 0xFF;
	GdkEventButton *bevent = (GdkEventButton *) event;

	if ((event->type == GDK_BUTTON_PRESS) && (bevent->button != 1)) {
		gtk_menu_popup(GTK_MENU(widget), NULL, NULL, NULL, NULL, bevent->button, bevent->time);
		return(TRUE);
	}

	if (((event->type == GDK_BUTTON_PRESS) || (event->type == GDK_BUTTON_RELEASE)) && (bevent->button == 1)) {

		for (i=0; i < hw->nkeyb; i++) {
			if ((bevent->x >= hw->keyb[i].xmin) && (bevent->y >= hw->keyb[i].ymin) &&
			    (bevent->x <= hw->keyb[i].xmax) && (bevent->y <= hw->keyb[i].ymax)) {

				if ((bevent->x > hw->keyb[i].margin_left) && ((((int) bevent->x - hw->keyb[i].margin_left) % hw->keyb[i].rmod_horiz) < hw->keyb[i].rx_void)) {
					tx = (bevent->x - hw->keyb[i].margin_left) / hw->keyb[i].rmod_horiz;

					if ((bevent->y > hw->keyb[i].menu_top) && (bevent->y < hw->keyb[i].menu_bas))
						ty = 0;

					if ((bevent->y > hw->keyb[i].margin_top) && ((((int) bevent->y - hw->keyb[i].margin_top) % hw->keyb[i].rmod_vert) < hw->keyb[i].ry_void))
						ty = (bevent->y - hw->keyb[i].margin_top) / hw->keyb[i].rmod_vert + 1; /*(bevent->y - (230 - 28)) / 28;*/
				}

				if (((tx  >= 3) && (ty <= 2) && (ty != 0)) || (tx > 4) || (ty > 9))
					tx = ty = 0xFF;

				if ((bevent->x > hw->keyb[i].ar_boxleft) && (bevent->y > hw->keyb[i].ar_boxtop) && (bevent->y < hw->keyb[i].ar_boxbas) && (bevent->x < hw->keyb[i].ar_boxright)) {
					if ((bevent->y > hw->keyb[i].ar_lrmin) && (bevent->y < hw->keyb[i].ar_lrmax)) {
						if (bevent->x < hw->keyb[i].ar_left) val = 0x01;
						if (bevent->x > hw->keyb[i].ar_right) val = 0x02;
					} else if ((bevent->x > hw->keyb[i].ar_tdmin) && (bevent->x < hw->keyb[i].ar_tdmax)) {
						if (bevent->y > hw->keyb[i].ar_down) val = 0x00;
						if (bevent->y < hw->keyb[i].ar_top) val = 0x03;
					}

				}

				if (tx != 0xFF) {
					if (ty >= 2)
						val = 8 * (5 - tx) + (9 - ty);

					if (ty <= 1)
						val = (6 * 8) + (ty?(5 + tx):(4 - tx));

				}

				if (val != 0xFF) {
					if (event->type == GDK_BUTTON_RELEASE)
						emkeyup(0x00, val);
					else
						emkeyin(0x00, val);
				}
				return(TRUE);
			}
		}

	}


	return(FALSE);
}

static gint show_menu_widget(GtkWidget *widget, GdkEvent *event)
{
	GdkEventButton *bevent = (GdkEventButton *) event;

	if ((event->type == GDK_BUTTON_PRESS) && (bevent->button != 1)) {
		gtk_menu_popup(GTK_MENU(widget), NULL, NULL, NULL, NULL, bevent->button, bevent->time);
		return(TRUE);
	}

	return(FALSE);
}

/* Static <<< */


/* >>> Exported */

void create_menus(GtkWidget *window, GtkItemFactoryEntry *items, int dieseitems, const char *menuname)
{
	GtkAccelGroup *accel_group;
	GtkItemFactory *factory;
	GtkWidget *menu;

	/*gtk_image_factory_parse_rc()*/

	accel_group = gtk_accel_group_new();
	factory = gtk_item_factory_new(GTK_TYPE_MENU, menuname, accel_group);
	/* translatefunc */
	gtk_item_factory_create_items(factory, dieseitems, items, window);
	menu = factory->widget;

	gtk_window_add_accel_group(GTK_WINDOW(window), accel_group);

	gtk_widget_add_events(window, GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
	gtk_signal_connect_object(GTK_OBJECT(window), "event", GTK_SIGNAL_FUNC(show_menu), GTK_OBJECT(menu));
}

void create_menus_widget(GtkWidget *widget, GtkWidget *window, GtkItemFactoryEntry *items, int dieseitems, const char *menuname)
{
	GtkAccelGroup *accel_group;
	GtkItemFactory *factory;
	GtkWidget *menu;

	/*gtk_image_factory_parse_rc()*/

	accel_group = gtk_accel_group_new();
	factory = gtk_item_factory_new(GTK_TYPE_MENU, menuname, accel_group);
	/* translatefunc */
	gtk_item_factory_create_items(factory, dieseitems, items, window);
	menu = factory->widget;

	if (GTK_IS_WINDOW(widget))
	  gtk_window_add_accel_group(GTK_WINDOW(widget), accel_group);

	gtk_widget_add_events(widget, GDK_BUTTON_PRESS_MASK);
	gtk_signal_connect_object(GTK_OBJECT(widget), "event", GTK_SIGNAL_FUNC(show_menu_widget), GTK_OBJECT(menu));
}

/* Exported <<< */
